#!/bin/bash
#
# Load/Unload Redpine RS9113 kernel modules
#
# Based on Redpine sample scripts (run them with 'sh -x' to grab parameters)
#
# Optional variables 
#     IFACE     wifi interface to be created
#     MODE      running mode of interface (STA|AP)

count=0;

. /etc/default/rcS
. /etc/exorint.funcs

log()
{
    echo "$@" | logger -t wifi
}

# Fastboot runs this script towards the end of rcS - block init invocation
[ "$ENABLE_FASTBOOT" = "yes" ] && [ $PPID -eq 1 ] && exit 0

RS9113_WIFI=0 #0 = not wifi on board
RS9113_PLCM=0 #0 = not wifi on PLCM,1 = PLCM with WIFI (needs RS9113_WIFI=1)
WIFI_SUPPORT=0 #0 = board not support RS9113 chipset
FACTORY_STATE=0 #0 = EEPROM normally programmed, 1 = detected factory condition - ignore EEPROM settings and do single detection attempt [BSP-2838]

#Get specific gpio number for this platform
#and WIFI_SUPPORT set. 1.0.x wifi_support=1 when file exists, 1.3.x wifi_support is set in wifi_gpio script
if [ -e /etc/default/wifi_gpio ]; then
    WIFI_SUPPORT=1
    . /etc/default/wifi_gpio
fi

if [ "$(exorint_manufacturercode)" = "65535" ]; then
    log "Detected factory reset condition - EEPROM settings will be ignored"
    FACTORY_STATE=1
fi

#check if PLCMxx is present with WIFI bit set
if [ ${FACTORY_STATE} -eq 0 -a ${RS9113_WIFI} -eq 0 ]; then
    plcdev="$(modem dev)"
    if [ ! -z ${plcdev} -a -e ${plcdev} ]; then
       funcarea="$(dd if=${plcdev}/eeprom skip=37 bs=1 count=1 2>/dev/null | hexdump -e '"%d"')"
       # check plcm eeprom for wifi bit
       if [ $(( ${funcarea} & 128 )) -eq 128 ]; then
           RS9113_WIFI=1
           RS9113_PLCM=1
           log "RS9113 hardware found on PLCM"
       fi
    fi
fi


#RS9113 present WHEN both eeprom bit are 0 OR plcm with wifi flag is present
#(2020/11/25) PLCM has priority
# Check WiFi disable bit in Jumper Flag Area
if [ ${FACTORY_STATE} -eq 0 -a ${RS9113_WIFI} -eq 0 -a ${WIFI_SUPPORT} -eq 1 ]; then
	if [ "$( offsetjumperarea_bit 6 )" -eq 1 ] && [ "$(exorint_ver_carrier)" != "CA16" ]; then
		#log_verb "Disabled via Jumper Flag Area 6"
		RS9113_WIFI=0
	else
		RS9113_WIFI=1
		# Now (2019/04/11) is used Jumper flag area, but we have to maintain for compatibility with old protos.
		# Check WiFi disable bit in SW Flag Area
		if [ "$( exorint_swflagarea_bit 16 )" -eq 1 ]; then
			#log "Disabled via SW Flag Area 16"
			RS9113_WIFI=0
		else
			log "RS9113 hardware found"
		fi
	fi
fi

if [ "${1}" == "wificheck" ]; then
   CHECK_RESULT=0
   if [ ${RS9113_WIFI} -eq 1 ]; then
       if [ ${RS9113_PLCM} -eq 1 ]; then
           CHECK_RESULT=1
       else
           CHECK_RESULT=2
       fi
   fi
   log "RS9113 hw check: ${CHECK_RESULT}"
   echo ${CHECK_RESULT}
   exit ${CHECK_RESULT}
fi

if [ ${FACTORY_STATE} -eq 0 -a ${RS9113_WIFI} -eq 0 ] || [ ${WIFI_SUPPORT} -eq 0 -a ${RS9113_PLCM} -eq 0 ]; then
    log "RS9113 hardware not found"
    exit 0
fi

cd /lib/modules/$(uname -r)/kernel/drivers/EXTERNAL/kmod-wifi-rs9113

if [ ! $? -eq 0 ]; then
  log "RS9113 Error: folder /lib/modules/$(uname -r)/kernel/drivers/EXTERNAL/kmod-wifi-rs9113 not found!"
  exit 1
fi

[ -z "$IFACE" ] && IFACE="wifi0"

# Set TEST=1 for certification testing
case "$TEST" in
    1)
        DRIVER_MODE=2 ;;
    *)
        DRIVER_MODE=1 ;;
esac

# COEX MODE:
#                           1    WLAN STATION /WIFI-Direct/WLAN PER
#                           2    WLAN ACCESS POINT(including muliple APs on different vaps)
#                           3    WLAN ACCESS POINT + STATION MODE(on multiple vaps)
#                           4    BT CLASSIC MODE/BT CLASSIC PER MODE
#                           5    WLAN STATION + BT CLASSIC MODE
#                           6    WLAN ACCESS POINT + BT CLASSIC MODE
#                           8    BT LE MODE /BT LE PER MODE
#                           9    WLAN STATION + BT LE MODE
#                           12   BT CLASSIC + BT LE MODE
#                           13   WLAN STATION + BT CLASSIC MODE + BT LE MODE
#                           14   WLAN ACCESS POINT + BT CLASSIC MODE+ BT LE MODE
#                           16   ZIGBEE MODE/ ZIGBEE PER MODE
#                           17   WLAN STATION + ZIGBEE
#                           32   ZIGBEE COORDINATOR MODE
#                           48   ZIGBEE ROUTER MODE
[ -z "$MODE" ] && MODE=$(sys_params -l network/wifi/interfaces/[name=$IFACE]/mode || true)
# if it's a valid numeric, leave unchanged otherwise remap according to known types
case "$MODE" in
    1|2|3|4|5|6|8|9|12|13|14|16|17|32|48)
        ;;
    STA|"")
        COEX_MODE=1 ;;
    AP)
        COEX_MODE=2 ;;
    *)
        log "invalid coex_mode: $COEX_MODE"
        exit 1 ;;
esac

# Antenna int/ext mode is read from SEEPROM 30:0 - currently allocated for CAREL only
if [ -z $ANTMODE ]; then
    ANTMODE=0 # default to internal antenna

    if [ "$(exorint_ver_carrier)" = "CA16" ]; then
        # Base antenna selection on seeprom byte 30
        "$(exorint_seeprom_byte 30)" = "1" ] && ANTMODE=1
    else
        # Base antenna selection on jumper area bit
        [ "$(offsetjumperarea_bit 5)" = "1" ] && ANTMODE=1
    fi

    # Use external antenna for PLCM
    [ ${RS9113_PLCM} -eq 1 ] && ANTMODE=1

    log "Detected ANTMODE: $ANTMODE (0:internal 1:external)"
fi

#
# based on start_sta.sh, wlan_enable.sh, onebox_insert.sh, common_insert.sh
#
driver_load()
{
    log "Loading Redpine RS9113 modules"

    #Un-Reset module WiFi
    if [ "$WIFI_GPIO" != "" ]
    then
        if [ ! -d /sys/class/gpio/gpio$WIFI_GPIO/ ]
        then
            echo $WIFI_GPIO > /sys/class/gpio/export
        fi;
        echo out > /sys/class/gpio/gpio$WIFI_GPIO/direction
        echo 1 > /sys/class/gpio/gpio$WIFI_GPIO/value
        #Don't unexport
        sleep 1
    fi;
    # prerequisites
    modprobe cfg80211

    insmod onebox_common_gpl.ko
    insmod wlan.ko
    insmod wlan_wep.ko
    insmod wlan_tkip.ko
    insmod wlan_ccmp.ko
    insmod wlan_acl.ko
    insmod wlan_xauth.ko
    insmod wlan_scan_sta.ko
    insmod onebox_wlan_nongpl.ko
    insmod onebox_wlan_gpl.ko
    log "driver mode=$DRIVER_MODE"
    log "coex mode=$COEX_MODE"
    log "ant mode=$ANTMODE"
    insmod onebox_nongpl.ko \
            driver_mode=$DRIVER_MODE \
            firmware_path=/lib/firmware/kmod-wifi-rs9113/ \
            onebox_zone_enabled=0x1 \
            ta_aggr=4 \
            skip_fw_load=0 \
            fw_load_mode=1 \
            sdio_clock=50000 \
            enable_antenna_diversity=0 \
            coex_mode=$COEX_MODE \
            wlan_rf_power_mode=0x00 \
            bt_rf_power_mode=0x22 \
            zigb_rf_power_mode=0x22 \
            country_code=0
    insmod onebox_gpl.ko

    usleep 100000

    # Enable WLAN protocol
    case "$COEX_MODE" in
        1|2|3|5|6|9|13|14|17)
            onebox_util rpine0 enable_protocol 1
            ;;
        *)
            ;;
    esac

    # Enable BT protocol
    case "$COEX_MODE" in
        4|5|6|8|9|12|13|14)
            onebox_util rpine0 enable_protocol 2
            ;;
        *)
            ;;
    esac

    # Select type of antenna
    usleep 600000  # not applied without a delay..
    if [ "$ANTMODE" == "1" ]; then
        onebox_util rpine0 ant_sel 3
        onebox_util rpine0 ant_type 2 3
    else
        onebox_util rpine0 ant_sel 2
    fi

    usleep 500000

    if [ "$TEST" == "1" ]; then
        log "Test mode - not creating vap"
        return 0
    fi

    if [ "$COEX_MODE" == "1" ]
    then
        # Station mode
        onebox_util rpine0 create_vap $IFACE sta sw_bmiss

        #onebox_util <base_interface> set_bgscan_params <bgscan_threshold> \
        #												<rssi_tolerance_threshold>
        #												<periodicity>
        #												<active_scan_duration>
        #												<passive_scan_duration>
        #												<two_probe_enable>
        #												<num_of_bgscan_channels>
        #												<channels_to_scan>
        #
        # bgscan disabled (periodicity=0) by default and enabled only when scan is necessary
        # [#766 WU16: WiFi packet loss due to bg scan]
        onebox_util rpine0 set_bgscan_params	1  \
		    									1  \
		    									0 \
		    									5  \
		    									10 \
		    									0  \
		    									14 \
		    									1 2 3 4 5 6 7 8 9 10 11 12 13 14
    elif [ "$COEX_MODE" == "2" ]
    then
        # Acess Point mode
        onebox_util rpine0 create_vap $IFACE ap
    else
        log "WARNING: Unsupported setup (coex_mode=$COEX_MODE)!"
    fi
}

unload()
{
    log "UN-loading Redpine RS9113 modules"

    # 
    # based on remove_all.sh
    # 
    rmmod onebox_gpl
    rmmod onebox_nongpl

    rmmod onebox_wlan_gpl
    rmmod onebox_wlan_nongpl
    rmmod wlan_scan_sta
    rmmod wlan_xauth
    rmmod wlan_acl
    rmmod wlan_tkip
    rmmod wlan_ccmp
    rmmod wlan_wep
    rmmod wlan

    rmmod onebox_common_gpl

    #wifi reset
    if [ "$WIFI_GPIO" != "" ]
    then
        if [ ! -d /sys/class/gpio/gpio$WIFI_GPIO/ ]
        then
            echo $WIFI_GPIO > /sys/class/gpio/export
        fi;
        echo out > /sys/class/gpio/gpio$WIFI_GPIO/direction
        echo 0 > /sys/class/gpio/gpio$WIFI_GPIO/value
        echo $WIFI_GPIO > /sys/class/gpio/unexport
        usleep 100000
    fi;
}

load()
{
    driver_load

    [ $FACTORY_STATE -eq 1 ] && return 0  # no retries

    while [ $count -lt 7 ]
    do
        count=$((count+1))
        if [ "$(ifconfig -a | grep rpine)" == "" ]
        then
            #First type of error,rpine interface not triggered
            x=$(ifconfig -a | grep rpine)
            echo $x
            log "Detected wifi error: rpine trigger"
            unload
            sleep 1
            driver_load
            sleep 1
        else
            log "Detected wifi interface"
            
            if [ "$(ifconfig -a | grep rpine | grep "00:00:00:00:00:00" )" != "" ]
            then
                #First type of error,rpine interface triggered with null mac address
                log "Detected wifi interface init error"
                log "restart $count!!!"
                unload
                sleep 1
                driver_load
                sleep 1
            else
                wifi_ok=$(ifconfig -a | grep rpine);
                log "Detected wifi: $wifi_ok"
                break
            fi
        fi
    done
}

case "$1" in
start)
    #start plcm10 usb wifi
    if [ ${RS9113_PLCM} -eq 1 ]; then
        log "Trying to start PLCMxx RS9113 WIFI module"
        modem wifion
        if [ ! $? -eq 0 ]; then
            log "PLCMxx RS9113 not started"
            exit 1 
        fi
        log "PLCMxx RS9113 enable WIFI"
        sleep 3 #wait for usb device
        if [ "$(lsusb | grep '1618:9113')" == "" ]; then
            log "PLCMxx RS9113 usb device not found"
            exit 1
        fi
        log "PLCMxx loading RS9113 driver"
        load
        if [ "$(sys_params -l -r network/wifi/interfaces/1/enabled)" = "true" ]; then
            log "PLCMxx wifi0 up"
            sleep 1
            ifup ${IFACE} &2>/dev/null
        fi
    else
        load
    fi
    ;;

stop)
    unload
    ;;

force-reload|restart)
    unload
    load
    ;;

esac
