"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TYPE = exports.Packet = exports.OFFSET = exports.HEADER_LENGTH = void 0;
exports.isPacketComplete = isPacketComplete;
exports.packetLength = packetLength;

var _sprintfJs = require("sprintf-js");

const HEADER_LENGTH = 8;
exports.HEADER_LENGTH = HEADER_LENGTH;
const TYPE = {
  SQL_BATCH: 0x01,
  RPC_REQUEST: 0x03,
  TABULAR_RESULT: 0x04,
  ATTENTION: 0x06,
  BULK_LOAD: 0x07,
  TRANSACTION_MANAGER: 0x0E,
  LOGIN7: 0x10,
  NTLMAUTH_PKT: 0x11,
  PRELOGIN: 0x12,
  FEDAUTH_TOKEN: 0x08
};
exports.TYPE = TYPE;
const typeByValue = {};

for (const name in TYPE) {
  typeByValue[TYPE[name]] = name;
}

const STATUS = {
  NORMAL: 0x00,
  EOM: 0x01,
  IGNORE: 0x02,
  RESETCONNECTION: 0x08,
  RESETCONNECTIONSKIPTRAN: 0x10
};
const OFFSET = {
  Type: 0,
  Status: 1,
  Length: 2,
  SPID: 4,
  PacketID: 6,
  Window: 7
};
exports.OFFSET = OFFSET;
const DEFAULT_SPID = 0;
const DEFAULT_PACKETID = 1;
const DEFAULT_WINDOW = 0;
const NL = '\n';

class Packet {
  constructor(typeOrBuffer) {
    this.buffer = void 0;

    if (typeOrBuffer instanceof Buffer) {
      this.buffer = typeOrBuffer;
    } else {
      const type = typeOrBuffer;
      this.buffer = Buffer.alloc(HEADER_LENGTH, 0);
      this.buffer.writeUInt8(type, OFFSET.Type);
      this.buffer.writeUInt8(STATUS.NORMAL, OFFSET.Status);
      this.buffer.writeUInt16BE(DEFAULT_SPID, OFFSET.SPID);
      this.buffer.writeUInt8(DEFAULT_PACKETID, OFFSET.PacketID);
      this.buffer.writeUInt8(DEFAULT_WINDOW, OFFSET.Window);
      this.setLength();
    }
  }

  setLength() {
    this.buffer.writeUInt16BE(this.buffer.length, OFFSET.Length);
  }

  length() {
    return this.buffer.readUInt16BE(OFFSET.Length);
  }

  resetConnection(reset) {
    let status = this.buffer.readUInt8(OFFSET.Status);

    if (reset) {
      status |= STATUS.RESETCONNECTION;
    } else {
      status &= 0xFF - STATUS.RESETCONNECTION;
    }

    this.buffer.writeUInt8(status, OFFSET.Status);
  }

  last(last) {
    let status = this.buffer.readUInt8(OFFSET.Status);

    if (arguments.length > 0) {
      if (last) {
        status |= STATUS.EOM;
      } else {
        status &= 0xFF - STATUS.EOM;
      }

      this.buffer.writeUInt8(status, OFFSET.Status);
    }

    return this.isLast();
  }

  ignore(last) {
    let status = this.buffer.readUInt8(OFFSET.Status);

    if (last) {
      status |= STATUS.IGNORE;
    } else {
      status &= 0xFF - STATUS.IGNORE;
    }

    this.buffer.writeUInt8(status, OFFSET.Status);
  }

  isLast() {
    return !!(this.buffer.readUInt8(OFFSET.Status) & STATUS.EOM);
  }

  packetId(packetId) {
    if (packetId) {
      this.buffer.writeUInt8(packetId % 256, OFFSET.PacketID);
    }

    return this.buffer.readUInt8(OFFSET.PacketID);
  }

  addData(data) {
    this.buffer = Buffer.concat([this.buffer, data]);
    this.setLength();
    return this;
  }

  data() {
    return this.buffer.slice(HEADER_LENGTH);
  }

  type() {
    return this.buffer.readUInt8(OFFSET.Type);
  }

  statusAsString() {
    const status = this.buffer.readUInt8(OFFSET.Status);
    const statuses = [];

    for (const name in STATUS) {
      const value = STATUS[name];

      if (status & value) {
        statuses.push(name);
      } else {
        statuses.push(undefined);
      }
    }

    return statuses.join(' ').trim();
  }

  headerToString(indent = '') {
    const text = (0, _sprintfJs.sprintf)('type:0x%02X(%s), status:0x%02X(%s), length:0x%04X, spid:0x%04X, packetId:0x%02X, window:0x%02X', this.buffer.readUInt8(OFFSET.Type), typeByValue[this.buffer.readUInt8(OFFSET.Type)], this.buffer.readUInt8(OFFSET.Status), this.statusAsString(), this.buffer.readUInt16BE(OFFSET.Length), this.buffer.readUInt16BE(OFFSET.SPID), this.buffer.readUInt8(OFFSET.PacketID), this.buffer.readUInt8(OFFSET.Window));
    return indent + text;
  }

  dataToString(indent = '') {
    const BYTES_PER_GROUP = 0x04;
    const CHARS_PER_GROUP = 0x08;
    const BYTES_PER_LINE = 0x20;
    const data = this.data();
    let dataDump = '';
    let chars = '';

    for (let offset = 0; offset < data.length; offset++) {
      if (offset % BYTES_PER_LINE === 0) {
        dataDump += indent;
        dataDump += (0, _sprintfJs.sprintf)('%04X  ', offset);
      }

      if (data[offset] < 0x20 || data[offset] > 0x7E) {
        chars += '.';

        if ((offset + 1) % CHARS_PER_GROUP === 0 && !((offset + 1) % BYTES_PER_LINE === 0)) {
          chars += ' ';
        }
      } else {
        chars += String.fromCharCode(data[offset]);
      }

      if (data[offset] != null) {
        dataDump += (0, _sprintfJs.sprintf)('%02X', data[offset]);
      }

      if ((offset + 1) % BYTES_PER_GROUP === 0 && !((offset + 1) % BYTES_PER_LINE === 0)) {
        dataDump += ' ';
      }

      if ((offset + 1) % BYTES_PER_LINE === 0) {
        dataDump += '  ' + chars;
        chars = '';

        if (offset < data.length - 1) {
          dataDump += NL;
        }
      }
    }

    if (chars.length) {
      dataDump += '  ' + chars;
    }

    return dataDump;
  }

  toString(indent = '') {
    return this.headerToString(indent) + '\n' + this.dataToString(indent + indent);
  }

  payloadString() {
    return '';
  }

}

exports.Packet = Packet;

function isPacketComplete(potentialPacketBuffer) {
  if (potentialPacketBuffer.length < HEADER_LENGTH) {
    return false;
  } else {
    return potentialPacketBuffer.length >= potentialPacketBuffer.readUInt16BE(OFFSET.Length);
  }
}

function packetLength(potentialPacketBuffer) {
  return potentialPacketBuffer.readUInt16BE(OFFSET.Length);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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