import { Headers, InfluxDB, AnnotatedCSVResponse } from '@influxdata/influxdb-client';

interface Error {
    /** code is the machine-readable error code. */
    readonly code: 'internal error' | 'not found' | 'conflict' | 'invalid' | 'unprocessable entity' | 'empty value' | 'unavailable' | 'forbidden' | 'too many requests' | 'unauthorized' | 'method not allowed' | 'request too large' | 'unsupported media type';
    /** Human-readable message. */
    readonly message?: string;
    /** Describes the logical code operation when the error occurred. Useful for debugging. */
    readonly op?: string;
    /** Stack of errors that occurred during processing of the request. Useful for debugging. */
    readonly err?: string;
}
interface Routes {
    authorizations?: string;
    buckets?: string;
    dashboards?: string;
    external?: {
        statusFeed?: string;
    };
    variables?: string;
    me?: string;
    flags?: string;
    orgs?: string;
    query?: {
        self?: string;
        ast?: string;
        analyze?: string;
        suggestions?: string;
    };
    setup?: string;
    signin?: string;
    signout?: string;
    sources?: string;
    system?: {
        metrics?: string;
        debug?: string;
        health?: string;
    };
    tasks?: string;
    telegrafs?: string;
    users?: string;
    write?: string;
}
interface DBRPs {
    content?: DBRP[];
}
interface DBRP {
    /** The ID of the DBRP mapping. */
    readonly id: string;
    /** The ID of the organization. */
    orgID: string;
    /** The ID of the bucket used as the target for the translation. */
    bucketID: string;
    /** InfluxDB v1 database */
    database: string;
    /** InfluxDB v1 retention policy */
    retention_policy: string;
    /** Mapping represents the default retention policy for the database specified. */
    default: boolean;
    /** Indicates an autogenerated, virtual mapping based on the bucket name. Currently only available in OSS. */
    virtual?: boolean;
    links?: Links;
}
/**
 * URI pointers for additional paged results.

*/
interface Links {
    next?: Link;
    self: Link;
    prev?: Link;
}
/**
 * URI of resource.
 */
type Link = string;
interface DBRPCreate {
    /** The ID of the organization. */
    orgID?: string;
    /** The name of the organization that owns this mapping. */
    org?: string;
    /** The ID of the bucket used as the target for the translation. */
    bucketID: string;
    /** InfluxDB v1 database */
    database: string;
    /** InfluxDB v1 retention policy */
    retention_policy: string;
    /** Mapping represents the default retention policy for the database specified. */
    default?: boolean;
}
interface DBRPGet {
    content?: DBRP;
}
interface DBRPUpdate {
    /** InfluxDB v1 retention policy */
    retention_policy?: string;
    default?: boolean;
}
interface TelegrafPlugins {
    version?: string;
    os?: string;
    plugins?: TelegrafPlugin[];
}
interface TelegrafPlugin {
    type?: string;
    name?: string;
    description?: string;
    config?: string;
}
interface Telegrafs {
    configurations?: Telegraf[];
}
type Telegraf = TelegrafRequest & {
    readonly id?: string;
    readonly links?: {
        self?: Link;
        labels?: Link;
        members?: Link;
        owners?: Link;
    };
    readonly labels?: Labels;
};
interface TelegrafRequest {
    name?: string;
    description?: string;
    metadata?: {
        buckets?: string[];
    };
    config?: string;
    orgID?: string;
}
type Labels = Label[];
interface Label {
    readonly id?: string;
    readonly orgID?: string;
    name?: string;
    /** Key-value pairs associated with this label.
  To remove a property, send an update with an empty value (`""`) for the key.
   */
    properties?: any;
}
interface TelegrafPluginRequest {
    name?: string;
    description?: string;
    plugins?: Array<{
        type?: string;
        name?: string;
        alias?: string;
        description?: string;
        config?: string;
    }>;
    metadata?: {
        buckets?: string[];
    };
    config?: string;
    orgID?: string;
}
interface LabelsResponse {
    labels?: Labels;
    links?: Links;
}
interface LabelMapping {
    /** Label ID.
  The ID of the label to attach.
   */
    labelID?: string;
}
interface LabelResponse {
    label?: Label;
    links?: Links;
}
interface ResourceMembers {
    links?: {
        self?: string;
    };
    users?: ResourceMember[];
}
type ResourceMember = UserResponse & {
    role?: 'member';
};
interface UserResponse {
    /** The ID of the user.
     */
    readonly id?: string;
    /** The name of the user.
     */
    name: string;
    /** The status of a user. An inactive user won't have access to resources.
     */
    status?: 'active' | 'inactive';
    readonly links?: {
        self?: string;
    };
}
interface AddResourceMemberRequestBody {
    /** The ID of the user to add to the resource.
     */
    id: string;
    /** The name of the user to add to the resource.
     */
    name?: string;
}
interface ResourceOwners {
    links?: {
        self?: string;
    };
    users?: ResourceOwner[];
}
type ResourceOwner = UserResponse & {
    role?: 'owner';
};
interface LineProtocolError {
    /** Code is the machine-readable error code. */
    readonly code: 'internal error' | 'not found' | 'conflict' | 'invalid' | 'empty value' | 'unavailable';
    /** Human-readable message. */
    readonly message?: string;
    /** Describes the logical code operation when the error occurred. Useful for debugging. */
    readonly op?: string;
    /** Stack of errors that occurred during processing of the request. Useful for debugging. */
    readonly err?: string;
    /** First line in the request body that contains malformed data. */
    readonly line?: number;
}
interface LineProtocolLengthError {
    /** Code is the machine-readable error code. */
    readonly code: 'invalid';
    /** Human-readable message. */
    readonly message: string;
}
/**
 * The delete predicate request.
 */
interface DeletePredicateRequest {
    /** A timestamp ([RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)).
  The earliest time to delete from.
   */
    start: string;
    /** A timestamp ([RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)).
  The latest time to delete from.
   */
    stop: string;
    /** An expression in [delete predicate syntax](https://docs.influxdata.com/influxdb/v2.3/reference/syntax/delete-predicate/).
     */
    predicate?: string;
}
interface LabelCreateRequest {
    orgID: string;
    name: string;
    /** Key-value pairs associated with this label.
  
  To remove a property, send an update with an empty value (`""`) for the key.
   */
    properties?: any;
}
interface LabelUpdate {
    name?: string;
    properties?: any;
}
type Dashboard = CreateDashboardRequest & {
    links?: {
        self?: Link;
        cells?: Link;
        members?: Link;
        owners?: Link;
        labels?: Link;
        org?: Link;
    };
    readonly id?: string;
    meta?: {
        createdAt?: string;
        updatedAt?: string;
    };
    cells?: Cells;
    labels?: Labels;
};
interface CreateDashboardRequest {
    /** The ID of the organization that owns the dashboard. */
    orgID: string;
    /** The user-facing name of the dashboard. */
    name: string;
    /** The user-facing description of the dashboard. */
    description?: string;
}
type Cells = Cell[];
interface Cell {
    id?: string;
    links?: {
        self?: string;
        view?: string;
    };
    x?: number;
    y?: number;
    w?: number;
    h?: number;
    /** The reference to a view from the views API. */
    viewID?: string;
}
type DashboardWithViewProperties = CreateDashboardRequest & {
    links?: {
        self?: Link;
        cells?: Link;
        members?: Link;
        owners?: Link;
        labels?: Link;
        org?: Link;
    };
    readonly id?: string;
    meta?: {
        createdAt?: string;
        updatedAt?: string;
    };
    cells?: CellsWithViewProperties;
    labels?: Labels;
};
type CellsWithViewProperties = CellWithViewProperties[];
type CellWithViewProperties = Cell & {
    name?: string;
    properties?: ViewProperties;
};
type ViewProperties = LinePlusSingleStatProperties | XYViewProperties | SingleStatViewProperties | HistogramViewProperties | GaugeViewProperties | TableViewProperties | SimpleTableViewProperties | MarkdownViewProperties | CheckViewProperties | ScatterViewProperties | HeatmapViewProperties | MosaicViewProperties | BandViewProperties | GeoViewProperties;
interface LinePlusSingleStatProperties {
    adaptiveZoomHide?: boolean;
    timeFormat?: string;
    type: 'line-plus-single-stat';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    axes: Axes;
    staticLegend?: StaticLegend;
    xColumn?: string;
    generateXAxisTicks?: string[];
    xTotalTicks?: number;
    xTickStart?: number;
    xTickStep?: number;
    yColumn?: string;
    generateYAxisTicks?: string[];
    yTotalTicks?: number;
    yTickStart?: number;
    yTickStep?: number;
    shadeBelow?: boolean;
    hoverDimension?: 'auto' | 'x' | 'y' | 'xy';
    position: 'overlaid' | 'stacked';
    prefix: string;
    suffix: string;
    decimalPlaces: DecimalPlaces;
    legendColorizeRows?: boolean;
    legendHide?: boolean;
    legendOpacity?: number;
    legendOrientationThreshold?: number;
}
interface DashboardQuery {
    /** The text of the Flux query. */
    text?: string;
    editMode?: QueryEditMode;
    name?: string;
    builderConfig?: BuilderConfig;
}
type QueryEditMode = 'builder' | 'advanced';
interface BuilderConfig {
    buckets?: string[];
    tags?: BuilderTagsType[];
    functions?: BuilderFunctionsType[];
    aggregateWindow?: {
        period?: string;
        fillValues?: boolean;
    };
}
interface BuilderTagsType {
    key?: string;
    values?: string[];
    aggregateFunctionType?: BuilderAggregateFunctionType;
}
type BuilderAggregateFunctionType = 'filter' | 'group';
interface BuilderFunctionsType {
    name?: string;
}
/**
 * Defines an encoding of data value into color space.
 */
interface DashboardColor {
    /** The unique ID of the view color. */
    id: string;
    /** Type is how the color is used. */
    type: 'min' | 'max' | 'threshold' | 'scale' | 'text' | 'background';
    /** The hex number of the color */
    hex: string;
    /** The user-facing name of the hex color. */
    name: string;
    /** The data value mapped to this color. */
    value: number;
}
/**
 * The viewport for a View's visualizations
 */
interface Axes {
    x: Axis;
    y: Axis;
}
/**
 * Axis used in a visualization.
 */
interface Axis {
    /** The extents of the axis in the form [lower, upper]. Clients determine whether bounds are inclusive or exclusive of their limits. */
    bounds?: string[];
    /** Description of the axis. */
    label?: string;
    /** Label prefix for formatting axis values. */
    prefix?: string;
    /** Label suffix for formatting axis values. */
    suffix?: string;
    /** Radix for formatting axis values. */
    base?: '' | '2' | '10';
    scale?: AxisScale;
}
/**
 * Scale is the axis formatting scale. Supported: "log", "linear"
 */
type AxisScale = 'log' | 'linear';
/**
 * StaticLegend represents the options specific to the static legend
 */
interface StaticLegend {
    colorizeRows?: boolean;
    heightRatio?: number;
    show?: boolean;
    opacity?: number;
    orientationThreshold?: number;
    valueAxis?: string;
    widthRatio?: number;
}
/**
 * Indicates whether decimal places should be enforced, and how many digits it should show.
 */
interface DecimalPlaces {
    /** Indicates whether decimal point setting should be enforced */
    isEnforced?: boolean;
    /** The number of digits after decimal to display */
    digits?: number;
}
interface XYViewProperties {
    adaptiveZoomHide?: boolean;
    timeFormat?: string;
    type: 'xy';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    /** An object that contains information about the color mapping */
    colorMapping?: ColorMapping;
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    axes: Axes;
    staticLegend?: StaticLegend;
    xColumn?: string;
    generateXAxisTicks?: string[];
    xTotalTicks?: number;
    xTickStart?: number;
    xTickStep?: number;
    yColumn?: string;
    generateYAxisTicks?: string[];
    yTotalTicks?: number;
    yTickStart?: number;
    yTickStep?: number;
    shadeBelow?: boolean;
    hoverDimension?: 'auto' | 'x' | 'y' | 'xy';
    position: 'overlaid' | 'stacked';
    geom: XYGeom;
    legendColorizeRows?: boolean;
    legendHide?: boolean;
    legendOpacity?: number;
    legendOrientationThreshold?: number;
}
/**
 * A color mapping is an object that maps time series data to a UI color scheme to allow the UI to render graphs consistent colors across reloads.
 */
type ColorMapping = any;
type XYGeom = 'line' | 'step' | 'stacked' | 'bar' | 'monotoneX' | 'stepBefore' | 'stepAfter';
interface SingleStatViewProperties {
    type: 'single-stat';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    prefix: string;
    tickPrefix: string;
    suffix: string;
    tickSuffix: string;
    staticLegend?: StaticLegend;
    decimalPlaces: DecimalPlaces;
}
interface HistogramViewProperties {
    type: 'histogram';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    xColumn: string;
    fillColumns: string[];
    xDomain: number[];
    xAxisLabel: string;
    position: 'overlaid' | 'stacked';
    binCount: number;
    legendColorizeRows?: boolean;
    legendHide?: boolean;
    legendOpacity?: number;
    legendOrientationThreshold?: number;
}
interface GaugeViewProperties {
    type: 'gauge';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    prefix: string;
    tickPrefix: string;
    suffix: string;
    tickSuffix: string;
    decimalPlaces: DecimalPlaces;
}
interface TableViewProperties {
    type: 'table';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    tableOptions: {
        /** verticalTimeAxis describes the orientation of the table by indicating whether the time axis will be displayed vertically */
        verticalTimeAxis?: boolean;
        sortBy?: RenamableField;
        /** Wrapping describes the text wrapping style to be used in table views */
        wrapping?: 'truncate' | 'wrap' | 'single-line';
        /** fixFirstColumn indicates whether the first column of the table should be locked */
        fixFirstColumn?: boolean;
    };
    /** fieldOptions represent the fields retrieved by the query with customization options */
    fieldOptions: RenamableField[];
    /** timeFormat describes the display format for time values according to moment.js date formatting */
    timeFormat: string;
    decimalPlaces: DecimalPlaces;
}
/**
 * Describes a field that can be renamed and made visible or invisible.
 */
interface RenamableField {
    /** The calculated name of a field. */
    readonly internalName?: string;
    /** The name that a field is renamed to by the user. */
    displayName?: string;
    /** Indicates whether this field should be visible on the table. */
    visible?: boolean;
}
interface SimpleTableViewProperties {
    type: 'simple-table';
    showAll: boolean;
    queries: DashboardQuery[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
}
interface MarkdownViewProperties {
    type: 'markdown';
    shape: 'chronograf-v2';
    note: string;
}
interface CheckViewProperties {
    adaptiveZoomHide?: boolean;
    type: 'check';
    shape: 'chronograf-v2';
    checkID: string;
    check?: Check;
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    legendColorizeRows?: boolean;
    legendHide?: boolean;
    legendOpacity?: number;
    legendOrientationThreshold?: number;
}
type Check = CheckDiscriminator;
type CheckDiscriminator = (DeadmanCheck & {
    type: string;
}) | (ThresholdCheck & {
    type: string;
}) | (CustomCheck & {
    type: string;
});
type DeadmanCheck = CheckBase & {
    type: 'deadman';
    /** String duration before deadman triggers. */
    timeSince?: string;
    /** String duration for time that a series is considered stale and should not trigger deadman. */
    staleTime?: string;
    /** If only zero values reported since time, trigger an alert */
    reportZero?: boolean;
    level?: CheckStatusLevel;
    /** Check repetition interval. */
    every?: string;
    /** Duration to delay after the schedule, before executing check. */
    offset?: string;
    /** List of tags to write to each status. */
    tags?: Array<{
        key?: string;
        value?: string;
    }>;
    /** The template used to generate and write a status message. */
    statusMessageTemplate?: string;
};
interface CheckBase {
    readonly id?: string;
    name: string;
    /** The ID of the organization that owns this check. */
    orgID: string;
    /** The ID of the task associated with this check. */
    taskID?: string;
    /** The ID of creator used to create this check. */
    readonly ownerID?: string;
    readonly createdAt?: string;
    readonly updatedAt?: string;
    query: DashboardQuery;
    status?: TaskStatusType;
    /** An optional description of the check. */
    description?: string;
    /** A timestamp ([RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)) of the latest scheduled and completed run. */
    readonly latestCompleted?: string;
    readonly lastRunStatus?: 'failed' | 'success' | 'canceled';
    readonly lastRunError?: string;
    labels?: Labels;
    readonly links?: {
        /** URL for this check */
        self?: Link;
        /** URL to retrieve labels for this check */
        labels?: Link;
        /** URL to retrieve members for this check */
        members?: Link;
        /** URL to retrieve owners for this check */
        owners?: Link;
        /** URL to retrieve flux script for this check */
        query?: Link;
    };
}
/**
 * `inactive` cancels scheduled runs and prevents manual runs of the task.

*/
type TaskStatusType = 'active' | 'inactive';
/**
 * The state to record if check matches a criteria.
 */
type CheckStatusLevel = 'UNKNOWN' | 'OK' | 'INFO' | 'CRIT' | 'WARN';
type ThresholdCheck = CheckBase & {
    type: 'threshold';
    thresholds?: Threshold[];
    /** Check repetition interval. */
    every?: string;
    /** Duration to delay after the schedule, before executing check. */
    offset?: string;
    /** List of tags to write to each status. */
    tags?: Array<{
        key?: string;
        value?: string;
    }>;
    /** The template used to generate and write a status message. */
    statusMessageTemplate?: string;
};
type Threshold = (GreaterThreshold & {
    type: string;
}) | (LesserThreshold & {
    type: string;
}) | (RangeThreshold & {
    type: string;
});
type GreaterThreshold = ThresholdBase & {
    type: 'greater';
    value: number;
};
interface ThresholdBase {
    level?: CheckStatusLevel;
    /** If true, only alert if all values meet threshold. */
    allValues?: boolean;
}
type LesserThreshold = ThresholdBase & {
    type: 'lesser';
    value: number;
};
type RangeThreshold = ThresholdBase & {
    type: 'range';
    min: number;
    max: number;
    within: boolean;
};
type CustomCheck = CheckBase & {
    type: 'custom';
};
interface ScatterViewProperties {
    adaptiveZoomHide?: boolean;
    timeFormat?: string;
    type: 'scatter';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: string[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    xColumn: string;
    generateXAxisTicks?: string[];
    xTotalTicks?: number;
    xTickStart?: number;
    xTickStep?: number;
    yColumn: string;
    generateYAxisTicks?: string[];
    yTotalTicks?: number;
    yTickStart?: number;
    yTickStep?: number;
    fillColumns: string[];
    symbolColumns: string[];
    xDomain: number[];
    yDomain: number[];
    xAxisLabel: string;
    yAxisLabel: string;
    xPrefix: string;
    xSuffix: string;
    yPrefix: string;
    ySuffix: string;
    legendColorizeRows?: boolean;
    legendHide?: boolean;
    legendOpacity?: number;
    legendOrientationThreshold?: number;
}
interface HeatmapViewProperties {
    adaptiveZoomHide?: boolean;
    timeFormat?: string;
    type: 'heatmap';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: string[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    xColumn: string;
    generateXAxisTicks?: string[];
    xTotalTicks?: number;
    xTickStart?: number;
    xTickStep?: number;
    yColumn: string;
    generateYAxisTicks?: string[];
    yTotalTicks?: number;
    yTickStart?: number;
    yTickStep?: number;
    xDomain: number[];
    yDomain: number[];
    xAxisLabel: string;
    yAxisLabel: string;
    xPrefix: string;
    xSuffix: string;
    yPrefix: string;
    ySuffix: string;
    binSize: number;
    legendColorizeRows?: boolean;
    legendHide?: boolean;
    legendOpacity?: number;
    legendOrientationThreshold?: number;
}
interface MosaicViewProperties {
    timeFormat?: string;
    type: 'mosaic';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: string[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    xColumn: string;
    generateXAxisTicks?: string[];
    xTotalTicks?: number;
    xTickStart?: number;
    xTickStep?: number;
    yLabelColumnSeparator?: string;
    yLabelColumns?: string[];
    ySeriesColumns: string[];
    fillColumns: string[];
    xDomain: number[];
    yDomain: number[];
    xAxisLabel: string;
    yAxisLabel: string;
    xPrefix: string;
    xSuffix: string;
    yPrefix: string;
    ySuffix: string;
    hoverDimension?: 'auto' | 'x' | 'y' | 'xy';
    legendColorizeRows?: boolean;
    legendHide?: boolean;
    legendOpacity?: number;
    legendOrientationThreshold?: number;
}
interface BandViewProperties {
    adaptiveZoomHide?: boolean;
    timeFormat?: string;
    type: 'band';
    queries: DashboardQuery[];
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    shape: 'chronograf-v2';
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    axes: Axes;
    staticLegend?: StaticLegend;
    xColumn?: string;
    generateXAxisTicks?: string[];
    xTotalTicks?: number;
    xTickStart?: number;
    xTickStep?: number;
    yColumn?: string;
    generateYAxisTicks?: string[];
    yTotalTicks?: number;
    yTickStart?: number;
    yTickStep?: number;
    upperColumn?: string;
    mainColumn?: string;
    lowerColumn?: string;
    hoverDimension?: 'auto' | 'x' | 'y' | 'xy';
    geom: XYGeom;
    legendColorizeRows?: boolean;
    legendHide?: boolean;
    legendOpacity?: number;
    legendOrientationThreshold?: number;
}
interface GeoViewProperties {
    type: 'geo';
    queries: DashboardQuery[];
    shape: 'chronograf-v2';
    /** Coordinates of the center of the map */
    center: {
        /** Latitude of the center of the map */
        lat: number;
        /** Longitude of the center of the map */
        lon: number;
    };
    /** Zoom level used for initial display of the map */
    zoom: number;
    /** If true, map zoom and pan controls are enabled on the dashboard view */
    allowPanAndZoom: boolean;
    /** If true, search results get automatically regroupped so that lon,lat and value are treated as columns */
    detectCoordinateFields: boolean;
    /** If true, S2 column is used to calculate lat/lon */
    useS2CellID?: boolean;
    /** String to define the column */
    s2Column?: string;
    latLonColumns?: LatLonColumns;
    /** Define map type - regular, satellite etc. */
    mapStyle?: string;
    note: string;
    /** If true, will display note when empty */
    showNoteWhenEmpty: boolean;
    /** Colors define color encoding of data into a visualization */
    colors?: DashboardColor[];
    /** List of individual layers shown in the map */
    layers: GeoViewLayer[];
}
/**
 * Object type to define lat/lon columns
 */
interface LatLonColumns {
    lat: LatLonColumn;
    lon: LatLonColumn;
}
/**
 * Object type for key and column definitions
 */
interface LatLonColumn {
    /** Key to determine whether the column is tag/field */
    key: string;
    /** Column to look up Lat/Lon */
    column: string;
}
type GeoViewLayer = GeoCircleViewLayer | GeoHeatMapViewLayer | GeoPointMapViewLayer | GeoTrackMapViewLayer;
type GeoCircleViewLayer = GeoViewLayerProperties & {
    /** Radius field */
    radiusField: string;
    radiusDimension: Axis;
    /** Circle color field */
    colorField: string;
    colorDimension: Axis;
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    /** Maximum radius size in pixels */
    radius?: number;
    /** Interpolate circle color based on displayed value */
    interpolateColors?: boolean;
};
interface GeoViewLayerProperties {
    type: 'heatmap' | 'circleMap' | 'pointMap' | 'trackMap';
}
type GeoHeatMapViewLayer = GeoViewLayerProperties & {
    /** Intensity field */
    intensityField: string;
    intensityDimension: Axis;
    /** Radius size in pixels */
    radius: number;
    /** Blur for heatmap points */
    blur: number;
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
};
type GeoPointMapViewLayer = GeoViewLayerProperties & {
    /** Marker color field */
    colorField: string;
    colorDimension: Axis;
    /** Colors define color encoding of data into a visualization */
    colors: DashboardColor[];
    /** Cluster close markers together */
    isClustered?: boolean;
    /** An array for which columns to display in tooltip */
    tooltipColumns?: string[];
};
type GeoTrackMapViewLayer = GeoViewLayerProperties & any;
interface CreateCell {
    name?: string;
    x?: number;
    y?: number;
    w?: number;
    h?: number;
    /** Makes a copy of the provided view. */
    usingView?: string;
}
interface CellUpdate {
    x?: number;
    y?: number;
    w?: number;
    h?: number;
}
interface View {
    readonly links?: {
        self?: string;
    };
    readonly id?: string;
    name: string;
    properties: ViewProperties;
}
/**
 * Flux query to be analyzed.
 */
interface LanguageRequest {
    /** The Flux query script to be analyzed.
     */
    query: string;
}
/**
 * Contains the AST for the supplied Flux query
 */
interface ASTResponse {
    ast?: Package;
}
/**
 * Represents a complete package source tree.
 */
interface Package {
    type?: NodeType;
    /** Package import path */
    path?: string;
    /** Package name */
    package?: string;
    /** Package files */
    files?: File[];
}
/**
 * Type of AST node
 */
type NodeType = string;
/**
 * Represents a source from a single file
 */
interface File {
    type?: NodeType;
    /** The name of the file. */
    name?: string;
    package?: PackageClause;
    /** A list of package imports */
    imports?: ImportDeclaration[];
    /** List of Flux statements */
    body?: Statement[];
}
/**
 * Defines a package identifier
 */
interface PackageClause {
    type?: NodeType;
    name?: Identifier;
}
/**
 * A valid Flux identifier
 */
interface Identifier {
    type?: NodeType;
    name?: string;
}
/**
 * Declares a package import
 */
interface ImportDeclaration {
    type?: NodeType;
    as?: Identifier;
    path?: StringLiteral;
}
/**
 * Expressions begin and end with double quote marks
 */
interface StringLiteral {
    type?: NodeType;
    value?: string;
}
type Statement = BadStatement | VariableAssignment | MemberAssignment | ExpressionStatement | ReturnStatement | OptionStatement | BuiltinStatement | TestStatement;
/**
 * A placeholder for statements for which no correct statement nodes can be created
 */
interface BadStatement {
    type?: NodeType;
    /** Raw source text */
    text?: string;
}
/**
 * Represents the declaration of a variable
 */
interface VariableAssignment {
    type?: NodeType;
    id?: Identifier;
    init?: Expression;
}
type Expression = ArrayExpression | DictExpression | FunctionExpression | BinaryExpression | CallExpression | ConditionalExpression | LogicalExpression | MemberExpression | IndexExpression | ObjectExpression | ParenExpression | PipeExpression | UnaryExpression | BooleanLiteral | DateTimeLiteral | DurationLiteral | FloatLiteral | IntegerLiteral | PipeLiteral | RegexpLiteral | StringLiteral | UnsignedIntegerLiteral | Identifier;
/**
 * Used to create and directly specify the elements of an array object
 */
interface ArrayExpression {
    type?: NodeType;
    /** Elements of the array */
    elements?: Expression[];
}
/**
 * Used to create and directly specify the elements of a dictionary
 */
interface DictExpression {
    type?: NodeType;
    /** Elements of the dictionary */
    elements?: DictItem[];
}
/**
 * A key-value pair in a dictionary.
 */
interface DictItem {
    type?: NodeType;
    key?: Expression;
    val?: Expression;
}
/**
 * Function expression
 */
interface FunctionExpression {
    type?: NodeType;
    /** Function parameters */
    params?: Property[];
    body?: Node;
}
/**
 * The value associated with a key
 */
interface Property {
    type?: NodeType;
    key?: PropertyKey;
    value?: Expression;
}
type PropertyKey = Identifier | StringLiteral;
type Node = Expression | Block;
/**
 * A set of statements
 */
interface Block {
    type?: NodeType;
    /** Block body */
    body?: Statement[];
}
/**
 * uses binary operators to act on two operands in an expression
 */
interface BinaryExpression {
    type?: NodeType;
    operator?: string;
    left?: Expression;
    right?: Expression;
}
/**
 * Represents a function call
 */
interface CallExpression {
    type?: NodeType;
    callee?: Expression;
    /** Function arguments */
    arguments?: Expression[];
}
/**
 * Selects one of two expressions, `Alternate` or `Consequent`, depending on a third boolean expression, `Test`
 */
interface ConditionalExpression {
    type?: NodeType;
    test?: Expression;
    alternate?: Expression;
    consequent?: Expression;
}
/**
 * Represents the rule conditions that collectively evaluate to either true or false
 */
interface LogicalExpression {
    type?: NodeType;
    operator?: string;
    left?: Expression;
    right?: Expression;
}
/**
 * Represents accessing a property of an object
 */
interface MemberExpression {
    type?: NodeType;
    object?: Expression;
    property?: PropertyKey;
}
/**
 * Represents indexing into an array
 */
interface IndexExpression {
    type?: NodeType;
    array?: Expression;
    index?: Expression;
}
/**
 * Allows the declaration of an anonymous object within a declaration
 */
interface ObjectExpression {
    type?: NodeType;
    /** Object properties */
    properties?: Property[];
}
/**
 * Represents an expression wrapped in parenthesis
 */
interface ParenExpression {
    type?: NodeType;
    expression?: Expression;
}
/**
 * Call expression with pipe argument
 */
interface PipeExpression {
    type?: NodeType;
    argument?: Expression;
    call?: CallExpression;
}
/**
 * Uses operators to act on a single operand in an expression
 */
interface UnaryExpression {
    type?: NodeType;
    operator?: string;
    argument?: Expression;
}
/**
 * Represents boolean values
 */
interface BooleanLiteral {
    type?: NodeType;
    value?: boolean;
}
/**
 * Represents an instant in time with nanosecond precision in [RFC3339Nano date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339nano-timestamp).
 */
interface DateTimeLiteral {
    type?: NodeType;
    value?: string;
}
/**
 * Represents the elapsed time between two instants as an int64 nanosecond count with syntax of golang's time.Duration
 */
interface DurationLiteral {
    type?: NodeType;
    /** Duration values */
    values?: Duration[];
}
/**
 * A pair consisting of length of time and the unit of time measured. It is the atomic unit from which all duration literals are composed.
 */
interface Duration {
    type?: NodeType;
    magnitude?: number;
    unit?: string;
}
/**
 * Represents floating point numbers according to the double representations defined by the IEEE-754-1985
 */
interface FloatLiteral {
    type?: NodeType;
    value?: number;
}
/**
 * Represents integer numbers
 */
interface IntegerLiteral {
    type?: NodeType;
    value?: string;
}
/**
 * Represents a specialized literal value, indicating the left hand value of a pipe expression
 */
interface PipeLiteral {
    type?: NodeType;
}
/**
 * Expressions begin and end with `/` and are regular expressions with syntax accepted by RE2
 */
interface RegexpLiteral {
    type?: NodeType;
    value?: string;
}
/**
 * Represents integer numbers
 */
interface UnsignedIntegerLiteral {
    type?: NodeType;
    value?: string;
}
/**
 * Object property assignment
 */
interface MemberAssignment {
    type?: NodeType;
    member?: MemberExpression;
    init?: Expression;
}
/**
 * May consist of an expression that doesn't return a value and is executed solely for its side-effects
 */
interface ExpressionStatement {
    type?: NodeType;
    expression?: Expression;
}
/**
 * Defines an expression to return
 */
interface ReturnStatement {
    type?: NodeType;
    argument?: Expression;
}
/**
 * A single variable declaration
 */
interface OptionStatement {
    type?: NodeType;
    assignment?: VariableAssignment | MemberAssignment;
}
/**
 * Declares a builtin identifier and its type
 */
interface BuiltinStatement {
    type?: NodeType;
    id?: Identifier;
}
/**
 * Declares a Flux test case
 */
interface TestStatement {
    type?: NodeType;
    assignment?: VariableAssignment;
}
interface FluxSuggestions {
    funcs?: FluxSuggestion[];
}
interface FluxSuggestion {
    name?: string;
    params?: any;
}
/**
 * Query InfluxDB with the Flux language
 */
interface Query {
    extern?: File;
    /** The query script to execute. */
    query: string;
    /** The type of query. Must be "flux". */
    type?: 'flux';
    /** Key-value pairs passed as parameters during query execution.
  
  To use parameters in your query, pass a _`query`_ with `params` references (in dot notation)--for example:
  
  ```json
    query: "from(bucket: params.mybucket) |> range(start: params.rangeStart) |> limit(n:1)"
  ```
  
  and pass _`params`_ with the key-value pairs--for example:
  
  ```json
    params: {
      "mybucket": "environment",
      "rangeStart": "-30d"
    }
  ```
  
  During query execution, InfluxDB passes _`params`_ to your script and substitutes the values.
  
  #### Limitations
  
  - If you use _`params`_, you can't use _`extern`_.
   */
    params?: any;
    dialect?: Dialect;
    /** Specifies the time that should be reported as `now` in the query.
  Default is the server `now` time.
   */
    now?: string;
}
/**
 * Options for tabular data output.
Default output is [annotated CSV](https://docs.influxdata.com/influxdb/v2.3/reference/syntax/annotated-csv/#csv-response-format) with headers.

For more information about tabular data **dialect**,
see [W3 metadata vocabulary for tabular data](https://www.w3.org/TR/2015/REC-tabular-metadata-20151217/#dialect-descriptions).

*/
interface Dialect {
    /** If true, the results contain a header row. */
    header?: boolean;
    /** The separator used between cells. Default is a comma (`,`). */
    delimiter?: string;
    /** Annotation rows to include in the results.
  An _annotation_ is metadata associated with an object (column) in the data model.
  
  #### Related guides
  
  - See [Annotated CSV annotations](https://docs.influxdata.com/influxdb/v2.3/reference/syntax/annotated-csv/#annotations) for examples and more information.
  
  For more information about **annotations** in tabular data,
  see [W3 metadata vocabulary for tabular data](https://www.w3.org/TR/2015/REC-tabular-data-model-20151217/#columns).
   */
    annotations?: Array<'group' | 'datatype' | 'default'>;
    /** The character prefixed to comment strings. Default is a number sign (`#`). */
    commentPrefix?: string;
    /** The format for timestamps in results.
  Default is [`RFC3339` date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp).
  To include nanoseconds in timestamps, use `RFC3339Nano`.
  
  #### Example formatted date/time values
  
  | Format      | Value                       |
  |:------------|:----------------------------|
  | `RFC3339`    | `"2006-01-02T15:04:05Z07:00"` |
  | `RFC3339Nano` | `"2006-01-02T15:04:05.999999999Z07:00"` |
   */
    dateTimeFormat?: 'RFC3339' | 'RFC3339Nano';
}
interface AnalyzeQueryResponse {
    errors?: Array<{
        line?: number;
        column?: number;
        character?: number;
        message?: string;
    }>;
}
interface Buckets {
    readonly links?: Links;
    buckets?: Bucket[];
}
interface Bucket {
    readonly links?: {
        /** URL to retrieve labels for this bucket. */
        labels?: Link;
        /** URL to retrieve members that can read this bucket. */
        members?: Link;
        /** URL to retrieve parent organization for this bucket. */
        org?: Link;
        /** URL to retrieve owners that can read and write to this bucket. */
        owners?: Link;
        /** URL for this bucket. */
        self?: Link;
        /** URL to write line protocol to this bucket. */
        write?: Link;
    };
    readonly id?: string;
    readonly type?: 'user' | 'system';
    name: string;
    description?: string;
    orgID?: string;
    rp?: string;
    schemaType?: SchemaType;
    readonly createdAt?: string;
    readonly updatedAt?: string;
    retentionRules: RetentionRules;
    labels?: Labels;
}
type SchemaType = 'implicit' | 'explicit';
/**
 * Retention rules to expire or retain data.
#### InfluxDB Cloud

- `retentionRules` is required.

#### InfluxDB OSS

- `retentionRules` isn't required.

*/
type RetentionRules = RetentionRule[];
interface RetentionRule {
    type?: 'expire';
    /** The duration in seconds for how long data will be kept in the database.
  The default duration is 2592000 (30 days).
  0 represents infinite retention.
   */
    everySeconds: number;
    /** The shard group duration.
  The duration or interval (in seconds) that each shard group covers.
  
  #### InfluxDB Cloud
  
  - Does not use `shardGroupDurationsSeconds`.
  
  #### InfluxDB OSS
  
  - Default value depends on the
  [bucket retention period](https://docs.influxdata.com/influxdb/v2.3/v2.3/reference/internals/shards/#shard-group-duration).
   */
    shardGroupDurationSeconds?: number;
}
interface PostBucketRequest {
    /** Organization ID.
  The ID of the organization.
   */
    orgID: string;
    /** The name of the bucket.
     */
    name: string;
    /** A description of the bucket.
     */
    description?: string;
    /** Retention policy is an InfluxDB 1.x concept that represents the duration
  of time that each data point in the retention policy persists. Use `rp`
  for compatibility with InfluxDB 1.x.
  The InfluxDB 2.x and Cloud equivalent is
  [retention period](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#retention-period).
   */
    rp?: string;
    retentionRules?: RetentionRules;
    /** Schema Type.
  Use `explicit` to enforce column names, tags, fields, and data types for
  your data.
  
  #### InfluxDB Cloud
  
  - Default is `implicit`.
  
  #### InfluxDB OSS
  
  - Doesn't support `schemaType`.
   */
    schemaType?: SchemaType;
}
/**
 * An object that contains updated bucket properties to apply.

*/
interface PatchBucketRequest {
    /** The name of the bucket.
     */
    name?: string;
    /** A description of the bucket.
     */
    description?: string;
    retentionRules?: PatchRetentionRules;
}
/**
 * Updates to rules to expire or retain data. No rules means no updates.
 */
type PatchRetentionRules = PatchRetentionRule[];
interface PatchRetentionRule {
    type?: 'expire';
    /** The number of seconds to keep data.
  Default duration is `2592000` (30 days).
  `0` represents infinite retention.
   */
    everySeconds: number;
    /** The [shard group duration](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#shard).
  The number of seconds that each shard group covers.
  
  #### InfluxDB Cloud
  
  - Doesn't use `shardGroupDurationsSeconds`.
  
  #### InfluxDB OSS
  
  - Default value depends on the [bucket retention period](https://docs.influxdata.com/influxdb/v2.3/reference/internals/shards/#shard-group-duration).
  
  #### Related guides
  
  - InfluxDB [shards and shard groups](https://docs.influxdata.com/influxdb/v2.3/reference/internals/shards/)
   */
    shardGroupDurationSeconds?: number;
}
interface Organizations {
    links?: Links;
    orgs?: Organization[];
}
interface Organization {
    readonly links?: {
        self?: Link;
        members?: Link;
        owners?: Link;
        labels?: Link;
        secrets?: Link;
        buckets?: Link;
        tasks?: Link;
        dashboards?: Link;
    };
    readonly id?: string;
    name: string;
    description?: string;
    readonly createdAt?: string;
    readonly updatedAt?: string;
    /** If inactive the organization is inactive. */
    status?: 'active' | 'inactive';
}
interface PostOrganizationRequest {
    /** The name of the organization.
     */
    name: string;
    /** The description of the organization.
     */
    description?: string;
}
/**
 * An object that contains updated organization properties to apply.

*/
interface PatchOrganizationRequest {
    /** The name of the organization.
     */
    name?: string;
    /** The description of the organization.
     */
    description?: string;
}
type SecretKeysResponse = SecretKeys & {
    readonly links?: {
        self?: string;
        org?: string;
    };
};
interface SecretKeys {
    secrets?: string[];
}
type Secrets = any;
interface Stack {
    id?: string;
    orgID?: string;
    readonly createdAt?: string;
    events?: Array<{
        eventType?: string;
        name?: string;
        description?: string;
        sources?: string[];
        resources?: Array<{
            apiVersion?: string;
            resourceID?: string;
            kind?: TemplateKind;
            templateMetaName?: string;
            associations?: Array<{
                kind?: TemplateKind;
                metaName?: string;
            }>;
            links?: {
                self?: string;
            };
        }>;
        urls?: string[];
        readonly updatedAt?: string;
    }>;
}
type TemplateKind = 'Bucket' | 'Check' | 'CheckDeadman' | 'CheckThreshold' | 'Dashboard' | 'Label' | 'NotificationEndpoint' | 'NotificationEndpointHTTP' | 'NotificationEndpointPagerDuty' | 'NotificationEndpointSlack' | 'NotificationRule' | 'Task' | 'Telegraf' | 'Variable';
interface TemplateApply {
    /** Only applies a dry run of the templates passed in the request.
  
  - Validates the template and generates a resource diff and summary.
  - Doesn't install templates or make changes to the InfluxDB instance.
   */
    dryRun?: boolean;
    /** Organization ID.
  InfluxDB applies templates to this organization.
  The organization owns all resources created by the template.
  
  To find your organization, see how to
  [view organizations](https://docs.influxdata.com/influxdb/v2.3/organizations/view-orgs/).
   */
    orgID?: string;
    /** ID of the stack to update.
  
  To apply templates to an existing stack in the organization, use the `stackID` parameter.
  If you apply templates without providing a stack ID,
  InfluxDB initializes a new stack with all new resources.
  
  To find a stack ID, use the InfluxDB [`/api/v2/stacks` API endpoint](#operation/ListStacks) to list stacks.
  
  #### Related guides
  
  - [Stacks](https://docs.influxdata.com/influxdb/v2.3/influxdb-templates/stacks/)
  - [View stacks](https://docs.influxdata.com/influxdb/v2.3/influxdb-templates/stacks/view/)
   */
    stackID?: string;
    /** A template object to apply.
  A template object has a `contents` property
  with an array of InfluxDB resource configurations.
  
  Pass `template` to apply only one template object.
  If you use `template`, you can't use the `templates` parameter.
  If you want to apply multiple template objects, use `templates` instead.
   */
    template?: {
        contentType?: string;
        sources?: string[];
        contents?: Template;
    };
    /** A list of template objects to apply.
  A template object has a `contents` property
  with an array of InfluxDB resource configurations.
  
  Use the `templates` parameter to apply multiple template objects.
  If you use `templates`, you can't use the `template` parameter.
   */
    templates?: Array<{
        contentType?: string;
        sources?: string[];
        contents?: Template;
    }>;
    /** An object with key-value pairs that map to **environment references** in templates.
  
  Environment references in templates are `envRef` objects with an `envRef.key`
  property.
  To substitute a custom environment reference value when applying templates,
  pass `envRefs` with the `envRef.key` and the value.
  
  When you apply a template, InfluxDB replaces `envRef` objects in the template
  with the values that you provide in the `envRefs` parameter.
  For more examples, see how to [define environment references](https://docs.influxdata.com/influxdb/v2.3/influxdb-templates/use/#define-environment-references).
  
  The following template fields may use environment references:
  
    - `metadata.name`
    - `spec.endpointName`
    - `spec.associations.name`
  
  For more information about including environment references in template fields, see how to
  [include user-definable resource names](https://docs.influxdata.com/influxdb/v2.3/influxdb-templates/create/#include-user-definable-resource-names).
   */
    envRefs?: any;
    /** An object with key-value pairs that map to **secrets** in queries.
  
  Queries may reference secrets stored in InfluxDB--for example,
  the following Flux script retrieves `POSTGRES_USERNAME` and `POSTGRES_PASSWORD`
  secrets and then uses them to connect to a PostgreSQL database:
  
  ```js
  import "sql"
  import "influxdata/influxdb/secrets"
  
  username = secrets.get(key: "POSTGRES_USERNAME")
  password = secrets.get(key: "POSTGRES_PASSWORD")
  
  sql.from(
    driverName: "postgres",
    dataSourceName: "postgresql://${username}:${password}@localhost:5432",
    query: "SELECT * FROM example_table",
  )
  ```
  
  To define secret values in your `/api/v2/templates/apply` request,
  pass the `secrets` parameter with key-value pairs--for example:
  
  ```json
  {
    ...
    "secrets": {
      "POSTGRES_USERNAME": "pguser",
      "POSTGRES_PASSWORD": "foo"
    }
    ...
  }
  ```
  
  InfluxDB stores the key-value pairs as secrets that you can access with `secrets.get()`.
  Once stored, you can't view secret values in InfluxDB.
  
  #### Related guides
  
  - [How to pass secrets when installing a template](https://docs.influxdata.com/influxdb/v2.3/influxdb-templates/use/#pass-secrets-when-installing-a-template)
   */
    secrets?: any;
    /** A list of URLs for template files.
  
  To apply a template manifest file located at a URL, pass `remotes`
  with an array that contains the URL.
   */
    remotes?: Array<{
        url: string;
        contentType?: string;
    }>;
    /** A list of `action` objects.
  Actions let you customize how InfluxDB applies templates in the request.
  
  You can use the following actions to prevent creating or updating resources:
  
  - A `skipKind` action skips template resources of a specified `kind`.
  - A `skipResource` action skips template resources with a specified `metadata.name`
    and `kind`.
   */
    actions?: Array<{
        action?: 'skipKind';
        properties?: {
            kind: TemplateKind;
        };
    } | {
        action?: 'skipResource';
        properties?: {
            kind: TemplateKind;
            resourceTemplateName: string;
        };
    }>;
}
type Template = Array<{
    apiVersion?: string;
    kind?: TemplateKind;
    /** Metadata properties used for the resource when the template is applied.
     */
    metadata?: {
        name?: string;
    };
    /** Configuration properties used for the resource when the template is applied.
  Key-value pairs map to the specification for the resource.
  
  The following code samples show `spec` configurations for template resources:
  
  - A bucket:
  
  ```json
  { "spec": {
      "name": "iot_center",
      "retentionRules": [{
          "everySeconds": 2.592e+06,
          "type": "expire"
        }]
    }
  }
  ```
  
  - A variable:
  
  ```json
  { "spec": {
      "language": "flux",
      "name": "Node_Service",
      "query": "import \"influxdata/influxdb/v1\"\r\nv1.tagValues(bucket: \"iot_center\",
          tag: \"service\")",
      "type": "query"
    }
  }
  ```
   */
    spec?: any;
}>;
interface TemplateSummary {
    sources?: string[];
    stackID?: string;
    summary?: {
        buckets?: Array<{
            id?: string;
            orgID?: string;
            kind?: TemplateKind;
            templateMetaName?: string;
            name?: string;
            description?: string;
            retentionPeriod?: number;
            labelAssociations?: TemplateSummaryLabel[];
            envReferences?: TemplateEnvReferences;
        }>;
        checks?: Array<CheckDiscriminator & {
            kind?: TemplateKind;
            templateMetaName?: string;
            labelAssociations?: TemplateSummaryLabel[];
            envReferences?: TemplateEnvReferences;
        }>;
        dashboards?: Array<{
            id?: string;
            orgID?: string;
            kind?: TemplateKind;
            templateMetaName?: string;
            name?: string;
            description?: string;
            labelAssociations?: TemplateSummaryLabel[];
            charts?: TemplateChart[];
            envReferences?: TemplateEnvReferences;
        }>;
        labels?: TemplateSummaryLabel[];
        labelMappings?: Array<{
            status?: string;
            resourceTemplateMetaName?: string;
            resourceName?: string;
            resourceID?: string;
            resourceType?: string;
            labelTemplateMetaName?: string;
            labelName?: string;
            labelID?: string;
        }>;
        missingEnvRefs?: string[];
        missingSecrets?: string[];
        notificationEndpoints?: Array<NotificationEndpointDiscriminator & {
            kind?: TemplateKind;
            templateMetaName?: string;
            labelAssociations?: TemplateSummaryLabel[];
            envReferences?: TemplateEnvReferences;
        }>;
        notificationRules?: Array<{
            kind?: TemplateKind;
            templateMetaName?: string;
            name?: string;
            description?: string;
            endpointTemplateMetaName?: string;
            endpointID?: string;
            endpointType?: string;
            every?: string;
            offset?: string;
            messageTemplate?: string;
            status?: string;
            statusRules?: Array<{
                currentLevel?: string;
                previousLevel?: string;
            }>;
            tagRules?: Array<{
                key?: string;
                value?: string;
                operator?: string;
            }>;
            labelAssociations?: TemplateSummaryLabel[];
            envReferences?: TemplateEnvReferences;
        }>;
        tasks?: Array<{
            kind?: TemplateKind;
            templateMetaName?: string;
            id?: string;
            name?: string;
            cron?: string;
            description?: string;
            every?: string;
            offset?: string;
            query?: string;
            status?: string;
            envReferences?: TemplateEnvReferences;
        }>;
        telegrafConfigs?: Array<TelegrafRequest & {
            kind?: TemplateKind;
            templateMetaName?: string;
            labelAssociations?: TemplateSummaryLabel[];
            envReferences?: TemplateEnvReferences;
        }>;
        variables?: Array<{
            kind?: TemplateKind;
            templateMetaName?: string;
            id?: string;
            orgID?: string;
            name?: string;
            description?: string;
            arguments?: VariableProperties;
            labelAssociations?: TemplateSummaryLabel[];
            envReferences?: TemplateEnvReferences;
        }>;
    };
    diff?: {
        buckets?: Array<{
            kind?: TemplateKind;
            stateStatus?: string;
            id?: string;
            templateMetaName?: string;
            new?: {
                name?: string;
                description?: string;
                retentionRules?: RetentionRules;
            };
            old?: {
                name?: string;
                description?: string;
                retentionRules?: RetentionRules;
            };
        }>;
        checks?: Array<{
            kind?: TemplateKind;
            stateStatus?: string;
            id?: string;
            templateMetaName?: string;
            new?: CheckDiscriminator;
            old?: CheckDiscriminator;
        }>;
        dashboards?: Array<{
            stateStatus?: string;
            id?: string;
            kind?: TemplateKind;
            templateMetaName?: string;
            new?: {
                name?: string;
                description?: string;
                charts?: TemplateChart[];
            };
            old?: {
                name?: string;
                description?: string;
                charts?: TemplateChart[];
            };
        }>;
        labels?: Array<{
            stateStatus?: string;
            kind?: TemplateKind;
            id?: string;
            templateMetaName?: string;
            new?: {
                name?: string;
                color?: string;
                description?: string;
            };
            old?: {
                name?: string;
                color?: string;
                description?: string;
            };
        }>;
        labelMappings?: Array<{
            status?: string;
            resourceType?: string;
            resourceID?: string;
            resourceTemplateMetaName?: string;
            resourceName?: string;
            labelID?: string;
            labelTemplateMetaName?: string;
            labelName?: string;
        }>;
        notificationEndpoints?: Array<{
            kind?: TemplateKind;
            stateStatus?: string;
            id?: string;
            templateMetaName?: string;
            new?: NotificationEndpointDiscriminator;
            old?: NotificationEndpointDiscriminator;
        }>;
        notificationRules?: Array<{
            kind?: TemplateKind;
            stateStatus?: string;
            id?: string;
            templateMetaName?: string;
            new?: {
                name?: string;
                description?: string;
                endpointName?: string;
                endpointID?: string;
                endpointType?: string;
                every?: string;
                offset?: string;
                messageTemplate?: string;
                status?: string;
                statusRules?: Array<{
                    currentLevel?: string;
                    previousLevel?: string;
                }>;
                tagRules?: Array<{
                    key?: string;
                    value?: string;
                    operator?: string;
                }>;
            };
            old?: {
                name?: string;
                description?: string;
                endpointName?: string;
                endpointID?: string;
                endpointType?: string;
                every?: string;
                offset?: string;
                messageTemplate?: string;
                status?: string;
                statusRules?: Array<{
                    currentLevel?: string;
                    previousLevel?: string;
                }>;
                tagRules?: Array<{
                    key?: string;
                    value?: string;
                    operator?: string;
                }>;
            };
        }>;
        tasks?: Array<{
            kind?: TemplateKind;
            stateStatus?: string;
            id?: string;
            templateMetaName?: string;
            new?: {
                name?: string;
                cron?: string;
                description?: string;
                every?: string;
                offset?: string;
                query?: string;
                status?: string;
            };
            old?: {
                name?: string;
                cron?: string;
                description?: string;
                every?: string;
                offset?: string;
                query?: string;
                status?: string;
            };
        }>;
        telegrafConfigs?: Array<{
            kind?: TemplateKind;
            stateStatus?: string;
            id?: string;
            templateMetaName?: string;
            new?: TelegrafRequest;
            old?: TelegrafRequest;
        }>;
        variables?: Array<{
            kind?: TemplateKind;
            stateStatus?: string;
            id?: string;
            templateMetaName?: string;
            new?: {
                name?: string;
                description?: string;
                args?: VariableProperties;
            };
            old?: {
                name?: string;
                description?: string;
                args?: VariableProperties;
            };
        }>;
    };
    errors?: Array<{
        kind?: TemplateKind;
        reason?: string;
        fields?: string[];
        indexes?: number[];
    }>;
}
interface TemplateSummaryLabel {
    id?: string;
    orgID?: string;
    kind?: TemplateKind;
    templateMetaName?: string;
    name?: string;
    properties?: {
        color?: string;
        description?: string;
    };
    envReferences?: TemplateEnvReferences;
}
type TemplateEnvReferences = Array<{
    /** Field the environment reference corresponds too */
    resourceField: string;
    /** Key identified as environment reference and is the key identified in the template */
    envRefKey: string;
    /** Value provided to fulfill reference */
    value?: string | number | number | boolean;
    /** Default value that will be provided for the reference when no value is provided */
    defaultValue?: string | number | number | boolean;
}>;
interface TemplateChart {
    xPos?: number;
    yPos?: number;
    height?: number;
    width?: number;
    properties?: ViewProperties;
}
type NotificationEndpointDiscriminator = (SlackNotificationEndpoint & {
    type: string;
}) | (PagerDutyNotificationEndpoint & {
    type: string;
}) | (HTTPNotificationEndpoint & {
    type: string;
}) | (TelegramNotificationEndpoint & {
    type: string;
});
type SlackNotificationEndpoint = NotificationEndpointBase & {
    /** Specifies the URL of the Slack endpoint. Specify either `URL` or `Token`. */
    url?: string;
    /** Specifies the API token string. Specify either `URL` or `Token`. */
    token?: string;
};
interface NotificationEndpointBase {
    id?: string;
    orgID?: string;
    userID?: string;
    readonly createdAt?: string;
    readonly updatedAt?: string;
    /** An optional description of the notification endpoint. */
    description?: string;
    name: string;
    /** The status of the endpoint. */
    status?: 'active' | 'inactive';
    labels?: Labels;
    readonly links?: {
        /** URL for this endpoint. */
        self?: Link;
        /** URL to retrieve labels for this endpoint. */
        labels?: Link;
        /** URL to retrieve members for this endpoint. */
        members?: Link;
        /** URL to retrieve owners for this endpoint. */
        owners?: Link;
    };
    type: NotificationEndpointType;
}
type NotificationEndpointType = 'slack' | 'pagerduty' | 'http' | 'telegram';
type PagerDutyNotificationEndpoint = NotificationEndpointBase & {
    clientURL?: string;
    routingKey: string;
};
type HTTPNotificationEndpoint = NotificationEndpointBase & {
    url: string;
    username?: string;
    password?: string;
    token?: string;
    method: 'POST' | 'GET' | 'PUT';
    authMethod: 'none' | 'basic' | 'bearer';
    contentTemplate?: string;
    /** Customized headers. */
    headers?: any;
};
type TelegramNotificationEndpoint = NotificationEndpointBase & {
    /** Specifies the Telegram bot token. See https://core.telegram.org/bots#creating-a-new-bot . */
    token: string;
    /** The ID of the telegram channel; a chat_id in https://core.telegram.org/bots/api#sendmessage . */
    channel: string;
};
type VariableProperties = QueryVariableProperties | ConstantVariableProperties | MapVariableProperties;
interface QueryVariableProperties {
    type?: 'query';
    values?: {
        query?: string;
        language?: string;
    };
}
interface ConstantVariableProperties {
    type?: 'constant';
    values?: string[];
}
interface MapVariableProperties {
    type?: 'map';
    values?: any;
}
interface TemplateExportByID {
    stackID?: string;
    orgIDs?: Array<{
        orgID?: string;
        resourceFilters?: {
            byLabel?: string[];
            byResourceKind?: TemplateKind[];
        };
    }>;
    resources?: Array<{
        id: string;
        kind: TemplateKind;
        /** if defined with id, name is used for resource exported by id. if defined independently, resources strictly matching name are exported */
        name?: string;
    }>;
}
interface TemplateExportByName {
    stackID?: string;
    orgIDs?: Array<{
        orgID?: string;
        resourceFilters?: {
            byLabel?: string[];
            byResourceKind?: TemplateKind[];
        };
    }>;
    resources?: Array<{
        kind: TemplateKind;
        name: string;
    }>;
}
interface Runs {
    links?: Links;
    runs?: Run[];
}
interface Run {
    readonly id?: string;
    readonly taskID?: string;
    readonly status?: 'scheduled' | 'started' | 'failed' | 'success' | 'canceled';
    /** The time [RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp) used for the run's `now` option. */
    scheduledFor?: string;
    /** An array of logs associated with the run. */
    readonly log?: LogEvent[];
    /** Flux used for the task */
    readonly flux?: string;
    /** The time ([RFC3339Nano date/time format](https://go.dev/src/time/format.go)) the run started executing. */
    readonly startedAt?: string;
    /** The time ([RFC3339Nano date/time format](https://go.dev/src/time/format.go)) the run finished executing. */
    readonly finishedAt?: string;
    /** The time ([RFC3339Nano date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339nano-timestamp)) the run was manually requested. */
    readonly requestedAt?: string;
    readonly links?: {
        self?: string;
        task?: string;
        retry?: string;
    };
}
interface LogEvent {
    /** The time ([RFC3339Nano date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339nano-timestamp)) that the event occurred. */
    readonly time?: string;
    /** A description of the event that occurred. */
    readonly message?: string;
    /** The ID of the task run that generated the event. */
    readonly runID?: string;
}
interface RunManually {
    /** The time [RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)
  used for the run's `now` option.
  Default is the server _now_ time.
   */
    scheduledFor?: string;
}
interface Logs {
    readonly events?: LogEvent[];
}
type Flags = any;
interface PasswordResetBody {
    password: string;
}
interface Checks {
    checks?: Check[];
    links?: Links;
}
type PostCheck = CheckDiscriminator;
interface CheckPatch {
    name?: string;
    description?: string;
    status?: 'active' | 'inactive';
}
interface NotificationRules {
    notificationRules?: NotificationRule[];
    links?: Links;
}
type NotificationRule = NotificationRuleDiscriminator;
type NotificationRuleDiscriminator = (SlackNotificationRule & {
    type: string;
}) | (SMTPNotificationRule & {
    type: string;
}) | (PagerDutyNotificationRule & {
    type: string;
}) | (HTTPNotificationRule & {
    type: string;
}) | (TelegramNotificationRule & {
    type: string;
});
type SlackNotificationRule = NotificationRuleBase & SlackNotificationRuleBase;
interface NotificationRuleBase {
    /** A timestamp ([RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)) of the latest scheduled and completed run. */
    readonly latestCompleted?: string;
    readonly lastRunStatus?: 'failed' | 'success' | 'canceled';
    readonly lastRunError?: string;
    readonly id?: string;
    endpointID: string;
    /** The ID of the organization that owns this notification rule. */
    orgID: string;
    /** The ID of the task associated with this notification rule. */
    taskID?: string;
    /** The ID of creator used to create this notification rule. */
    readonly ownerID?: string;
    readonly createdAt?: string;
    readonly updatedAt?: string;
    status: TaskStatusType;
    /** Human-readable name describing the notification rule. */
    name: string;
    sleepUntil?: string;
    /** The notification repetition interval. */
    every?: string;
    /** Duration to delay after the schedule, before executing check. */
    offset?: string;
    runbookLink?: string;
    /** Don't notify me more than <limit> times every <limitEvery> seconds. If set, limit cannot be empty. */
    limitEvery?: number;
    /** Don't notify me more than <limit> times every <limitEvery> seconds. If set, limitEvery cannot be empty. */
    limit?: number;
    /** List of tag rules the notification rule attempts to match. */
    tagRules?: TagRule[];
    /** An optional description of the notification rule. */
    description?: string;
    /** List of status rules the notification rule attempts to match. */
    statusRules: StatusRule[];
    labels?: Labels;
    readonly links?: {
        /** URL for this endpoint. */
        self?: Link;
        /** URL to retrieve labels for this notification rule. */
        labels?: Link;
        /** URL to retrieve members for this notification rule. */
        members?: Link;
        /** URL to retrieve owners for this notification rule. */
        owners?: Link;
        /** URL to retrieve flux script for this notification rule. */
        query?: Link;
    };
}
interface TagRule {
    key?: string;
    value?: string;
    operator?: 'equal' | 'notequal' | 'equalregex' | 'notequalregex';
}
interface StatusRule {
    currentLevel?: RuleStatusLevel;
    previousLevel?: RuleStatusLevel;
    count?: number;
    period?: string;
}
/**
 * The state to record if check matches a criteria.
 */
type RuleStatusLevel = 'UNKNOWN' | 'OK' | 'INFO' | 'CRIT' | 'WARN' | 'ANY';
interface SlackNotificationRuleBase {
    type: 'slack';
    channel?: string;
    messageTemplate: string;
}
type SMTPNotificationRule = NotificationRuleBase & SMTPNotificationRuleBase;
interface SMTPNotificationRuleBase {
    type: 'smtp';
    subjectTemplate: string;
    bodyTemplate?: string;
    to: string;
}
type PagerDutyNotificationRule = NotificationRuleBase & PagerDutyNotificationRuleBase;
interface PagerDutyNotificationRuleBase {
    type: 'pagerduty';
    messageTemplate: string;
}
type HTTPNotificationRule = NotificationRuleBase & HTTPNotificationRuleBase;
interface HTTPNotificationRuleBase {
    type: 'http';
    url?: string;
}
type TelegramNotificationRule = NotificationRuleBase & TelegramNotificationRuleBase;
interface TelegramNotificationRuleBase {
    /** The discriminator between other types of notification rules is "telegram". */
    type: 'telegram';
    /** The message template as a flux interpolated string. */
    messageTemplate: string;
    /** Parse mode of the message text per https://core.telegram.org/bots/api#formatting-options. Defaults to "MarkdownV2". */
    parseMode?: 'MarkdownV2' | 'HTML' | 'Markdown';
    /** Disables preview of web links in the sent messages when "true". Defaults to "false". */
    disableWebPagePreview?: boolean;
}
type PostNotificationRule = NotificationRuleDiscriminator;
/**
 * Rendered flux that backs the check or notification.
 */
interface FluxResponse {
    flux?: string;
}
interface NotificationRuleUpdate {
    name?: string;
    description?: string;
    status?: 'active' | 'inactive';
}
interface NotificationEndpoints {
    notificationEndpoints?: NotificationEndpoint[];
    links?: Links;
}
type NotificationEndpoint = NotificationEndpointDiscriminator;
type PostNotificationEndpoint = NotificationEndpointDiscriminator;
interface NotificationEndpointUpdate {
    name?: string;
    description?: string;
    status?: 'active' | 'inactive';
}
interface HealthCheck {
    name: string;
    message?: string;
    checks?: HealthCheck[];
    status: 'pass' | 'fail';
    version?: string;
    commit?: string;
}
interface Ready {
    status?: 'ready';
    started?: string;
    up?: string;
}
interface Users {
    links?: {
        self?: string;
    };
    users?: UserResponse[];
}
interface User {
    readonly id?: string;
    oauthID?: string;
    name: string;
    /** If inactive the user is inactive. */
    status?: 'active' | 'inactive';
}
interface IsOnboarding {
    /** If `true`, the InfluxDB instance hasn't had initial setup;
  `false` otherwise.
   */
    allowed?: boolean;
}
interface OnboardingRequest {
    username: string;
    password?: string;
    org: string;
    bucket: string;
    retentionPeriodSeconds?: number;
    /** Retention period *in nanoseconds* for the new bucket. This key's name has been misleading since OSS 2.0 GA, please transition to use `retentionPeriodSeconds`
     */
    retentionPeriodHrs?: number;
    /** Authentication token to set on the initial user. If not specified, the server will generate a token.
     */
    token?: string;
}
interface OnboardingResponse {
    user?: UserResponse;
    org?: Organization;
    bucket?: Bucket;
    auth?: Authorization;
}
type Authorization = AuthorizationUpdateRequest & {
    readonly createdAt?: string;
    readonly updatedAt?: string;
    /** The ID of the organization. */
    orgID?: string;
    /** A list of permissions for an authorization.
  An authorization must have at least one permission.
   */
    permissions?: Permission[];
    readonly id?: string;
    /** The API token for authenticating InfluxDB API and CLI requests.
     */
    readonly token?: string;
    /** The ID of the user that the token is scoped to. */
    readonly userID?: string;
    /** The name of the user that the token is scoped to. */
    readonly user?: string;
    /** The name of the organization that owns the token. */
    readonly org?: string;
    readonly links?: {
        readonly self?: Link;
        readonly user?: Link;
    };
};
interface AuthorizationUpdateRequest {
    /** Status of the token. If `inactive`, requests using the token will be rejected. */
    status?: 'active' | 'inactive';
    /** A description of the token. */
    description?: string;
}
interface Permission {
    action: 'read' | 'write';
    resource: Resource;
}
interface Resource {
    /** The type of resource.
  In a `permission`, applies the permission to all resources of this type.
   */
    type: 'authorizations' | 'buckets' | 'dashboards' | 'orgs' | 'sources' | 'tasks' | 'telegrafs' | 'users' | 'variables' | 'scrapers' | 'secrets' | 'labels' | 'views' | 'documents' | 'notificationRules' | 'notificationEndpoints' | 'checks' | 'dbrp' | 'notebooks' | 'annotations' | 'remotes' | 'replications' | 'instance' | 'flows' | 'functions';
    /** The ID of a specific resource.
  In a `permission`, applies the permission to only the resource with this ID.
   */
    id?: string;
    /** Optional: A name for the resource.
  Not all resource types have a name field.
   */
    name?: string;
    /** The ID of the organization that owns the resource.
  In a `permission`, applies the permission to all resources of `type` owned by this organization.
   */
    orgID?: string;
    /** Optional: The name of the organization with `orgID`.
     */
    org?: string;
}
interface Authorizations {
    readonly links?: Links;
    authorizations?: Authorization[];
}
type AuthorizationPostRequest = AuthorizationUpdateRequest & {
    /** The ID of the organization that owns the authorization.
     */
    orgID?: string;
    /** The ID of the user that the authorization is scoped to.
     */
    userID?: string;
    /** A list of permissions for an authorization.
  An authorization must have at least one permission.
   */
    permissions?: Permission[];
};
interface Variables {
    variables?: Variable[];
}
interface Variable {
    readonly links?: {
        self?: string;
        org?: string;
        labels?: string;
    };
    readonly id?: string;
    orgID: string;
    name: string;
    description?: string;
    selected?: string[];
    labels?: Labels;
    arguments: VariableProperties;
    createdAt?: string;
    updatedAt?: string;
}
interface Sources {
    links?: {
        self?: string;
    };
    sources?: Source[];
}
interface Source {
    links?: {
        self?: string;
        query?: string;
        health?: string;
        buckets?: string;
    };
    id?: string;
    orgID?: string;
    default?: boolean;
    name?: string;
    type?: 'v1' | 'v2' | 'self';
    url?: string;
    insecureSkipVerify?: boolean;
    telegraf?: string;
    token?: string;
    username?: string;
    password?: string;
    sharedSecret?: string;
    metaUrl?: string;
    defaultRP?: string;
    readonly languages?: Array<'flux' | 'influxql'>;
}
interface ScraperTargetResponses {
    configurations?: ScraperTargetResponse[];
}
type ScraperTargetResponse = ScraperTargetRequest & {
    readonly id?: string;
    /** The name of the organization. */
    org?: string;
    /** The bucket name. */
    bucket?: string;
    readonly links?: {
        self?: Link;
        members?: Link;
        owners?: Link;
        bucket?: Link;
        organization?: Link;
    };
};
interface ScraperTargetRequest {
    /** The name of the scraper target. */
    name?: string;
    /** The type of the metrics to be parsed. */
    type?: 'prometheus';
    /** The URL of the metrics endpoint. */
    url?: string;
    /** The organization ID. */
    orgID?: string;
    /** The ID of the bucket to write to. */
    bucketID?: string;
    /** Skip TLS verification on endpoint. */
    allowInsecure?: boolean;
}
interface MetadataBackup {
    kv: string;
    sql: string;
    buckets: BucketMetadataManifests;
}
type BucketMetadataManifests = BucketMetadataManifest[];
interface BucketMetadataManifest {
    organizationID: string;
    organizationName: string;
    bucketID: string;
    bucketName: string;
    description?: string;
    defaultRetentionPolicy: string;
    retentionPolicies: RetentionPolicyManifests;
}
type RetentionPolicyManifests = RetentionPolicyManifest[];
interface RetentionPolicyManifest {
    name: string;
    replicaN: number;
    duration: number;
    shardGroupDuration: number;
    shardGroups: ShardGroupManifests;
    subscriptions: SubscriptionManifests;
}
type ShardGroupManifests = ShardGroupManifest[];
interface ShardGroupManifest {
    id: number;
    startTime: string;
    endTime: string;
    deletedAt?: string;
    truncatedAt?: string;
    shards: ShardManifests;
}
type ShardManifests = ShardManifest[];
interface ShardManifest {
    id: number;
    shardOwners: ShardOwners;
}
type ShardOwners = ShardOwner[];
interface ShardOwner {
    /** The ID of the node that owns the shard. */
    nodeID: number;
}
type SubscriptionManifests = SubscriptionManifest[];
interface SubscriptionManifest {
    name: string;
    mode: string;
    destinations: string[];
}
interface RestoredBucketMappings {
    /** New ID of the restored bucket */
    id: string;
    name: string;
    shardMappings: BucketShardMappings;
}
type BucketShardMappings = BucketShardMapping[];
interface BucketShardMapping {
    oldId: number;
    newId: number;
}
interface Config {
    config?: any;
}
interface RemoteConnections {
    remotes?: RemoteConnection[];
}
interface RemoteConnection {
    id: string;
    name: string;
    orgID: string;
    description?: string;
    remoteURL: string;
    remoteOrgID: string;
    allowInsecureTLS: boolean;
}
interface RemoteConnectionCreationRequest {
    name: string;
    description?: string;
    orgID: string;
    remoteURL: string;
    remoteAPIToken: string;
    remoteOrgID: string;
    allowInsecureTLS: boolean;
}
interface RemoteConnectionUpdateRequest {
    name?: string;
    description?: string;
    remoteURL?: string;
    remoteAPIToken?: string;
    remoteOrgID?: string;
    allowInsecureTLS?: boolean;
}
interface Replications {
    replications?: Replication[];
}
interface Replication {
    id: string;
    name: string;
    description?: string;
    orgID: string;
    remoteID: string;
    localBucketID: string;
    remoteBucketID?: string;
    remoteBucketName?: string;
    maxQueueSizeBytes: number;
    currentQueueSizeBytes: number;
    latestResponseCode?: number;
    latestErrorMessage?: string;
    dropNonRetryableData?: boolean;
}
interface ReplicationCreationRequest {
    name: string;
    description?: string;
    orgID: string;
    remoteID: string;
    localBucketID: string;
    remoteBucketID?: string;
    remoteBucketName?: string;
    maxQueueSizeBytes: number;
    dropNonRetryableData?: boolean;
    maxAgeSeconds: number;
}
interface ReplicationUpdateRequest {
    name?: string;
    description?: string;
    remoteID?: string;
    remoteBucketID?: string;
    remoteBucketName?: string;
    maxQueueSizeBytes?: number;
    dropNonRetryableData?: boolean;
    maxAgeSeconds?: number;
}
interface Dashboards {
    links?: Links;
    dashboards?: Dashboard[];
}
interface Tasks {
    readonly links?: Links;
    tasks?: Task[];
}
interface Task {
    readonly id: string;
    /** An [organization](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#organization) ID.
  Specifies the organization that owns the task.
   */
    orgID: string;
    /** An [organization](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#organization) name.
  Specifies the organization that owns the task.
   */
    org?: string;
    /** The name of the task. */
    name: string;
    /** A [user](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#user) ID.
  Specifies the owner of the task.
  
  To find a user ID, you can use the
  [`GET /api/v2/users` endpoint](#operation/GetUsers) to
  list users.
   */
    ownerID?: string;
    /** A description of the task. */
    description?: string;
    status?: TaskStatusType;
    labels?: Labels;
    /** An authorization ID.
  Specifies the authorization used when the task communicates with the query engine.
  
  To find an authorization ID, you can use the
  [`GET /api/v2/authorizations` endpoint](#operation/GetAuthorizations) to
  list authorizations.
   */
    authorizationID?: string;
    /** The Flux script that the task executes. */
    flux: string;
    /** The interval ([duration literal](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)) at which the task runs. `every` also determines when the task first runs, depending on the specified time. */
    every?: string;
    /** A [Cron expression](https://en.wikipedia.org/wiki/Cron#Overview) that defines the schedule on which the task runs. InfluxDB uses the system time when evaluating Cron expressions. */
    cron?: string;
    /** A [duration](https://docs.influxdata.com/flux/v0.x/spec/lexical-elements/#duration-literals) to delay execution of the task after the scheduled time has elapsed. `0` removes the offset. */
    offset?: string;
    /** A timestamp ([RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)) of the latest scheduled and completed run. */
    readonly latestCompleted?: string;
    readonly lastRunStatus?: 'failed' | 'success' | 'canceled';
    readonly lastRunError?: string;
    readonly createdAt?: string;
    readonly updatedAt?: string;
    readonly links?: {
        self?: Link;
        owners?: Link;
        members?: Link;
        runs?: Link;
        logs?: Link;
        labels?: Link;
    };
}
interface TaskCreateRequest {
    /** The ID of the organization that owns this Task. */
    orgID?: string;
    /** The name of the organization that owns this Task. */
    org?: string;
    status?: TaskStatusType;
    /** The Flux script to run for this task. */
    flux: string;
    /** An optional description of the task. */
    description?: string;
}
interface TaskUpdateRequest {
    status?: TaskStatusType;
    /** The Flux script that the task runs. */
    flux?: string;
    /** Update the 'name' option in the flux script. */
    name?: string;
    /** Update the 'every' option in the flux script. */
    every?: string;
    /** Update the 'cron' option in the flux script. */
    cron?: string;
    /** Update the 'offset' option in the flux script. */
    offset?: string;
    /** Update the description of the task. */
    description?: string;
}
interface Scripts {
    scripts?: Script[];
}
interface Script {
    readonly id?: string;
    name: string;
    description?: string;
    orgID: string;
    /** The script to execute. */
    script: string;
    language?: ScriptLanguage;
    /** The invocation endpoint address. */
    url?: string;
    readonly createdAt?: string;
    readonly updatedAt?: string;
    /** The list of label names associated with the script. */
    labels?: string[];
}
type ScriptLanguage = 'flux';
interface ScriptCreateRequest {
    /** Script name. The name must be unique within the organization. */
    name: string;
    /** Script description. A description of the script. */
    description: string;
    /** The script to execute. */
    script: string;
    language: ScriptLanguage;
}
interface ScriptUpdateRequest {
    name?: string;
    description?: string;
    /** script is script to be executed */
    script?: string;
}
interface ScriptInvocationParams {
    /** The script parameters.
  `params` contains key-value pairs that map values to the **params.keys**
  in a script.
  When you invoke a script with `params`, InfluxDB passes the values as
  invocation parameters to the script.
   */
    params?: any;
}
/**
 * The response body contains the results of the executed script.
The response is user-defined and dynamic.

*/
type ScriptHTTPResponseData = string;

interface RequestOptions {
    /** HTTP request headers */
    headers?: {
        [key: string]: string;
    };
    /**
     * Informs about a start of response processing.
     * @param headers - response HTTP headers
     * @param statusCode - response status code
     */
    responseStarted?: (headers: Headers, statusCode?: number) => void;
}

interface GetRoutesRequest {
}
/**
 * Root API
 */
declare class RootAPI {
    private base;
    /**
     * Creates RootAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all top level routes.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetRoutes }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getRoutes(request?: GetRoutesRequest, requestOptions?: RequestOptions): Promise<Routes>;
}

interface GetAuthorizationsRequest {
    /** A user ID.
  Only returns authorizations scoped to this user.
   */
    userID?: string;
    /** A user name.
  Only returns authorizations scoped to this user.
   */
    user?: string;
    /** An organization ID. Only returns authorizations that belong to this organization. */
    orgID?: string;
    /** An organization name.
  Only returns authorizations that belong to this organization.
   */
    org?: string;
}
interface PostAuthorizationsRequest {
    /** The authorization to create. */
    body: AuthorizationPostRequest;
}
interface GetAuthorizationsIDRequest {
    /** The ID of the authorization to get. */
    authID: string;
}
interface PatchAuthorizationsIDRequest {
    /** The ID of the authorization to update. */
    authID: string;
    /** Authorization to update */
    body: AuthorizationUpdateRequest;
}
interface DeleteAuthorizationsIDRequest {
    /** The ID of the authorization to delete. */
    authID: string;
}
/**
 * Authorizations API
 */
declare class AuthorizationsAPI {
    private base;
    /**
     * Creates AuthorizationsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List authorizations.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetAuthorizations }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getAuthorizations(request?: GetAuthorizationsRequest, requestOptions?: RequestOptions): Promise<Authorizations>;
    /**
     * Create an authorization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostAuthorizations }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postAuthorizations(request: PostAuthorizationsRequest, requestOptions?: RequestOptions): Promise<Authorization>;
    /**
     * Retrieve an authorization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetAuthorizationsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getAuthorizationsID(request: GetAuthorizationsIDRequest, requestOptions?: RequestOptions): Promise<Authorization>;
    /**
     * Update an authorization to be active or inactive.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchAuthorizationsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchAuthorizationsID(request: PatchAuthorizationsIDRequest, requestOptions?: RequestOptions): Promise<Authorization>;
    /**
     * Delete an authorization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteAuthorizationsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteAuthorizationsID(request: DeleteAuthorizationsIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetBackupKVRequest {
}
interface GetBackupMetadataRequest {
}
interface GetBackupShardIdRequest {
    /** The shard ID. */
    shardID: number;
    /** The earliest time [RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp) to include in the snapshot. */
    since?: string;
}
/**
 * Backup API
 */
declare class BackupAPI {
    private base;
    /**
     * Creates BackupAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Download snapshot of metadata stored in the server's embedded KV store. Don't use with InfluxDB versions greater than InfluxDB 2.1.x.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetBackupKV }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getBackupKV(request?: GetBackupKVRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Download snapshot of all metadata in the server.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetBackupMetadata }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getBackupMetadata(request?: GetBackupMetadataRequest, requestOptions?: RequestOptions): Promise<MetadataBackup>;
    /**
     * Download snapshot of all TSM data in a shard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetBackupShardId }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getBackupShardId(request: GetBackupShardIdRequest, requestOptions?: RequestOptions): Promise<string>;
}

interface GetBucketsRequest {
    /** The offset for pagination.
  The number of records to skip.
   */
    offset?: number;
    /** Limits the number of records returned. Default is `20`.
     */
    limit?: number;
    /** Resource ID to seek from. Results are not inclusive of this ID. Use `after` instead of `offset`.
     */
    after?: string;
    /** Organization name.
  The name of the organization.
  
  #### InfluxDB Cloud
  
   - Doesn't use `org` or `orgID`.
   - Creates a bucket in the organization associated with the authorization (API token).
  
   #### InfluxDB OSS
  
   - Accepts either `org` or `orgID`.
   - InfluxDB creates the bucket within this organization.
   */
    org?: string;
    /** Organization ID.
  The organization ID.
  
  #### InfluxDB Cloud
  
   - Doesn't use `org` or `orgID`.
   - Creates a bucket in the organization associated with the authorization (API token).
  
   #### InfluxDB OSS
  
   - Accepts either `org` or `orgID`.
   - InfluxDB creates the bucket within this organization.
   */
    orgID?: string;
    /** Bucket name.
  Only returns buckets with this specific name.
   */
    name?: string;
    /** Bucket ID.
  Only returns the bucket with this ID.
   */
    id?: string;
}
interface PostBucketsRequest {
    /** Bucket to create */
    body: PostBucketRequest;
}
interface GetBucketsIDRequest {
    /** The ID of the bucket to retrieve.
     */
    bucketID: string;
}
interface PatchBucketsIDRequest {
    /** The bucket ID. */
    bucketID: string;
    /** The bucket update to apply. */
    body: PatchBucketRequest;
}
interface DeleteBucketsIDRequest {
    /** Bucket ID.
  The ID of the bucket to delete.
   */
    bucketID: string;
}
interface GetBucketsIDLabelsRequest {
    /** The ID of the bucket to retrieve labels for.
     */
    bucketID: string;
}
interface PostBucketsIDLabelsRequest {
    /** Bucket ID.
  The ID of the bucket to label.
   */
    bucketID: string;
    /** An object that contains a _`labelID`_ to add to the bucket. */
    body: LabelMapping;
}
interface DeleteBucketsIDLabelsIDRequest {
    /** The bucket ID. */
    bucketID: string;
    /** The ID of the label to delete. */
    labelID: string;
}
interface GetBucketsIDMembersRequest {
    /** The ID of the bucket to retrieve users for.
     */
    bucketID: string;
}
interface PostBucketsIDMembersRequest {
    /** The ID of the bucket to retrieve users for.
     */
    bucketID: string;
    /** A user to add as a member to the bucket. */
    body: AddResourceMemberRequestBody;
}
interface DeleteBucketsIDMembersIDRequest {
    /** The ID of the user to remove.
     */
    userID: string;
    /** The ID of the bucket to remove a user from.
     */
    bucketID: string;
}
interface GetBucketsIDOwnersRequest {
    /** The bucket ID. */
    bucketID: string;
}
interface PostBucketsIDOwnersRequest {
    /** The bucket ID. */
    bucketID: string;
    /** User to add as owner */
    body: AddResourceMemberRequestBody;
}
interface DeleteBucketsIDOwnersIDRequest {
    /** The ID of the owner to remove. */
    userID: string;
    /** The bucket ID. */
    bucketID: string;
}
/**
 * Buckets API
 */
declare class BucketsAPI {
    private base;
    /**
     * Creates BucketsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List buckets.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetBuckets }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getBuckets(request?: GetBucketsRequest, requestOptions?: RequestOptions): Promise<Buckets>;
    /**
     * Create a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostBuckets }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postBuckets(request: PostBucketsRequest, requestOptions?: RequestOptions): Promise<Bucket>;
    /**
     * Retrieve a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetBucketsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getBucketsID(request: GetBucketsIDRequest, requestOptions?: RequestOptions): Promise<Bucket>;
    /**
     * Update a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchBucketsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchBucketsID(request: PatchBucketsIDRequest, requestOptions?: RequestOptions): Promise<Bucket>;
    /**
     * Delete a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteBucketsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteBucketsID(request: DeleteBucketsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all labels for a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetBucketsIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getBucketsIDLabels(request: GetBucketsIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostBucketsIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postBucketsIDLabels(request: PostBucketsIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete a label from a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteBucketsIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteBucketsIDLabelsID(request: DeleteBucketsIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all users with member privileges for a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetBucketsIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getBucketsIDMembers(request: GetBucketsIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMembers>;
    /**
     * Add a member to a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostBucketsIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postBucketsIDMembers(request: PostBucketsIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMember>;
    /**
     * Remove a member from a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteBucketsIDMembersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteBucketsIDMembersID(request: DeleteBucketsIDMembersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all owners of a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetBucketsIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getBucketsIDOwners(request: GetBucketsIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwners>;
    /**
     * Add an owner to a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostBucketsIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postBucketsIDOwners(request: PostBucketsIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwner>;
    /**
     * Remove an owner from a bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteBucketsIDOwnersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteBucketsIDOwnersID(request: DeleteBucketsIDOwnersIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetChecksRequest {
    /** The offset for pagination.
  The number of records to skip.
   */
    offset?: number;
    /** Limits the number of records returned. Default is `20`.
     */
    limit?: number;
    /** Only show checks that belong to a specific organization ID. */
    orgID: string;
}
interface CreateCheckRequest {
    /** Check to create */
    body: PostCheck;
}
interface GetChecksIDRequest {
    /** The check ID. */
    checkID: string;
}
interface PutChecksIDRequest {
    /** The check ID. */
    checkID: string;
    /** Check update to apply */
    body: Check;
}
interface PatchChecksIDRequest {
    /** The check ID. */
    checkID: string;
    /** Check update to apply */
    body: CheckPatch;
}
interface DeleteChecksIDRequest {
    /** The check ID. */
    checkID: string;
}
interface GetChecksIDLabelsRequest {
    /** The check ID. */
    checkID: string;
}
interface PostChecksIDLabelsRequest {
    /** The check ID. */
    checkID: string;
    /** Label to add */
    body: LabelMapping;
}
interface DeleteChecksIDLabelsIDRequest {
    /** The check ID. */
    checkID: string;
    /** The ID of the label to delete. */
    labelID: string;
}
interface GetChecksIDQueryRequest {
    /** The check ID. */
    checkID: string;
}
/**
 * Checks API
 */
declare class ChecksAPI {
    private base;
    /**
     * Creates ChecksAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all checks.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetChecks }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getChecks(request: GetChecksRequest, requestOptions?: RequestOptions): Promise<Checks>;
    /**
     * Add new check.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/CreateCheck }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    createCheck(request: CreateCheckRequest, requestOptions?: RequestOptions): Promise<Check>;
    /**
     * Retrieve a check.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetChecksID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getChecksID(request: GetChecksIDRequest, requestOptions?: RequestOptions): Promise<Check>;
    /**
     * Update a check.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PutChecksID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    putChecksID(request: PutChecksIDRequest, requestOptions?: RequestOptions): Promise<Check>;
    /**
     * Update a check.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchChecksID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchChecksID(request: PatchChecksIDRequest, requestOptions?: RequestOptions): Promise<Check>;
    /**
     * Delete a check.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteChecksID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteChecksID(request: DeleteChecksIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all labels for a check.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetChecksIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getChecksIDLabels(request: GetChecksIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a check.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostChecksIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postChecksIDLabels(request: PostChecksIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete label from a check.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteChecksIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteChecksIDLabelsID(request: DeleteChecksIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Retrieve a check query.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetChecksIDQuery }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getChecksIDQuery(request: GetChecksIDQueryRequest, requestOptions?: RequestOptions): Promise<FluxResponse>;
}

interface GetConfigRequest {
}
/**
 * Config API
 */
declare class ConfigAPI {
    private base;
    /**
     * Creates ConfigAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Retrieve runtime configuration.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetConfig }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getConfig(request?: GetConfigRequest, requestOptions?: RequestOptions): Promise<Config>;
}

interface GetDashboardsIDRequest {
    /** The ID of the dashboard to update. */
    dashboardID: string;
    /** If `properties`, includes the cell view properties in the response. */
    include?: string;
}
interface PatchDashboardsIDRequest {
    /** The ID of the dashboard to update. */
    dashboardID: string;
    /** Patching of a dashboard */
    body: {
        /** optional, when provided will replace the name */
        name?: string;
        /** optional, when provided will replace the description */
        description?: string;
        /** optional, when provided will replace all existing cells with the cells provided */
        cells?: CellWithViewProperties;
    };
}
interface DeleteDashboardsIDRequest {
    /** The ID of the dashboard to update. */
    dashboardID: string;
}
interface PostDashboardsIDCellsRequest {
    /** The ID of the dashboard to update. */
    dashboardID: string;
    /** Cell that will be added */
    body: CreateCell;
}
interface PutDashboardsIDCellsRequest {
    /** The ID of the dashboard to update. */
    dashboardID: string;
    /** entity body */
    body: Cells;
}
interface PatchDashboardsIDCellsIDRequest {
    /** The ID of the dashboard to update. */
    dashboardID: string;
    /** The ID of the cell to update. */
    cellID: string;
    /** entity body */
    body: CellUpdate;
}
interface DeleteDashboardsIDCellsIDRequest {
    /** The ID of the dashboard to delete. */
    dashboardID: string;
    /** The ID of the cell to delete. */
    cellID: string;
}
interface GetDashboardsIDCellsIDViewRequest {
    /** The dashboard ID. */
    dashboardID: string;
    /** The cell ID. */
    cellID: string;
}
interface PatchDashboardsIDCellsIDViewRequest {
    /** The ID of the dashboard to update. */
    dashboardID: string;
    /** The ID of the cell to update. */
    cellID: string;
    /** entity body */
    body: View;
}
interface GetDashboardsIDLabelsRequest {
    /** The dashboard ID. */
    dashboardID: string;
}
interface PostDashboardsIDLabelsRequest {
    /** The dashboard ID. */
    dashboardID: string;
    /** Label to add */
    body: LabelMapping;
}
interface DeleteDashboardsIDLabelsIDRequest {
    /** The dashboard ID. */
    dashboardID: string;
    /** The ID of the label to delete. */
    labelID: string;
}
interface GetDashboardsIDMembersRequest {
    /** The dashboard ID. */
    dashboardID: string;
}
interface PostDashboardsIDMembersRequest {
    /** The dashboard ID. */
    dashboardID: string;
    /** User to add as member */
    body: AddResourceMemberRequestBody;
}
interface DeleteDashboardsIDMembersIDRequest {
    /** The ID of the member to remove. */
    userID: string;
    /** The dashboard ID. */
    dashboardID: string;
}
interface GetDashboardsIDOwnersRequest {
    /** The dashboard ID. */
    dashboardID: string;
}
interface PostDashboardsIDOwnersRequest {
    /** The dashboard ID. */
    dashboardID: string;
    /** User to add as owner */
    body: AddResourceMemberRequestBody;
}
interface DeleteDashboardsIDOwnersIDRequest {
    /** The ID of the owner to remove. */
    userID: string;
    /** The dashboard ID. */
    dashboardID: string;
}
interface GetDashboardsRequest {
    /** The offset for pagination.
  The number of records to skip.
   */
    offset?: number;
    /** Limits the number of records returned. Default is `20`.
     */
    limit?: number;
    descending?: any;
    /** A user identifier. Returns only dashboards where this user has the `owner` role. */
    owner?: string;
    /** The column to sort by. */
    sortBy?: string;
    /** A list of dashboard identifiers. Returns only the listed dashboards. If both `id` and `owner` are specified, only `id` is used. */
    id?: any;
    /** The identifier of the organization. */
    orgID?: string;
    /** The name of the organization. */
    org?: string;
}
interface PostDashboardsRequest {
    /** Dashboard to create */
    body: CreateDashboardRequest;
}
/**
 * Dashboards API
 */
declare class DashboardsAPI {
    private base;
    /**
     * Creates DashboardsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Retrieve a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDashboardsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDashboardsID(request: GetDashboardsIDRequest, requestOptions?: RequestOptions): Promise<Dashboard | DashboardWithViewProperties>;
    /**
     * Update a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchDashboardsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchDashboardsID(request: PatchDashboardsIDRequest, requestOptions?: RequestOptions): Promise<Dashboard>;
    /**
     * Delete a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteDashboardsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteDashboardsID(request: DeleteDashboardsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Create a dashboard cell.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostDashboardsIDCells }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postDashboardsIDCells(request: PostDashboardsIDCellsRequest, requestOptions?: RequestOptions): Promise<Cell>;
    /**
     * Replace cells in a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PutDashboardsIDCells }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    putDashboardsIDCells(request: PutDashboardsIDCellsRequest, requestOptions?: RequestOptions): Promise<Dashboard>;
    /**
     * Update the non-positional information related to a cell.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchDashboardsIDCellsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchDashboardsIDCellsID(request: PatchDashboardsIDCellsIDRequest, requestOptions?: RequestOptions): Promise<Cell>;
    /**
     * Delete a dashboard cell.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteDashboardsIDCellsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteDashboardsIDCellsID(request: DeleteDashboardsIDCellsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Retrieve the view for a cell.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDashboardsIDCellsIDView }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDashboardsIDCellsIDView(request: GetDashboardsIDCellsIDViewRequest, requestOptions?: RequestOptions): Promise<View>;
    /**
     * Update the view for a cell.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchDashboardsIDCellsIDView }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchDashboardsIDCellsIDView(request: PatchDashboardsIDCellsIDViewRequest, requestOptions?: RequestOptions): Promise<View>;
    /**
     * List all labels for a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDashboardsIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDashboardsIDLabels(request: GetDashboardsIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostDashboardsIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postDashboardsIDLabels(request: PostDashboardsIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete a label from a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteDashboardsIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteDashboardsIDLabelsID(request: DeleteDashboardsIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all dashboard members.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDashboardsIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDashboardsIDMembers(request: GetDashboardsIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMembers>;
    /**
     * Add a member to a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostDashboardsIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postDashboardsIDMembers(request: PostDashboardsIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMember>;
    /**
     * Remove a member from a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteDashboardsIDMembersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteDashboardsIDMembersID(request: DeleteDashboardsIDMembersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all dashboard owners.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDashboardsIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDashboardsIDOwners(request: GetDashboardsIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwners>;
    /**
     * Add an owner to a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostDashboardsIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postDashboardsIDOwners(request: PostDashboardsIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwner>;
    /**
     * Remove an owner from a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteDashboardsIDOwnersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteDashboardsIDOwnersID(request: DeleteDashboardsIDOwnersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all dashboards.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDashboards }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDashboards(request?: GetDashboardsRequest, requestOptions?: RequestOptions): Promise<Dashboards>;
    /**
     * Create a dashboard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostDashboards }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postDashboards(request: PostDashboardsRequest, requestOptions?: RequestOptions): Promise<Dashboard | DashboardWithViewProperties>;
}

interface GetDBRPsRequest {
    /** Specifies the organization ID to filter on */
    orgID?: string;
    /** Specifies the organization name to filter on */
    org?: string;
    /** Specifies the mapping ID to filter on */
    id?: string;
    /** Specifies the bucket ID to filter on */
    bucketID?: string;
    /** Specifies filtering on default */
    default?: any;
    /** Specifies the database to filter on */
    db?: string;
    /** Specifies the retention policy to filter on */
    rp?: string;
}
interface PostDBRPRequest {
    /** The database retention policy mapping to add */
    body: DBRPCreate;
}
interface GetDBRPsIDRequest {
    /** The database retention policy mapping ID */
    dbrpID: string;
    /** Specifies the organization ID of the mapping */
    orgID?: string;
    /** Specifies the organization name of the mapping */
    org?: string;
}
interface PatchDBRPIDRequest {
    /** The database retention policy mapping. */
    dbrpID: string;
    /** Database retention policy update to apply */
    body: DBRPUpdate;
    /** Specifies the organization ID of the mapping */
    orgID?: string;
    /** Specifies the organization name of the mapping */
    org?: string;
}
interface DeleteDBRPIDRequest {
    /** The database retention policy mapping */
    dbrpID: string;
    /** Specifies the organization ID of the mapping */
    orgID?: string;
    /** Specifies the organization name of the mapping */
    org?: string;
}
/**
 * Dbrps API
 */
declare class DbrpsAPI {
    private base;
    /**
     * Creates DbrpsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List database retention policy mappings.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDBRPs }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDBRPs(request?: GetDBRPsRequest, requestOptions?: RequestOptions): Promise<DBRPs>;
    /**
     * Add a database retention policy mapping.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostDBRP }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postDBRP(request: PostDBRPRequest, requestOptions?: RequestOptions): Promise<DBRP>;
    /**
     * Retrieve a database retention policy mapping.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDBRPsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDBRPsID(request: GetDBRPsIDRequest, requestOptions?: RequestOptions): Promise<DBRPGet>;
    /**
     * Update a database retention policy mapping.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchDBRPID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchDBRPID(request: PatchDBRPIDRequest, requestOptions?: RequestOptions): Promise<DBRPGet>;
    /**
     * Delete a database retention policy.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteDBRPID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteDBRPID(request: DeleteDBRPIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetDebugPprofAllProfilesRequest {
    /** Collects and returns CPU profiling data for the specified [duration](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#duration).
     */
    cpu?: string;
}
interface GetDebugPprofAllocsRequest {
    /** - `0`: (Default) Return the report as a gzip-compressed protocol buffer.
  - `1`: Return a response body with the report formatted as human-readable text.
    The report contains comments that translate addresses to function names and line numbers for debugging.
  
  `debug=1` is mutually exclusive with the `seconds` query parameter.
   */
    debug?: number;
    /** Number of seconds to collect statistics.
  
  `seconds` is mutually exclusive with `debug=1`.
   */
    seconds?: string;
}
interface GetDebugPprofBlockRequest {
    /** - `0`: (Default) Return the report as a gzip-compressed protocol buffer.
  - `1`: Return a response body with the report formatted as human-readable text.
    The report contains comments that translate addresses to function names and line numbers for debugging.
  
  `debug=1` is mutually exclusive with the `seconds` query parameter.
   */
    debug?: number;
    /** Number of seconds to collect statistics.
  
  `seconds` is mutually exclusive with `debug=1`.
   */
    seconds?: string;
}
interface GetDebugPprofCmdlineRequest {
}
interface GetDebugPprofGoroutineRequest {
    /** - `0`: (Default) Return the report as a gzip-compressed protocol buffer.
  - `1`: Return a response body with the report formatted as
         human-readable text with comments that translate addresses to
         function names and line numbers for debugging.
  
  `debug=1` is mutually exclusive with the `seconds` query parameter.
   */
    debug?: number;
    /** Number of seconds to collect statistics.
  
  `seconds` is mutually exclusive with `debug=1`.
   */
    seconds?: string;
}
interface GetDebugPprofHeapRequest {
    /** - `0`: (Default) Return the report as a gzip-compressed protocol buffer.
  - `1`: Return a response body with the report formatted as human-readable text.
    The report contains comments that translate addresses to function names and line numbers for debugging.
  
  `debug=1` is mutually exclusive with the `seconds` query parameter.
   */
    debug?: number;
    /** Number of seconds to collect statistics.
  
  `seconds` is mutually exclusive with `debug=1`.
   */
    seconds?: string;
    /** - `0`: (Default) don't force garbage collection before sampling.
  - `1`: Force garbage collection before sampling.
   */
    gc?: number;
}
interface GetDebugPprofMutexRequest {
    /** - `0`: (Default) Return the report as a gzip-compressed protocol buffer.
  - `1`: Return a response body with the report formatted as human-readable text.
    The report contains comments that translate addresses to function names and line numbers for debugging.
  
  `debug=1` is mutually exclusive with the `seconds` query parameter.
   */
    debug?: number;
    /** Number of seconds to collect statistics.
  
  `seconds` is mutually exclusive with `debug=1`.
   */
    seconds?: string;
}
interface GetDebugPprofProfileRequest {
    /** Number of seconds to collect profile data. Default is `30` seconds. */
    seconds?: string;
}
interface GetDebugPprofThreadCreateRequest {
    /** - `0`: (Default) Return the report as a gzip-compressed protocol buffer.
  - `1`: Return a response body with the report formatted as human-readable text.
    The report contains comments that translate addresses to function names and line numbers for debugging.
  
  `debug=1` is mutually exclusive with the `seconds` query parameter.
   */
    debug?: number;
    /** Number of seconds to collect statistics.
  
  `seconds` is mutually exclusive with `debug=1`.
   */
    seconds?: string;
}
interface GetDebugPprofTraceRequest {
    /** Number of seconds to collect profile data. */
    seconds?: string;
}
/**
 * Debug API
 */
declare class DebugAPI {
    private base;
    /**
     * Creates DebugAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Retrieve all runtime profiles.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofAllProfiles }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofAllProfiles(request?: GetDebugPprofAllProfilesRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the memory allocations runtime profile.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofAllocs }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofAllocs(request?: GetDebugPprofAllocsRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the block runtime profile.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofBlock }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofBlock(request?: GetDebugPprofBlockRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the command line invocation.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofCmdline }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofCmdline(request?: GetDebugPprofCmdlineRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the goroutines runtime profile.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofGoroutine }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofGoroutine(request?: GetDebugPprofGoroutineRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the heap runtime profile.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofHeap }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofHeap(request?: GetDebugPprofHeapRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the mutual exclusion (mutex) runtime profile.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofMutex }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofMutex(request?: GetDebugPprofMutexRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the CPU runtime profile.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofProfile }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofProfile(request?: GetDebugPprofProfileRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the threadcreate runtime profile.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofThreadCreate }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofThreadCreate(request?: GetDebugPprofThreadCreateRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Retrieve the runtime execution trace.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetDebugPprofTrace }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getDebugPprofTrace(request?: GetDebugPprofTraceRequest, requestOptions?: RequestOptions): Promise<string>;
}

interface PostDeleteRequest {
    /** Time range parameters and an optional **delete predicate expression**.
  
  To select points to delete within the specified time range, pass a
  **delete predicate expression** in the `predicate` property of the request body.
  If you don't pass a `predicate`, InfluxDB deletes all data with timestamps
  in the specified time range.
  
  #### Related guides
  
  - [Delete data](https://docs.influxdata.com/influxdb/v2.3/write-data/delete-data/).
  - Learn how to use [delete predicate syntax](https://docs.influxdata.com/influxdb/v2.3/reference/syntax/delete-predicate/).
   */
    body: DeletePredicateRequest;
    /** The organization to delete data from.
  If you pass both `orgID` and `org`, they must both be valid.
  
  #### InfluxDB Cloud
  
  - Doesn't require `org` or `orgID`.
  - Deletes data from the bucket in the organization associated with the authorization (API token).
  
  #### InfluxDB OSS
  
  - Requires either `org` or `orgID`.
   */
    org?: string;
    /** The name or ID of the bucket to delete data from.
  If you pass both `bucket` and `bucketID`, `bucketID` takes precedence.
   */
    bucket?: string;
    /** The ID of the organization to delete data from.
  If you pass both `orgID` and `org`, they must both be valid.
  
  #### InfluxDB Cloud
  
  - Doesn't require `org` or `orgID`.
  - Deletes data from the bucket in the organization associated with the authorization (API token).
  
  #### InfluxDB OSS
  
  - Requires either `org` or `orgID`.
   */
    orgID?: string;
    /** The ID of the bucket to delete data from.
  If you pass both `bucket` and `bucketID`, `bucketID` takes precedence.
   */
    bucketID?: string;
}
/**
 * Delete API
 */
declare class DeleteAPI {
    private base;
    /**
     * Creates DeleteAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Delete data.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostDelete }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postDelete(request: PostDeleteRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetFlagsRequest {
}
/**
 * Flags API
 */
declare class FlagsAPI {
    private base;
    /**
     * Creates FlagsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Return the feature flags for the currently authenticated user.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetFlags }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getFlags(request?: GetFlagsRequest, requestOptions?: RequestOptions): Promise<Flags>;
}

interface GetHealthRequest {
}
/**
 * Health API
 */
declare class HealthAPI {
    private base;
    /**
     * Creates HealthAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Retrieve the health of the instance.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetHealth }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getHealth(request?: GetHealthRequest, requestOptions?: RequestOptions): Promise<HealthCheck>;
}

interface GetLabelsRequest {
    /** The organization ID. */
    orgID?: string;
}
interface PostLabelsRequest {
    /** The label to create. */
    body: LabelCreateRequest;
}
interface GetLabelsIDRequest {
    /** The ID of the label to update. */
    labelID: string;
}
interface PatchLabelsIDRequest {
    /** The ID of the label to update. */
    labelID: string;
    /** A label update. */
    body: LabelUpdate;
}
interface DeleteLabelsIDRequest {
    /** The ID of the label to delete. */
    labelID: string;
}
/**
 * Labels API
 */
declare class LabelsAPI {
    private base;
    /**
     * Creates LabelsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all labels.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getLabels(request?: GetLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Create a label.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postLabels(request: PostLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Retrieve a label.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getLabelsID(request: GetLabelsIDRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Update a label.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchLabelsID(request: PatchLabelsIDRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete a label.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteLabelsID(request: DeleteLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetMeRequest {
}
interface PutMePasswordRequest {
    auth?: {
        user: string;
        password: string;
    };
    /** The new password. */
    body: PasswordResetBody;
}
/**
 * Me API
 */
declare class MeAPI {
    private base;
    /**
     * Creates MeAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Retrieve the currently authenticated user.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetMe }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getMe(request?: GetMeRequest, requestOptions?: RequestOptions): Promise<UserResponse>;
    /**
     * Update a password.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PutMePassword }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    putMePassword(request: PutMePasswordRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetMetricsRequest {
}
/**
 * Metrics API
 */
declare class MetricsAPI {
    private base;
    /**
     * Creates MetricsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Retrieve workload performance metrics.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetMetrics }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getMetrics(request?: GetMetricsRequest, requestOptions?: RequestOptions): Promise<string>;
}

interface GetNotificationEndpointsRequest {
    /** The offset for pagination.
  The number of records to skip.
   */
    offset?: number;
    /** Limits the number of records returned. Default is `20`.
     */
    limit?: number;
    /** Only show notification endpoints that belong to specific organization ID. */
    orgID: string;
}
interface CreateNotificationEndpointRequest {
    /** Notification endpoint to create */
    body: PostNotificationEndpoint;
}
interface GetNotificationEndpointsIDRequest {
    /** The notification endpoint ID. */
    endpointID: string;
}
interface PutNotificationEndpointsIDRequest {
    /** The notification endpoint ID. */
    endpointID: string;
    /** A new notification endpoint to replace the existing endpoint with */
    body: NotificationEndpoint;
}
interface PatchNotificationEndpointsIDRequest {
    /** The notification endpoint ID. */
    endpointID: string;
    /** Check update to apply */
    body: NotificationEndpointUpdate;
}
interface DeleteNotificationEndpointsIDRequest {
    /** The notification endpoint ID. */
    endpointID: string;
}
interface GetNotificationEndpointsIDLabelsRequest {
    /** The notification endpoint ID. */
    endpointID: string;
}
interface PostNotificationEndpointIDLabelsRequest {
    /** The notification endpoint ID. */
    endpointID: string;
    /** Label to add */
    body: LabelMapping;
}
interface DeleteNotificationEndpointsIDLabelsIDRequest {
    /** The notification endpoint ID. */
    endpointID: string;
    /** The ID of the label to delete. */
    labelID: string;
}
/**
 * NotificationEndpoints API
 */
declare class NotificationEndpointsAPI {
    private base;
    /**
     * Creates NotificationEndpointsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all notification endpoints.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetNotificationEndpoints }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getNotificationEndpoints(request: GetNotificationEndpointsRequest, requestOptions?: RequestOptions): Promise<NotificationEndpoints>;
    /**
     * Add a notification endpoint.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/CreateNotificationEndpoint }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    createNotificationEndpoint(request: CreateNotificationEndpointRequest, requestOptions?: RequestOptions): Promise<NotificationEndpoint>;
    /**
     * Retrieve a notification endpoint.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetNotificationEndpointsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getNotificationEndpointsID(request: GetNotificationEndpointsIDRequest, requestOptions?: RequestOptions): Promise<NotificationEndpoint>;
    /**
     * Update a notification endpoint.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PutNotificationEndpointsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    putNotificationEndpointsID(request: PutNotificationEndpointsIDRequest, requestOptions?: RequestOptions): Promise<NotificationEndpoint>;
    /**
     * Update a notification endpoint.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchNotificationEndpointsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchNotificationEndpointsID(request: PatchNotificationEndpointsIDRequest, requestOptions?: RequestOptions): Promise<NotificationEndpoint>;
    /**
     * Delete a notification endpoint.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteNotificationEndpointsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteNotificationEndpointsID(request: DeleteNotificationEndpointsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all labels for a notification endpoint.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetNotificationEndpointsIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getNotificationEndpointsIDLabels(request: GetNotificationEndpointsIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a notification endpoint.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostNotificationEndpointIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postNotificationEndpointIDLabels(request: PostNotificationEndpointIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete a label from a notification endpoint.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteNotificationEndpointsIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteNotificationEndpointsIDLabelsID(request: DeleteNotificationEndpointsIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetNotificationRulesRequest {
    /** The offset for pagination.
  The number of records to skip.
   */
    offset?: number;
    /** Limits the number of records returned. Default is `20`.
     */
    limit?: number;
    /** Only show notification rules that belong to a specific organization ID. */
    orgID: string;
    /** Only show notifications that belong to the specific check ID. */
    checkID?: string;
    /** Only return notification rules that "would match" statuses which contain the tag key value pairs provided. */
    tag?: string;
}
interface CreateNotificationRuleRequest {
    /** Notification rule to create */
    body: PostNotificationRule;
}
interface GetNotificationRulesIDRequest {
    /** The notification rule ID. */
    ruleID: string;
}
interface PutNotificationRulesIDRequest {
    /** The notification rule ID. */
    ruleID: string;
    /** Notification rule update to apply */
    body: NotificationRule;
}
interface PatchNotificationRulesIDRequest {
    /** The notification rule ID. */
    ruleID: string;
    /** Notification rule update to apply */
    body: NotificationRuleUpdate;
}
interface DeleteNotificationRulesIDRequest {
    /** The notification rule ID. */
    ruleID: string;
}
interface GetNotificationRulesIDLabelsRequest {
    /** The notification rule ID. */
    ruleID: string;
}
interface PostNotificationRuleIDLabelsRequest {
    /** The notification rule ID. */
    ruleID: string;
    /** Label to add */
    body: LabelMapping;
}
interface DeleteNotificationRulesIDLabelsIDRequest {
    /** The notification rule ID. */
    ruleID: string;
    /** The ID of the label to delete. */
    labelID: string;
}
interface GetNotificationRulesIDQueryRequest {
    /** The notification rule ID. */
    ruleID: string;
}
/**
 * NotificationRules API
 */
declare class NotificationRulesAPI {
    private base;
    /**
     * Creates NotificationRulesAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all notification rules.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetNotificationRules }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getNotificationRules(request: GetNotificationRulesRequest, requestOptions?: RequestOptions): Promise<NotificationRules>;
    /**
     * Add a notification rule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/CreateNotificationRule }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    createNotificationRule(request: CreateNotificationRuleRequest, requestOptions?: RequestOptions): Promise<NotificationRule>;
    /**
     * Retrieve a notification rule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetNotificationRulesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getNotificationRulesID(request: GetNotificationRulesIDRequest, requestOptions?: RequestOptions): Promise<NotificationRule>;
    /**
     * Update a notification rule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PutNotificationRulesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    putNotificationRulesID(request: PutNotificationRulesIDRequest, requestOptions?: RequestOptions): Promise<NotificationRule>;
    /**
     * Update a notification rule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchNotificationRulesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchNotificationRulesID(request: PatchNotificationRulesIDRequest, requestOptions?: RequestOptions): Promise<NotificationRule>;
    /**
     * Delete a notification rule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteNotificationRulesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteNotificationRulesID(request: DeleteNotificationRulesIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all labels for a notification rule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetNotificationRulesIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getNotificationRulesIDLabels(request: GetNotificationRulesIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a notification rule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostNotificationRuleIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postNotificationRuleIDLabels(request: PostNotificationRuleIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete label from a notification rule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteNotificationRulesIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteNotificationRulesIDLabelsID(request: DeleteNotificationRulesIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Retrieve a notification rule query.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetNotificationRulesIDQuery }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getNotificationRulesIDQuery(request: GetNotificationRulesIDQueryRequest, requestOptions?: RequestOptions): Promise<FluxResponse>;
}

interface GetOrgsRequest {
    /** The offset for pagination.
  The number of records to skip.
   */
    offset?: number;
    /** Limits the number of records returned. Default is `20`.
     */
    limit?: number;
    descending?: any;
    /** An organization name.
  Only returns organizations with this name.
   */
    org?: string;
    /** An organization ID.
  Only returns the organization with this ID.
   */
    orgID?: string;
    /** A user ID.
  Only returns organizations where this user is a member or owner.
   */
    userID?: string;
}
interface PostOrgsRequest {
    /** The organization to create. */
    body: PostOrganizationRequest;
}
interface GetOrgsIDRequest {
    /** The ID of the organization to retrieve.
     */
    orgID: string;
}
interface PatchOrgsIDRequest {
    /** The ID of the organization to update.
     */
    orgID: string;
    /** The organization update to apply. */
    body: PatchOrganizationRequest;
}
interface DeleteOrgsIDRequest {
    /** The ID of the organization to delete.
     */
    orgID: string;
}
interface GetOrgsIDSecretsRequest {
    /** The organization ID. */
    orgID: string;
}
interface PatchOrgsIDSecretsRequest {
    /** The organization ID. */
    orgID: string;
    /** Secret key value pairs to update/add */
    body: Secrets;
}
interface GetOrgsIDMembersRequest {
    /** The ID of the organization to retrieve users for.
     */
    orgID: string;
}
interface PostOrgsIDMembersRequest {
    /** The ID of the organization.
     */
    orgID: string;
    /** The user to add to the organization.
     */
    body: AddResourceMemberRequestBody;
}
interface DeleteOrgsIDMembersIDRequest {
    /** The ID of the user to remove. */
    userID: string;
    /** The ID of the organization to remove a user from. */
    orgID: string;
}
interface GetOrgsIDOwnersRequest {
    /** The ID of the organization to list owners for.
     */
    orgID: string;
}
interface PostOrgsIDOwnersRequest {
    /** The ID of the organization that you want to add an owner for. */
    orgID: string;
    /** The user to add as an owner of the organization. */
    body: AddResourceMemberRequestBody;
}
interface DeleteOrgsIDOwnersIDRequest {
    /** The ID of the user to remove. */
    userID: string;
    /** The ID of the organization to remove an owner from.
     */
    orgID: string;
}
interface PostOrgsIDSecretsRequest {
    /** The organization ID. */
    orgID: string;
    /** Secret key to delete */
    body: SecretKeys;
}
interface DeleteOrgsIDSecretsIDRequest {
    /** The organization ID. */
    orgID: string;
    /** The secret ID. */
    secretID: string;
}
/**
 * Orgs API
 */
declare class OrgsAPI {
    private base;
    /**
     * Creates OrgsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List organizations.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetOrgs }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getOrgs(request?: GetOrgsRequest, requestOptions?: RequestOptions): Promise<Organizations>;
    /**
     * Create an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostOrgs }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postOrgs(request: PostOrgsRequest, requestOptions?: RequestOptions): Promise<Organization>;
    /**
     * Retrieve an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetOrgsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getOrgsID(request: GetOrgsIDRequest, requestOptions?: RequestOptions): Promise<Organization>;
    /**
     * Update an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchOrgsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchOrgsID(request: PatchOrgsIDRequest, requestOptions?: RequestOptions): Promise<Organization>;
    /**
     * Delete an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteOrgsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteOrgsID(request: DeleteOrgsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all secret keys for an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetOrgsIDSecrets }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getOrgsIDSecrets(request: GetOrgsIDSecretsRequest, requestOptions?: RequestOptions): Promise<SecretKeysResponse>;
    /**
     * Update secrets in an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchOrgsIDSecrets }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchOrgsIDSecrets(request: PatchOrgsIDSecretsRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all members of an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetOrgsIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getOrgsIDMembers(request: GetOrgsIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMembers>;
    /**
     * Add a member to an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostOrgsIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postOrgsIDMembers(request: PostOrgsIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMember>;
    /**
     * Remove a member from an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteOrgsIDMembersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteOrgsIDMembersID(request: DeleteOrgsIDMembersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all owners of an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetOrgsIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getOrgsIDOwners(request: GetOrgsIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwners>;
    /**
     * Add an owner to an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostOrgsIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postOrgsIDOwners(request: PostOrgsIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwner>;
    /**
     * Remove an owner from an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteOrgsIDOwnersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteOrgsIDOwnersID(request: DeleteOrgsIDOwnersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Delete secrets from an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostOrgsIDSecrets }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postOrgsIDSecrets(request: PostOrgsIDSecretsRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Delete a secret from an organization.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteOrgsIDSecretsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteOrgsIDSecretsID(request: DeleteOrgsIDSecretsIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetPingRequest {
}
/**
 * Ping API
 */
declare class PingAPI {
    private base;
    /**
     * Creates PingAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Get the status and version of the instance.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetPing }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getPing(request?: GetPingRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface PostQueryAstRequest {
    /** The Flux query to analyze. */
    body: LanguageRequest;
}
interface GetQuerySuggestionsRequest {
}
interface GetQuerySuggestionsNameRequest {
    /** A Flux Function name.
  Only returns functions with this name.
   */
    name: string;
}
interface PostQueryAnalyzeRequest {
    /** Flux query to analyze */
    body: Query;
}
interface PostQueryRequest {
    /** Flux query or specification to execute */
    body: Query;
    /** The name or ID of the organization executing the query.
  
  #### InfluxDB Cloud
  
  - Doesn't use `org` or `orgID`.
  - Queries the bucket in the organization associated with the authorization (API token).
  
  #### InfluxDB OSS
  
  - Requires either `org` or `orgID`.
   */
    org?: string;
    /** The ID of the organization executing the query.
  
  #### InfluxDB Cloud
  
  - Doesn't use `org` or `orgID`.
  - Queries the bucket in the organization associated with the authorization (API token).
  
  #### InfluxDB OSS
  
  - Requires either `org` or `orgID`.
   */
    orgID?: string;
}
/**
 * Query API
 */
declare class QueryAPI {
    private base;
    /**
     * Creates QueryAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Generate a query Abstract Syntax Tree (AST).
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostQueryAst }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postQueryAst(request: PostQueryAstRequest, requestOptions?: RequestOptions): Promise<ASTResponse>;
    /**
     * Retrieve Flux query suggestions.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetQuerySuggestions }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getQuerySuggestions(request?: GetQuerySuggestionsRequest, requestOptions?: RequestOptions): Promise<FluxSuggestions>;
    /**
     * Retrieve a query suggestion for a branching suggestion.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetQuerySuggestionsName }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getQuerySuggestionsName(request: GetQuerySuggestionsNameRequest, requestOptions?: RequestOptions): Promise<FluxSuggestion>;
    /**
     * Analyze a Flux query.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostQueryAnalyze }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postQueryAnalyze(request: PostQueryAnalyzeRequest, requestOptions?: RequestOptions): Promise<AnalyzeQueryResponse>;
    /**
     * Query data.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostQuery }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postQuery(request: PostQueryRequest, requestOptions?: RequestOptions): Promise<string>;
}

interface GetReadyRequest {
}
/**
 * Ready API
 */
declare class ReadyAPI {
    private base;
    /**
     * Creates ReadyAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Get the readiness of an instance at startup.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetReady }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getReady(request?: GetReadyRequest, requestOptions?: RequestOptions): Promise<Ready>;
}

interface GetRemoteConnectionsRequest {
    /** The organization ID. */
    orgID: string;
    name?: string;
    remoteURL?: string;
}
interface PostRemoteConnectionRequest {
    /** entity body */
    body: RemoteConnectionCreationRequest;
}
interface GetRemoteConnectionByIDRequest {
    remoteID: string;
}
interface PatchRemoteConnectionByIDRequest {
    remoteID: string;
    /** entity body */
    body: RemoteConnectionUpdateRequest;
}
interface DeleteRemoteConnectionByIDRequest {
    remoteID: string;
}
/**
 * Remotes API
 */
declare class RemotesAPI {
    private base;
    /**
     * Creates RemotesAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all remote connections.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetRemoteConnections }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getRemoteConnections(request: GetRemoteConnectionsRequest, requestOptions?: RequestOptions): Promise<RemoteConnections>;
    /**
     * Register a new remote connection.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostRemoteConnection }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postRemoteConnection(request: PostRemoteConnectionRequest, requestOptions?: RequestOptions): Promise<RemoteConnection>;
    /**
     * Retrieve a remote connection.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetRemoteConnectionByID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getRemoteConnectionByID(request: GetRemoteConnectionByIDRequest, requestOptions?: RequestOptions): Promise<RemoteConnection>;
    /**
     * Update a remote connection.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchRemoteConnectionByID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchRemoteConnectionByID(request: PatchRemoteConnectionByIDRequest, requestOptions?: RequestOptions): Promise<RemoteConnection>;
    /**
     * Delete a remote connection.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteRemoteConnectionByID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteRemoteConnectionByID(request: DeleteRemoteConnectionByIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetReplicationsRequest {
    /** The organization ID. */
    orgID: string;
    name?: string;
    remoteID?: string;
    localBucketID?: string;
}
interface PostReplicationRequest {
    /** entity body */
    body: ReplicationCreationRequest;
    /** If true, validate the replication, but don't save it. */
    validate?: any;
}
interface GetReplicationByIDRequest {
    replicationID: string;
}
interface PatchReplicationByIDRequest {
    replicationID: string;
    /** entity body */
    body: ReplicationUpdateRequest;
    /** If true, validate the updated information, but don't save it. */
    validate?: any;
}
interface DeleteReplicationByIDRequest {
    replicationID: string;
}
interface PostValidateReplicationByIDRequest {
    replicationID: string;
}
/**
 * Replications API
 */
declare class ReplicationsAPI {
    private base;
    /**
     * Creates ReplicationsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all replications.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetReplications }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getReplications(request: GetReplicationsRequest, requestOptions?: RequestOptions): Promise<Replications>;
    /**
     * Register a new replication.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostReplication }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postReplication(request: PostReplicationRequest, requestOptions?: RequestOptions): Promise<Replication>;
    /**
     * Retrieve a replication.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetReplicationByID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getReplicationByID(request: GetReplicationByIDRequest, requestOptions?: RequestOptions): Promise<Replication>;
    /**
     * Update a replication.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchReplicationByID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchReplicationByID(request: PatchReplicationByIDRequest, requestOptions?: RequestOptions): Promise<Replication>;
    /**
     * Delete a replication.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteReplicationByID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteReplicationByID(request: DeleteReplicationByIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Validate a replication.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostValidateReplicationByID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postValidateReplicationByID(request: PostValidateReplicationByIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetResourcesRequest {
}
/**
 * Resources API
 */
declare class ResourcesAPI {
    private base;
    /**
     * Creates ResourcesAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all known resources.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetResources }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getResources(request?: GetResourcesRequest, requestOptions?: RequestOptions): Promise<string[]>;
}

interface PostRestoreKVRequest {
    /** Full KV snapshot. */
    body: string;
}
interface PostRestoreSQLRequest {
    /** Full SQL snapshot. */
    body: string;
}
interface PostRestoreBucketIDRequest {
    /** The bucket ID. */
    bucketID: string;
    /** Database info serialized as protobuf. */
    body: string;
}
interface PostRestoreBucketMetadataRequest {
    /** Metadata manifest for a bucket. */
    body: BucketMetadataManifest;
}
interface PostRestoreShardIdRequest {
    /** The shard ID. */
    shardID: string;
    /** TSM snapshot. */
    body: string;
}
/**
 * Restore API
 */
declare class RestoreAPI {
    private base;
    /**
     * Creates RestoreAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Overwrite the embedded KV store on the server with a backed-up snapshot.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostRestoreKV }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postRestoreKV(request: PostRestoreKVRequest, requestOptions?: RequestOptions): Promise<{
        /** token is the root token for the instance after restore (this is overwritten during the restore) */
        token?: string;
    }>;
    /**
     * Overwrite the embedded SQL store on the server with a backed-up snapshot.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostRestoreSQL }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postRestoreSQL(request: PostRestoreSQLRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Overwrite storage metadata for a bucket with shard info from a backup.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostRestoreBucketID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postRestoreBucketID(request: PostRestoreBucketIDRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Create a new bucket pre-seeded with shard info from a backup.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostRestoreBucketMetadata }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postRestoreBucketMetadata(request: PostRestoreBucketMetadataRequest, requestOptions?: RequestOptions): Promise<RestoredBucketMappings>;
    /**
     * Restore a TSM snapshot into a shard.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostRestoreShardId }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postRestoreShardId(request: PostRestoreShardIdRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetScrapersRequest {
    /** Specifies the name of the scraper target. */
    name?: string;
    /** List of scraper target IDs to return. If both `id` and `owner` are specified, only `id` is used. */
    id?: any;
    /** Specifies the organization ID of the scraper target. */
    orgID?: string;
    /** Specifies the organization name of the scraper target. */
    org?: string;
}
interface PostScrapersRequest {
    /** Scraper target to create */
    body: ScraperTargetRequest;
}
interface GetScrapersIDRequest {
    /** The identifier of the scraper target. */
    scraperTargetID: string;
}
interface PatchScrapersIDRequest {
    /** The identifier of the scraper target. */
    scraperTargetID: string;
    /** Scraper target update to apply */
    body: ScraperTargetRequest;
}
interface DeleteScrapersIDRequest {
    /** The identifier of the scraper target. */
    scraperTargetID: string;
}
interface GetScrapersIDLabelsRequest {
    /** The scraper target ID. */
    scraperTargetID: string;
}
interface PostScrapersIDLabelsRequest {
    /** The scraper target ID. */
    scraperTargetID: string;
    /** Label to add */
    body: LabelMapping;
}
interface DeleteScrapersIDLabelsIDRequest {
    /** The scraper target ID. */
    scraperTargetID: string;
    /** The label ID. */
    labelID: string;
}
interface GetScrapersIDMembersRequest {
    /** The scraper target ID. */
    scraperTargetID: string;
}
interface PostScrapersIDMembersRequest {
    /** The scraper target ID. */
    scraperTargetID: string;
    /** User to add as member */
    body: AddResourceMemberRequestBody;
}
interface DeleteScrapersIDMembersIDRequest {
    /** The ID of member to remove. */
    userID: string;
    /** The scraper target ID. */
    scraperTargetID: string;
}
interface GetScrapersIDOwnersRequest {
    /** The scraper target ID. */
    scraperTargetID: string;
}
interface PostScrapersIDOwnersRequest {
    /** The scraper target ID. */
    scraperTargetID: string;
    /** User to add as owner */
    body: AddResourceMemberRequestBody;
}
interface DeleteScrapersIDOwnersIDRequest {
    /** The ID of owner to remove. */
    userID: string;
    /** The scraper target ID. */
    scraperTargetID: string;
}
/**
 * Scrapers API
 */
declare class ScrapersAPI {
    private base;
    /**
     * Creates ScrapersAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all scraper targets.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetScrapers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getScrapers(request?: GetScrapersRequest, requestOptions?: RequestOptions): Promise<ScraperTargetResponses>;
    /**
     * Create a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostScrapers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postScrapers(request: PostScrapersRequest, requestOptions?: RequestOptions): Promise<ScraperTargetResponse>;
    /**
     * Retrieve a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetScrapersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getScrapersID(request: GetScrapersIDRequest, requestOptions?: RequestOptions): Promise<ScraperTargetResponse>;
    /**
     * Update a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchScrapersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchScrapersID(request: PatchScrapersIDRequest, requestOptions?: RequestOptions): Promise<ScraperTargetResponse>;
    /**
     * Delete a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteScrapersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteScrapersID(request: DeleteScrapersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all labels for a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetScrapersIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getScrapersIDLabels(request: GetScrapersIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostScrapersIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postScrapersIDLabels(request: PostScrapersIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete a label from a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteScrapersIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteScrapersIDLabelsID(request: DeleteScrapersIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all users with member privileges for a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetScrapersIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getScrapersIDMembers(request: GetScrapersIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMembers>;
    /**
     * Add a member to a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostScrapersIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postScrapersIDMembers(request: PostScrapersIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMember>;
    /**
     * Remove a member from a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteScrapersIDMembersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteScrapersIDMembersID(request: DeleteScrapersIDMembersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all owners of a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetScrapersIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getScrapersIDOwners(request: GetScrapersIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwners>;
    /**
     * Add an owner to a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostScrapersIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postScrapersIDOwners(request: PostScrapersIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwner>;
    /**
     * Remove an owner from a scraper target.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteScrapersIDOwnersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteScrapersIDOwnersID(request: DeleteScrapersIDOwnersIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetScriptsRequest {
    /** The maximum number of scripts to return. Default is `100`.
     */
    limit?: number;
    /** The offset for pagination.
  Specifies the number of records to skip in the result.
   */
    offset?: number;
    /** The name of the script. */
    name?: string;
    /** A list of label names.
  Only returns scripts that have all the specified labels.
  To retrieve a script, each name you pass in `labelNames` must exactly
  match the label for a script.
   */
    labelNames?: any;
    /** A part of the label name.
  Returns scripts that have a label that contains the specified phrase.
   */
    labelContains?: string;
}
interface PostScriptsRequest {
    /** The script to create. */
    body: ScriptCreateRequest;
}
interface GetScriptsIDRequest {
    /** A script ID.
  Specifies the script to retrieve.
   */
    scriptID: string;
}
interface PatchScriptsIDRequest {
    /** A script ID.
  Specifies the script to update.
   */
    scriptID: string;
    /** The script update to apply. */
    body: ScriptUpdateRequest;
}
interface DeleteScriptsIDRequest {
    /** A script ID.
  Specifies the script to delete.
   */
    scriptID: string;
}
interface PostScriptsIDInvokeRequest {
    /** A script ID.
  Specifies the script to execute.
   */
    scriptID: string;
    /** entity body */
    body: ScriptInvocationParams;
}
interface PatchScriptsIDAddLabelsRequest {
    /** The script ID.
  Specifies the script to add labels to.
   */
    scriptID: string;
    /** The labels to add to the script. */
    body: {
        /** A list of label names to add. */
        labels?: string[];
    };
}
interface PatchScriptsIDRemoveLabelsRequest {
    /** A script ID. Specifies the script to remove labels from. */
    scriptID: string;
    /** The labels to remove from the script. */
    body: {
        /** A list of label names to remove. */
        labels?: string[];
    };
}
/**
 * Scripts API
 */
declare class ScriptsAPI {
    private base;
    /**
     * Creates ScriptsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List scripts.
     * See {@link https://docs.influxdata.com/influxdb/cloud/api/#operation/GetScripts }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getScripts(request?: GetScriptsRequest, requestOptions?: RequestOptions): Promise<Scripts>;
    /**
     * Create a script.
     * See {@link https://docs.influxdata.com/influxdb/cloud/api/#operation/PostScripts }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postScripts(request: PostScriptsRequest, requestOptions?: RequestOptions): Promise<Script>;
    /**
     * Retrieve a script.
     * See {@link https://docs.influxdata.com/influxdb/cloud/api/#operation/GetScriptsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getScriptsID(request: GetScriptsIDRequest, requestOptions?: RequestOptions): Promise<Script>;
    /**
     * Update a script.
     * See {@link https://docs.influxdata.com/influxdb/cloud/api/#operation/PatchScriptsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchScriptsID(request: PatchScriptsIDRequest, requestOptions?: RequestOptions): Promise<Script>;
    /**
     * Delete a script.
     * See {@link https://docs.influxdata.com/influxdb/cloud/api/#operation/DeleteScriptsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteScriptsID(request: DeleteScriptsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Invoke a script.
     * See {@link https://docs.influxdata.com/influxdb/cloud/api/#operation/PostScriptsIDInvoke }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postScriptsIDInvoke(request: PostScriptsIDInvokeRequest, requestOptions?: RequestOptions): Promise<ScriptHTTPResponseData>;
    /**
     * Add labels to a script.
     * See {@link https://docs.influxdata.com/influxdb/cloud/api/#operation/PatchScriptsIDAddLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchScriptsIDAddLabels(request: PatchScriptsIDAddLabelsRequest, requestOptions?: RequestOptions): Promise<Script>;
    /**
     * Remove labels from a script.
     * See {@link https://docs.influxdata.com/influxdb/cloud/api/#operation/PatchScriptsIDRemoveLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchScriptsIDRemoveLabels(request: PatchScriptsIDRemoveLabelsRequest, requestOptions?: RequestOptions): Promise<Script>;
}

interface GetSetupRequest {
}
interface PostSetupRequest {
    /** Source to create */
    body: OnboardingRequest;
}
/**
 * Setup API
 */
declare class SetupAPI {
    private base;
    /**
     * Creates SetupAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Check if database has default user, org, bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetSetup }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getSetup(request?: GetSetupRequest, requestOptions?: RequestOptions): Promise<IsOnboarding>;
    /**
     * Set up initial user, org and bucket.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostSetup }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postSetup(request: PostSetupRequest, requestOptions?: RequestOptions): Promise<OnboardingResponse>;
}

interface PostSigninRequest {
    auth?: {
        user: string;
        password: string;
    };
}
/**
 * Signin API
 */
declare class SigninAPI {
    private base;
    /**
     * Creates SigninAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Create a user session.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostSignin }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postSignin(request: PostSigninRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface PostSignoutRequest {
}
/**
 * Signout API
 */
declare class SignoutAPI {
    private base;
    /**
     * Creates SignoutAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Expire the current UI session.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostSignout }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postSignout(request?: PostSignoutRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetSourcesRequest {
    /** The name of the organization. */
    org?: string;
}
interface PostSourcesRequest {
    /** Source to create */
    body: Source;
}
interface GetSourcesIDRequest {
    /** The source ID. */
    sourceID: string;
}
interface PatchSourcesIDRequest {
    /** The source ID. */
    sourceID: string;
    /** Source update */
    body: Source;
}
interface DeleteSourcesIDRequest {
    /** The source ID. */
    sourceID: string;
}
interface GetSourcesIDHealthRequest {
    /** The source ID. */
    sourceID: string;
}
interface GetSourcesIDBucketsRequest {
    /** The source ID. */
    sourceID: string;
    /** The name of the organization. */
    org?: string;
}
/**
 * Sources API
 */
declare class SourcesAPI {
    private base;
    /**
     * Creates SourcesAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all sources.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetSources }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getSources(request?: GetSourcesRequest, requestOptions?: RequestOptions): Promise<Sources>;
    /**
     * Create a source.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostSources }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postSources(request: PostSourcesRequest, requestOptions?: RequestOptions): Promise<Source>;
    /**
     * Retrieve a source.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetSourcesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getSourcesID(request: GetSourcesIDRequest, requestOptions?: RequestOptions): Promise<Source>;
    /**
     * Update a Source.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchSourcesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchSourcesID(request: PatchSourcesIDRequest, requestOptions?: RequestOptions): Promise<Source>;
    /**
     * Delete a source.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteSourcesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteSourcesID(request: DeleteSourcesIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Get the health of a source.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetSourcesIDHealth }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getSourcesIDHealth(request: GetSourcesIDHealthRequest, requestOptions?: RequestOptions): Promise<HealthCheck>;
    /**
     * Get buckets in a source.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetSourcesIDBuckets }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getSourcesIDBuckets(request: GetSourcesIDBucketsRequest, requestOptions?: RequestOptions): Promise<Buckets>;
}

interface ListStacksRequest {
    /** The ID of the organization that owns the stacks.
  Only returns stacks owned by this organization.
  
  #### InfluxDB Cloud
  
  - Doesn't require this parameter;
    InfluxDB only returns resources allowed by the API token.
   */
    orgID: string;
    /** The stack name.
  Finds stack `events` with this name and returns the stacks.
  
  Repeatable.
  To filter for more than one stack name,
  repeat this parameter with each name--for example:
  
  - `http://localhost:8086/api/v2/stacks?&orgID=INFLUX_ORG_ID&name=project-stack-0&name=project-stack-1`
   */
    name?: string;
    /** The stack ID.
  Only returns stacks with this ID.
  
  Repeatable.
  To filter for more than one stack ID,
  repeat this parameter with each ID--for example:
  
  - `http://localhost:8086/api/v2/stacks?&orgID=INFLUX_ORG_ID&stackID=09bd87cd33be3000&stackID=09bef35081fe3000`
   */
    stackID?: string;
}
interface CreateStackRequest {
    /** The stack to create. */
    body: {
        orgID?: string;
        name?: string;
        description?: string;
        urls?: string[];
    };
}
interface ReadStackRequest {
    /** The identifier of the stack. */
    stack_id: string;
}
interface UpdateStackRequest {
    /** The identifier of the stack. */
    stack_id: string;
    /** The stack to update. */
    body: {
        name?: string;
        description?: string;
        templateURLs?: string[];
        additionalResources?: Array<{
            resourceID: string;
            kind: string;
            templateMetaName?: string;
        }>;
    };
}
interface DeleteStackRequest {
    /** The identifier of the stack. */
    stack_id: string;
    /** The identifier of the organization. */
    orgID: string;
}
interface UninstallStackRequest {
    /** The identifier of the stack. */
    stack_id: string;
}
/**
 * Stacks API
 */
declare class StacksAPI {
    private base;
    /**
     * Creates StacksAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List installed stacks.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/ListStacks }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    listStacks(request: ListStacksRequest, requestOptions?: RequestOptions): Promise<{
        stacks?: Stack[];
    }>;
    /**
     * Create a stack.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/CreateStack }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    createStack(request: CreateStackRequest, requestOptions?: RequestOptions): Promise<Stack>;
    /**
     * Retrieve a stack.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/ReadStack }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    readStack(request: ReadStackRequest, requestOptions?: RequestOptions): Promise<Stack>;
    /**
     * Update a stack.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/UpdateStack }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    updateStack(request: UpdateStackRequest, requestOptions?: RequestOptions): Promise<Stack>;
    /**
     * Delete a stack and associated resources.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteStack }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteStack(request: DeleteStackRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Uninstall a stack.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/UninstallStack }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    uninstallStack(request: UninstallStackRequest, requestOptions?: RequestOptions): Promise<Stack>;
}

interface GetTasksIDRunsRequest {
    /** The ID of the task to get runs for.
  Only returns runs for this task.
   */
    taskID: string;
    /** A task run ID. Only returns runs created after this run. */
    after?: string;
    /** Limits the number of task runs returned. Default is `100`.
     */
    limit?: number;
    /** A timestamp ([RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)).
  Only returns runs scheduled after this time.
   */
    afterTime?: string;
    /** A timestamp ([RFC3339 date/time format](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#rfc3339-timestamp)).
  Only returns runs scheduled before this time.
   */
    beforeTime?: string;
}
interface PostTasksIDRunsRequest {
    taskID: string;
    /** entity body */
    body: RunManually;
}
interface GetTasksIDRunsIDRequest {
    /** The ID of the task to retrieve runs for. */
    taskID: string;
    /** The ID of the run to retrieve. */
    runID: string;
}
interface DeleteTasksIDRunsIDRequest {
    /** The ID of the task to cancel. */
    taskID: string;
    /** The ID of the task run to cancel. */
    runID: string;
}
interface PostTasksIDRunsIDRetryRequest {
    /** A [task](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#task)  ID.
  Specifies the task to retry.
   */
    taskID: string;
    /** A [task](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#task) run ID.
  Specifies the task run to retry.
  
  To find a task run ID, use the
  [`GET /api/v2/tasks/{taskID}/runs` endpoint](#operation/GetTasksIDRuns)
  to list task runs.
   */
    runID: string;
    /** entity body */
    body: any;
}
interface GetTasksIDLogsRequest {
    /** The task ID. */
    taskID: string;
}
interface GetTasksIDRunsIDLogsRequest {
    /** The ID of the task to get logs for. */
    taskID: string;
    /** The ID of the run to get logs for. */
    runID: string;
}
interface GetTasksIDLabelsRequest {
    /** The ID of the task to retrieve labels for. */
    taskID: string;
}
interface PostTasksIDLabelsRequest {
    /** The ID of the task to label. */
    taskID: string;
    /** An object that contains a _`labelID`_ to add to the task. */
    body: LabelMapping;
}
interface DeleteTasksIDLabelsIDRequest {
    /** The ID of the task to delete the label from. */
    taskID: string;
    /** The ID of the label to delete. */
    labelID: string;
}
interface GetTasksIDMembersRequest {
    /** The task ID. */
    taskID: string;
}
interface PostTasksIDMembersRequest {
    /** The task ID. */
    taskID: string;
    /** A user to add as a member of the task. */
    body: AddResourceMemberRequestBody;
}
interface DeleteTasksIDMembersIDRequest {
    /** The ID of the member to remove. */
    userID: string;
    /** The task ID. */
    taskID: string;
}
interface GetTasksIDOwnersRequest {
    /** The ID of the task to retrieve owners for. */
    taskID: string;
}
interface PostTasksIDOwnersRequest {
    /** The task ID. */
    taskID: string;
    /** A user to add as an owner of the task. */
    body: AddResourceMemberRequestBody;
}
interface DeleteTasksIDOwnersIDRequest {
    /** The ID of the owner to remove. */
    userID: string;
    /** The task ID. */
    taskID: string;
}
interface GetTasksRequest {
    /** A [task](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#task) name.
  Only returns tasks with the specified name.
  Different tasks may have the same name.
   */
    name?: string;
    /** A [task](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#task) ID.
  Only returns tasks created after the specified task.
   */
    after?: string;
    /** A [user](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#user) ID.
  Only returns tasks owned by the specified user.
   */
    user?: string;
    /** An [organization](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#organization) name.
  Only returns tasks owned by the specified organization.
   */
    org?: string;
    /** An [organization](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#organization) ID.
  Only returns tasks owned by the specified organization.
   */
    orgID?: string;
    /** A [task](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#task) status.
  Only returns tasks that have the specified status (`active` or `inactive`).
   */
    status?: string;
    /** The maximum number of [tasks](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#task) to return.
  Default is `100`.
  The minimum is `1` and the maximum is `500`.
  
  To reduce the payload size, combine _`type=basic`_ and _`limit`_ (see _Request samples_).
  For more information about the `basic` response, see the _`type`_ parameter.
   */
    limit?: number;
    /** A [task](https://docs.influxdata.com/influxdb/v2.3/reference/glossary/#task) type (`basic` or `system`).
  Default is `system`.
  Specifies the level of detail for tasks in the response.
  The default (`system`) response contains all the metadata properties for tasks.
  To reduce the response size, pass `basic` to omit some task properties (`flux`, `createdAt`, `updatedAt`).
   */
    type?: string;
}
interface PostTasksRequest {
    /** The task to create. */
    body: TaskCreateRequest;
}
interface GetTasksIDRequest {
    /** The ID of the task to retrieve. */
    taskID: string;
}
interface PatchTasksIDRequest {
    /** The ID of the task to update. */
    taskID: string;
    /** An object that contains updated task properties to apply. */
    body: TaskUpdateRequest;
}
interface DeleteTasksIDRequest {
    /** The ID of the task to delete. */
    taskID: string;
}
/**
 * Tasks API
 */
declare class TasksAPI {
    private base;
    /**
     * Creates TasksAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List runs for a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasksIDRuns }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasksIDRuns(request: GetTasksIDRunsRequest, requestOptions?: RequestOptions): Promise<Runs>;
    /**
     * Start a task run, overriding the schedule.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTasksIDRuns }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTasksIDRuns(request: PostTasksIDRunsRequest, requestOptions?: RequestOptions): Promise<Run>;
    /**
     * Retrieve a run for a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasksIDRunsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasksIDRunsID(request: GetTasksIDRunsIDRequest, requestOptions?: RequestOptions): Promise<Run>;
    /**
     * Cancel a running task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTasksIDRunsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTasksIDRunsID(request: DeleteTasksIDRunsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * Retry a task run.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTasksIDRunsIDRetry }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTasksIDRunsIDRetry(request: PostTasksIDRunsIDRetryRequest, requestOptions?: RequestOptions): Promise<Run>;
    /**
     * Retrieve all logs for a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasksIDLogs }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasksIDLogs(request: GetTasksIDLogsRequest, requestOptions?: RequestOptions): Promise<Logs>;
    /**
     * Retrieve all logs for a run.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasksIDRunsIDLogs }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasksIDRunsIDLogs(request: GetTasksIDRunsIDLogsRequest, requestOptions?: RequestOptions): Promise<Logs>;
    /**
     * List labels for a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasksIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasksIDLabels(request: GetTasksIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTasksIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTasksIDLabels(request: PostTasksIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete a label from a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTasksIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTasksIDLabelsID(request: DeleteTasksIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all task members.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasksIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasksIDMembers(request: GetTasksIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMembers>;
    /**
     * Add a member to a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTasksIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTasksIDMembers(request: PostTasksIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMember>;
    /**
     * Remove a member from a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTasksIDMembersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTasksIDMembersID(request: DeleteTasksIDMembersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all owners of a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasksIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasksIDOwners(request: GetTasksIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwners>;
    /**
     * Add an owner for a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTasksIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTasksIDOwners(request: PostTasksIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwner>;
    /**
     * Remove an owner from a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTasksIDOwnersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTasksIDOwnersID(request: DeleteTasksIDOwnersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List tasks.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasks }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasks(request?: GetTasksRequest, requestOptions?: RequestOptions): Promise<Tasks>;
    /**
     * Create a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTasks }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTasks(request: PostTasksRequest, requestOptions?: RequestOptions): Promise<Task>;
    /**
     * Retrieve a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTasksID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTasksID(request: GetTasksIDRequest, requestOptions?: RequestOptions): Promise<Task>;
    /**
     * Update a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchTasksID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchTasksID(request: PatchTasksIDRequest, requestOptions?: RequestOptions): Promise<Task>;
    /**
     * Delete a task.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTasksID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTasksID(request: DeleteTasksIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetTelegrafPluginsRequest {
    /** The type of plugin desired. */
    type?: string;
}
/**
 * Telegraf API
 */
declare class TelegrafAPI {
    private base;
    /**
     * Creates TelegrafAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all Telegraf plugins.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTelegrafPlugins }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTelegrafPlugins(request?: GetTelegrafPluginsRequest, requestOptions?: RequestOptions): Promise<TelegrafPlugins>;
}

interface GetTelegrafsRequest {
    /** The organization ID the Telegraf config belongs to. */
    orgID?: string;
}
interface PostTelegrafsRequest {
    /** Telegraf configuration to create */
    body: TelegrafPluginRequest;
}
interface GetTelegrafsIDRequest {
    /** The Telegraf configuration ID. */
    telegrafID: string;
}
interface PutTelegrafsIDRequest {
    /** The Telegraf config ID. */
    telegrafID: string;
    /** Telegraf configuration update to apply */
    body: TelegrafPluginRequest;
}
interface DeleteTelegrafsIDRequest {
    /** The Telegraf configuration ID. */
    telegrafID: string;
}
interface GetTelegrafsIDLabelsRequest {
    /** The Telegraf config ID. */
    telegrafID: string;
}
interface PostTelegrafsIDLabelsRequest {
    /** The Telegraf config ID. */
    telegrafID: string;
    /** Label to add */
    body: LabelMapping;
}
interface DeleteTelegrafsIDLabelsIDRequest {
    /** The Telegraf config ID. */
    telegrafID: string;
    /** The label ID. */
    labelID: string;
}
interface GetTelegrafsIDMembersRequest {
    /** The Telegraf config ID. */
    telegrafID: string;
}
interface PostTelegrafsIDMembersRequest {
    /** The Telegraf config ID. */
    telegrafID: string;
    /** User to add as member */
    body: AddResourceMemberRequestBody;
}
interface DeleteTelegrafsIDMembersIDRequest {
    /** The ID of the member to remove. */
    userID: string;
    /** The Telegraf config ID. */
    telegrafID: string;
}
interface GetTelegrafsIDOwnersRequest {
    /** The Telegraf configuration ID. */
    telegrafID: string;
}
interface PostTelegrafsIDOwnersRequest {
    /** The Telegraf configuration ID. */
    telegrafID: string;
    /** User to add as owner */
    body: AddResourceMemberRequestBody;
}
interface DeleteTelegrafsIDOwnersIDRequest {
    /** The ID of the owner to remove. */
    userID: string;
    /** The Telegraf config ID. */
    telegrafID: string;
}
/**
 * Telegrafs API
 */
declare class TelegrafsAPI {
    private base;
    /**
     * Creates TelegrafsAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all Telegraf configurations.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTelegrafs }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTelegrafs(request?: GetTelegrafsRequest, requestOptions?: RequestOptions): Promise<Telegrafs>;
    /**
     * Create a Telegraf configuration.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTelegrafs }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTelegrafs(request: PostTelegrafsRequest, requestOptions?: RequestOptions): Promise<Telegraf>;
    /**
     * Retrieve a Telegraf configuration.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTelegrafsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTelegrafsID(request: GetTelegrafsIDRequest, requestOptions?: RequestOptions): Promise<string>;
    /**
     * Update a Telegraf configuration.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PutTelegrafsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    putTelegrafsID(request: PutTelegrafsIDRequest, requestOptions?: RequestOptions): Promise<Telegraf>;
    /**
     * Delete a Telegraf configuration.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTelegrafsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTelegrafsID(request: DeleteTelegrafsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all labels for a Telegraf config.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTelegrafsIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTelegrafsIDLabels(request: GetTelegrafsIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a Telegraf config.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTelegrafsIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTelegrafsIDLabels(request: PostTelegrafsIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete a label from a Telegraf config.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTelegrafsIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTelegrafsIDLabelsID(request: DeleteTelegrafsIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all users with member privileges for a Telegraf config.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTelegrafsIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTelegrafsIDMembers(request: GetTelegrafsIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMembers>;
    /**
     * Add a member to a Telegraf config.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTelegrafsIDMembers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTelegrafsIDMembers(request: PostTelegrafsIDMembersRequest, requestOptions?: RequestOptions): Promise<ResourceMember>;
    /**
     * Remove a member from a Telegraf config.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTelegrafsIDMembersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTelegrafsIDMembersID(request: DeleteTelegrafsIDMembersIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all owners of a Telegraf configuration.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetTelegrafsIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getTelegrafsIDOwners(request: GetTelegrafsIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwners>;
    /**
     * Add an owner to a Telegraf configuration.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostTelegrafsIDOwners }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postTelegrafsIDOwners(request: PostTelegrafsIDOwnersRequest, requestOptions?: RequestOptions): Promise<ResourceOwner>;
    /**
     * Remove an owner from a Telegraf config.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteTelegrafsIDOwnersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteTelegrafsIDOwnersID(request: DeleteTelegrafsIDOwnersIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface ApplyTemplateRequest {
    /** Parameters for applying templates.
     */
    body: TemplateApply;
}
interface ExportTemplateRequest {
    /** Export resources as an InfluxDB template. */
    body: TemplateExportByID | TemplateExportByName;
}
/**
 * Templates API
 */
declare class TemplatesAPI {
    private base;
    /**
     * Creates TemplatesAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Apply or dry-run a template.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/ApplyTemplate }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    applyTemplate(request: ApplyTemplateRequest, requestOptions?: RequestOptions): Promise<TemplateSummary>;
    /**
     * Export a new template.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/ExportTemplate }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    exportTemplate(request: ExportTemplateRequest, requestOptions?: RequestOptions): Promise<Template>;
}

interface PostUsersIDPasswordRequest {
    /** The user ID. */
    userID: string;
    auth?: {
        user: string;
        password: string;
    };
    /** New password */
    body: PasswordResetBody;
}
interface GetUsersRequest {
    /** The offset for pagination.
  The number of records to skip.
   */
    offset?: number;
    /** Limits the number of records returned. Default is `20`.
     */
    limit?: number;
    /** Resource ID to seek from. Results are not inclusive of this ID. Use `after` instead of `offset`.
     */
    after?: string;
    name?: string;
    id?: string;
}
interface PostUsersRequest {
    /** The user to create. */
    body: User;
}
interface GetUsersIDRequest {
    /** The user ID. */
    userID: string;
}
interface PatchUsersIDRequest {
    /** The ID of the user to update. */
    userID: string;
    /** User update to apply */
    body: User;
}
interface DeleteUsersIDRequest {
    /** The ID of the user to delete. */
    userID: string;
}
/**
 * Users API
 */
declare class UsersAPI {
    private base;
    /**
     * Creates UsersAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Update a password.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostUsersIDPassword }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postUsersIDPassword(request: PostUsersIDPasswordRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List users.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetUsers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getUsers(request?: GetUsersRequest, requestOptions?: RequestOptions): Promise<Users>;
    /**
     * Create a user.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostUsers }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postUsers(request: PostUsersRequest, requestOptions?: RequestOptions): Promise<UserResponse>;
    /**
     * Retrieve a user.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetUsersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getUsersID(request: GetUsersIDRequest, requestOptions?: RequestOptions): Promise<UserResponse>;
    /**
     * Update a user.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchUsersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchUsersID(request: PatchUsersIDRequest, requestOptions?: RequestOptions): Promise<UserResponse>;
    /**
     * Delete a user.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteUsersID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteUsersID(request: DeleteUsersIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface GetVariablesIDLabelsRequest {
    /** The variable ID. */
    variableID: string;
}
interface PostVariablesIDLabelsRequest {
    /** The variable ID. */
    variableID: string;
    /** Label to add */
    body: LabelMapping;
}
interface DeleteVariablesIDLabelsIDRequest {
    /** The variable ID. */
    variableID: string;
    /** The label ID to delete. */
    labelID: string;
}
interface GetVariablesRequest {
    /** The name of the organization. */
    org?: string;
    /** The organization ID. */
    orgID?: string;
}
interface PostVariablesRequest {
    /** Variable to create */
    body: Variable;
}
interface GetVariablesIDRequest {
    /** The variable ID. */
    variableID: string;
}
interface PutVariablesIDRequest {
    /** The variable ID. */
    variableID: string;
    /** Variable to replace */
    body: Variable;
}
interface PatchVariablesIDRequest {
    /** The variable ID. */
    variableID: string;
    /** Variable update to apply */
    body: Variable;
}
interface DeleteVariablesIDRequest {
    /** The variable ID. */
    variableID: string;
}
/**
 * Variables API
 */
declare class VariablesAPI {
    private base;
    /**
     * Creates VariablesAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * List all labels for a variable.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetVariablesIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getVariablesIDLabels(request: GetVariablesIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelsResponse>;
    /**
     * Add a label to a variable.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostVariablesIDLabels }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postVariablesIDLabels(request: PostVariablesIDLabelsRequest, requestOptions?: RequestOptions): Promise<LabelResponse>;
    /**
     * Delete a label from a variable.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteVariablesIDLabelsID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteVariablesIDLabelsID(request: DeleteVariablesIDLabelsIDRequest, requestOptions?: RequestOptions): Promise<void>;
    /**
     * List all variables.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetVariables }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getVariables(request?: GetVariablesRequest, requestOptions?: RequestOptions): Promise<Variables>;
    /**
     * Create a variable.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostVariables }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postVariables(request: PostVariablesRequest, requestOptions?: RequestOptions): Promise<Variable>;
    /**
     * Retrieve a variable.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/GetVariablesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    getVariablesID(request: GetVariablesIDRequest, requestOptions?: RequestOptions): Promise<Variable>;
    /**
     * Replace a variable.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PutVariablesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    putVariablesID(request: PutVariablesIDRequest, requestOptions?: RequestOptions): Promise<Variable>;
    /**
     * Update a variable.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PatchVariablesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    patchVariablesID(request: PatchVariablesIDRequest, requestOptions?: RequestOptions): Promise<Variable>;
    /**
     * Delete a variable.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/DeleteVariablesID }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    deleteVariablesID(request: DeleteVariablesIDRequest, requestOptions?: RequestOptions): Promise<void>;
}

interface PostWriteRequest {
    /** Data in line protocol format.
  
  To send compressed data, do the following:
  
    1. Use [GZIP](https://www.gzip.org/) to compress the line protocol data.
    2. In your request, send the compressed data and the
       `Content-Encoding: gzip` header.
  
  #### Related guides
  
  - [Best practices for optimizing writes](https://docs.influxdata.com/influxdb/v2.3/write-data/best-practices/optimize-writes/).
   */
    body: string;
    /** The destination organization for writes.
  InfluxDB writes all points in the batch to this organization.
  If you pass both `orgID` and `org`, they must both be valid.
  
  #### InfluxDB Cloud
  
  - Doesn't require `org` or `orgID`.
  - Writes to the bucket in the organization associated with the authorization (API token).
  
  #### InfluxDB OSS
  
  - Requires either `org` or `orgID`.
  - InfluxDB writes all points in the batch to this organization.
   */
    org: string;
    /** The ID of the destination organization for writes.
  If you pass both `orgID` and `org`, they must both be valid.
  
  #### InfluxDB Cloud
  
  - Doesn't require `org` or `orgID`.
  - Writes to the bucket in the organization associated with the authorization (API token).
  
  
  #### InfluxDB OSS
  
  - Requires either `org` or `orgID`.
  - InfluxDB writes all points in the batch to this organization.
   */
    orgID?: string;
    /** The destination bucket for writes.
  InfluxDB writes all points in the batch to this bucket.
   */
    bucket: string;
    /** The precision for unix timestamps in the line protocol batch. */
    precision?: any;
}
/**
 * Write API
 */
declare class WriteAPI {
    private base;
    /**
     * Creates WriteAPI
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     */
    constructor(influxDB: InfluxDB);
    /**
     * Write data.
     * See {@link https://docs.influxdata.com/influxdb/v2.4/api/#operation/PostWrite }
     * @param request - request parameters and body (if supported)
     * @param requestOptions - optional transport options
     * @returns promise of response
     */
    postWrite(request: PostWriteRequest, requestOptions?: RequestOptions): Promise<void>;
}

/** ExecutionOptions contains execution options for a flux script. */
interface ExecutionOptions {
    /**
     * Requests gzip encoded response.
     */
    gzip?: boolean;
    /**
     * HTTP headers that will be sent with every query request.
     */
    headers?: {
        [key: string]: string;
    };
}
/**
 * FluxScriptInvocationAPI executes flux 'API invokable script' and parses the result CSV annotated data.
 * See https://docs.influxdata.com/influxdb/cloud/api-guide/api-invokable-scripts/ .
 */
declare class FluxScriptInvocationAPI {
    private transport;
    private processCSVResponse;
    private options;
    /**
     * Creates FluxScriptInvocationAPI with the supplied InfluxDB instance and a particular script identifier.
     * @param influxDB - an instance that knows how to communicate with InfluxDB server
     * @param options - script execution options
     */
    constructor(influxDB: InfluxDB, options?: ExecutionOptions);
    /**
     * Invoke returns a parsed response data stream that executes
     * the supplied script when asked for data.
     * @param scriptID - script identifier
     * @param params  - script parameters
     * @returns response with various methods to process data from the returned annotated
     * CSV response data stream
     */
    invoke(scriptID: string, params?: Record<string, any>): AnnotatedCSVResponse;
}

export { type ASTResponse, type AddResourceMemberRequestBody, type AnalyzeQueryResponse, type ApplyTemplateRequest, type ArrayExpression, type Authorization, type AuthorizationPostRequest, type AuthorizationUpdateRequest, type Authorizations, AuthorizationsAPI, type Axes, type Axis, type AxisScale, BackupAPI, type BadStatement, type BandViewProperties, type BinaryExpression, type Block, type BooleanLiteral, type Bucket, type BucketMetadataManifest, type BucketMetadataManifests, type BucketShardMapping, type BucketShardMappings, type Buckets, BucketsAPI, type BuilderAggregateFunctionType, type BuilderConfig, type BuilderFunctionsType, type BuilderTagsType, type BuiltinStatement, type CallExpression, type Cell, type CellUpdate, type CellWithViewProperties, type Cells, type CellsWithViewProperties, type Check, type CheckBase, type CheckDiscriminator, type CheckPatch, type CheckStatusLevel, type CheckViewProperties, type Checks, ChecksAPI, type ColorMapping, type ConditionalExpression, type Config, ConfigAPI, type ConstantVariableProperties, type CreateCell, type CreateCheckRequest, type CreateDashboardRequest, type CreateNotificationEndpointRequest, type CreateNotificationRuleRequest, type CreateStackRequest, type CustomCheck, type DBRP, type DBRPCreate, type DBRPGet, type DBRPUpdate, type DBRPs, type Dashboard, type DashboardColor, type DashboardQuery, type DashboardWithViewProperties, type Dashboards, DashboardsAPI, type DateTimeLiteral, DbrpsAPI, type DeadmanCheck, DebugAPI, type DecimalPlaces, DeleteAPI, type DeleteAuthorizationsIDRequest, type DeleteBucketsIDLabelsIDRequest, type DeleteBucketsIDMembersIDRequest, type DeleteBucketsIDOwnersIDRequest, type DeleteBucketsIDRequest, type DeleteChecksIDLabelsIDRequest, type DeleteChecksIDRequest, type DeleteDBRPIDRequest, type DeleteDashboardsIDCellsIDRequest, type DeleteDashboardsIDLabelsIDRequest, type DeleteDashboardsIDMembersIDRequest, type DeleteDashboardsIDOwnersIDRequest, type DeleteDashboardsIDRequest, type DeleteLabelsIDRequest, type DeleteNotificationEndpointsIDLabelsIDRequest, type DeleteNotificationEndpointsIDRequest, type DeleteNotificationRulesIDLabelsIDRequest, type DeleteNotificationRulesIDRequest, type DeleteOrgsIDMembersIDRequest, type DeleteOrgsIDOwnersIDRequest, type DeleteOrgsIDRequest, type DeleteOrgsIDSecretsIDRequest, type DeletePredicateRequest, type DeleteRemoteConnectionByIDRequest, type DeleteReplicationByIDRequest, type DeleteScrapersIDLabelsIDRequest, type DeleteScrapersIDMembersIDRequest, type DeleteScrapersIDOwnersIDRequest, type DeleteScrapersIDRequest, type DeleteScriptsIDRequest, type DeleteSourcesIDRequest, type DeleteStackRequest, type DeleteTasksIDLabelsIDRequest, type DeleteTasksIDMembersIDRequest, type DeleteTasksIDOwnersIDRequest, type DeleteTasksIDRequest, type DeleteTasksIDRunsIDRequest, type DeleteTelegrafsIDLabelsIDRequest, type DeleteTelegrafsIDMembersIDRequest, type DeleteTelegrafsIDOwnersIDRequest, type DeleteTelegrafsIDRequest, type DeleteUsersIDRequest, type DeleteVariablesIDLabelsIDRequest, type DeleteVariablesIDRequest, type Dialect, type DictExpression, type DictItem, type Duration, type DurationLiteral, type Error, type ExecutionOptions, type ExportTemplateRequest, type Expression, type ExpressionStatement, type File, type Flags, FlagsAPI, type FloatLiteral, type FluxResponse, FluxScriptInvocationAPI, type FluxSuggestion, type FluxSuggestions, type FunctionExpression, type GaugeViewProperties, type GeoCircleViewLayer, type GeoHeatMapViewLayer, type GeoPointMapViewLayer, type GeoTrackMapViewLayer, type GeoViewLayer, type GeoViewLayerProperties, type GeoViewProperties, type GetAuthorizationsIDRequest, type GetAuthorizationsRequest, type GetBackupKVRequest, type GetBackupMetadataRequest, type GetBackupShardIdRequest, type GetBucketsIDLabelsRequest, type GetBucketsIDMembersRequest, type GetBucketsIDOwnersRequest, type GetBucketsIDRequest, type GetBucketsRequest, type GetChecksIDLabelsRequest, type GetChecksIDQueryRequest, type GetChecksIDRequest, type GetChecksRequest, type GetConfigRequest, type GetDBRPsIDRequest, type GetDBRPsRequest, type GetDashboardsIDCellsIDViewRequest, type GetDashboardsIDLabelsRequest, type GetDashboardsIDMembersRequest, type GetDashboardsIDOwnersRequest, type GetDashboardsIDRequest, type GetDashboardsRequest, type GetDebugPprofAllProfilesRequest, type GetDebugPprofAllocsRequest, type GetDebugPprofBlockRequest, type GetDebugPprofCmdlineRequest, type GetDebugPprofGoroutineRequest, type GetDebugPprofHeapRequest, type GetDebugPprofMutexRequest, type GetDebugPprofProfileRequest, type GetDebugPprofThreadCreateRequest, type GetDebugPprofTraceRequest, type GetFlagsRequest, type GetHealthRequest, type GetLabelsIDRequest, type GetLabelsRequest, type GetMeRequest, type GetMetricsRequest, type GetNotificationEndpointsIDLabelsRequest, type GetNotificationEndpointsIDRequest, type GetNotificationEndpointsRequest, type GetNotificationRulesIDLabelsRequest, type GetNotificationRulesIDQueryRequest, type GetNotificationRulesIDRequest, type GetNotificationRulesRequest, type GetOrgsIDMembersRequest, type GetOrgsIDOwnersRequest, type GetOrgsIDRequest, type GetOrgsIDSecretsRequest, type GetOrgsRequest, type GetPingRequest, type GetQuerySuggestionsNameRequest, type GetQuerySuggestionsRequest, type GetReadyRequest, type GetRemoteConnectionByIDRequest, type GetRemoteConnectionsRequest, type GetReplicationByIDRequest, type GetReplicationsRequest, type GetResourcesRequest, type GetRoutesRequest, type GetScrapersIDLabelsRequest, type GetScrapersIDMembersRequest, type GetScrapersIDOwnersRequest, type GetScrapersIDRequest, type GetScrapersRequest, type GetScriptsIDRequest, type GetScriptsRequest, type GetSetupRequest, type GetSourcesIDBucketsRequest, type GetSourcesIDHealthRequest, type GetSourcesIDRequest, type GetSourcesRequest, type GetTasksIDLabelsRequest, type GetTasksIDLogsRequest, type GetTasksIDMembersRequest, type GetTasksIDOwnersRequest, type GetTasksIDRequest, type GetTasksIDRunsIDLogsRequest, type GetTasksIDRunsIDRequest, type GetTasksIDRunsRequest, type GetTasksRequest, type GetTelegrafPluginsRequest, type GetTelegrafsIDLabelsRequest, type GetTelegrafsIDMembersRequest, type GetTelegrafsIDOwnersRequest, type GetTelegrafsIDRequest, type GetTelegrafsRequest, type GetUsersIDRequest, type GetUsersRequest, type GetVariablesIDLabelsRequest, type GetVariablesIDRequest, type GetVariablesRequest, type GreaterThreshold, type HTTPNotificationEndpoint, type HTTPNotificationRule, type HTTPNotificationRuleBase, HealthAPI, type HealthCheck, type HeatmapViewProperties, type HistogramViewProperties, type Identifier, type ImportDeclaration, type IndexExpression, type IntegerLiteral, type IsOnboarding, type Label, type LabelCreateRequest, type LabelMapping, type LabelResponse, type LabelUpdate, type Labels, LabelsAPI, type LabelsResponse, type LanguageRequest, type LatLonColumn, type LatLonColumns, type LesserThreshold, type LinePlusSingleStatProperties, type LineProtocolError, type LineProtocolLengthError, type Link, type Links, type ListStacksRequest, type LogEvent, type LogicalExpression, type Logs, type MapVariableProperties, type MarkdownViewProperties, MeAPI, type MemberAssignment, type MemberExpression, type MetadataBackup, MetricsAPI, type MosaicViewProperties, type Node, type NodeType, type NotificationEndpoint, type NotificationEndpointBase, type NotificationEndpointDiscriminator, type NotificationEndpointType, type NotificationEndpointUpdate, type NotificationEndpoints, NotificationEndpointsAPI, type NotificationRule, type NotificationRuleBase, type NotificationRuleDiscriminator, type NotificationRuleUpdate, type NotificationRules, NotificationRulesAPI, type ObjectExpression, type OnboardingRequest, type OnboardingResponse, type OptionStatement, type Organization, type Organizations, OrgsAPI, type Package, type PackageClause, type PagerDutyNotificationEndpoint, type PagerDutyNotificationRule, type PagerDutyNotificationRuleBase, type ParenExpression, type PasswordResetBody, type PatchAuthorizationsIDRequest, type PatchBucketRequest, type PatchBucketsIDRequest, type PatchChecksIDRequest, type PatchDBRPIDRequest, type PatchDashboardsIDCellsIDRequest, type PatchDashboardsIDCellsIDViewRequest, type PatchDashboardsIDRequest, type PatchLabelsIDRequest, type PatchNotificationEndpointsIDRequest, type PatchNotificationRulesIDRequest, type PatchOrganizationRequest, type PatchOrgsIDRequest, type PatchOrgsIDSecretsRequest, type PatchRemoteConnectionByIDRequest, type PatchReplicationByIDRequest, type PatchRetentionRule, type PatchRetentionRules, type PatchScrapersIDRequest, type PatchScriptsIDAddLabelsRequest, type PatchScriptsIDRemoveLabelsRequest, type PatchScriptsIDRequest, type PatchSourcesIDRequest, type PatchTasksIDRequest, type PatchUsersIDRequest, type PatchVariablesIDRequest, type Permission, PingAPI, type PipeExpression, type PipeLiteral, type PostAuthorizationsRequest, type PostBucketRequest, type PostBucketsIDLabelsRequest, type PostBucketsIDMembersRequest, type PostBucketsIDOwnersRequest, type PostBucketsRequest, type PostCheck, type PostChecksIDLabelsRequest, type PostDBRPRequest, type PostDashboardsIDCellsRequest, type PostDashboardsIDLabelsRequest, type PostDashboardsIDMembersRequest, type PostDashboardsIDOwnersRequest, type PostDashboardsRequest, type PostDeleteRequest, type PostLabelsRequest, type PostNotificationEndpoint, type PostNotificationEndpointIDLabelsRequest, type PostNotificationRule, type PostNotificationRuleIDLabelsRequest, type PostOrganizationRequest, type PostOrgsIDMembersRequest, type PostOrgsIDOwnersRequest, type PostOrgsIDSecretsRequest, type PostOrgsRequest, type PostQueryAnalyzeRequest, type PostQueryAstRequest, type PostQueryRequest, type PostRemoteConnectionRequest, type PostReplicationRequest, type PostRestoreBucketIDRequest, type PostRestoreBucketMetadataRequest, type PostRestoreKVRequest, type PostRestoreSQLRequest, type PostRestoreShardIdRequest, type PostScrapersIDLabelsRequest, type PostScrapersIDMembersRequest, type PostScrapersIDOwnersRequest, type PostScrapersRequest, type PostScriptsIDInvokeRequest, type PostScriptsRequest, type PostSetupRequest, type PostSigninRequest, type PostSignoutRequest, type PostSourcesRequest, type PostTasksIDLabelsRequest, type PostTasksIDMembersRequest, type PostTasksIDOwnersRequest, type PostTasksIDRunsIDRetryRequest, type PostTasksIDRunsRequest, type PostTasksRequest, type PostTelegrafsIDLabelsRequest, type PostTelegrafsIDMembersRequest, type PostTelegrafsIDOwnersRequest, type PostTelegrafsRequest, type PostUsersIDPasswordRequest, type PostUsersRequest, type PostValidateReplicationByIDRequest, type PostVariablesIDLabelsRequest, type PostVariablesRequest, type PostWriteRequest, type Property, type PropertyKey, type PutChecksIDRequest, type PutDashboardsIDCellsRequest, type PutMePasswordRequest, type PutNotificationEndpointsIDRequest, type PutNotificationRulesIDRequest, type PutTelegrafsIDRequest, type PutVariablesIDRequest, type Query, QueryAPI, type QueryEditMode, type QueryVariableProperties, type RangeThreshold, type ReadStackRequest, type Ready, ReadyAPI, type RegexpLiteral, type RemoteConnection, type RemoteConnectionCreationRequest, type RemoteConnectionUpdateRequest, type RemoteConnections, RemotesAPI, type RenamableField, type Replication, type ReplicationCreationRequest, type ReplicationUpdateRequest, type Replications, ReplicationsAPI, type RequestOptions, type Resource, type ResourceMember, type ResourceMembers, type ResourceOwner, type ResourceOwners, ResourcesAPI, RestoreAPI, type RestoredBucketMappings, type RetentionPolicyManifest, type RetentionPolicyManifests, type RetentionRule, type RetentionRules, type ReturnStatement, RootAPI, type Routes, type RuleStatusLevel, type Run, type RunManually, type Runs, type SMTPNotificationRule, type SMTPNotificationRuleBase, type ScatterViewProperties, type SchemaType, type ScraperTargetRequest, type ScraperTargetResponse, type ScraperTargetResponses, ScrapersAPI, type Script, type ScriptCreateRequest, type ScriptHTTPResponseData, type ScriptInvocationParams, type ScriptLanguage, type ScriptUpdateRequest, type Scripts, ScriptsAPI, type SecretKeys, type SecretKeysResponse, type Secrets, SetupAPI, type ShardGroupManifest, type ShardGroupManifests, type ShardManifest, type ShardManifests, type ShardOwner, type ShardOwners, SigninAPI, SignoutAPI, type SimpleTableViewProperties, type SingleStatViewProperties, type SlackNotificationEndpoint, type SlackNotificationRule, type SlackNotificationRuleBase, type Source, type Sources, SourcesAPI, type Stack, StacksAPI, type Statement, type StaticLegend, type StatusRule, type StringLiteral, type SubscriptionManifest, type SubscriptionManifests, type TableViewProperties, type TagRule, type Task, type TaskCreateRequest, type TaskStatusType, type TaskUpdateRequest, type Tasks, TasksAPI, type Telegraf, TelegrafAPI, type TelegrafPlugin, type TelegrafPluginRequest, type TelegrafPlugins, type TelegrafRequest, type Telegrafs, TelegrafsAPI, type TelegramNotificationEndpoint, type TelegramNotificationRule, type TelegramNotificationRuleBase, type Template, type TemplateApply, type TemplateChart, type TemplateEnvReferences, type TemplateExportByID, type TemplateExportByName, type TemplateKind, type TemplateSummary, type TemplateSummaryLabel, TemplatesAPI, type TestStatement, type Threshold, type ThresholdBase, type ThresholdCheck, type UnaryExpression, type UninstallStackRequest, type UnsignedIntegerLiteral, type UpdateStackRequest, type User, type UserResponse, type Users, UsersAPI, type Variable, type VariableAssignment, type VariableProperties, type Variables, VariablesAPI, type View, type ViewProperties, WriteAPI, type XYGeom, type XYViewProperties };
