var base = require('./base');

class Mewtocol extends base
{
    constructor()
    {
        super();
        this.StartAddr = 0;
        this.NumberOfRegs = 1;
        this.datatype = "DT";
        this.Station = 0;
        this.writeData = [];
        this.writeBit = 0;
        this.selfProtocol = "MEWTOCOL";
        this.MAX_READ_REGS = 26;
        this.MAX_WRITE_REGS = 24;
        this.MAX_READ_BITS = 8;
    }

    /***************************************************************************/
    /**
     * @brief GenerateReadRegisters - Generate the Mewtocol read command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateReadRegisters(startAddr = this.StartAddr, numbOfRegs = this.NumberOfRegs, station = this.Station, datatype = this.datatype)
    {
        let zeroes = "00000";
        let result = "";

        switch (true)
        {
            case station === 0:
                result = "%EE";
                break;
            case station > 0 && station < 10:
                result = "%0" + station.toString();
                break;
            case station >= 10 && station <= this.MAX_STATION:
                result = "%" + station.toString();
                break;
            default:
                result = "%EE";
        }

        switch (datatype.toUpperCase())
        {
            case "DT":
                result += "#RDD";
                break;
            case "FL":
                result += "#RDF";
                break;
            case "LD":
                result += "#RDL";
                break;
            default:
                result += "#RDD";
        }
        
        let reg = zeroes + startAddr.toString();
        result += reg.slice(reg.length - zeroes.length);
        reg = zeroes + (startAddr + numbOfRegs - 1).toString();
        result += reg.slice(reg.length - zeroes.length);
        result += this.bcc(result) + String.fromCharCode(13);
        return result;
    }

    /***************************************************************************/
    /**
     * @brief GenerateWriteRegisters - Generate the Mewtocol write command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateWriteRegisters(startAddr = this.StartAddr, station = this.Station, datatype = this.datatype, writedata = this.writeData)
    {
        let zeroes = "00000";
        let result = "";
        let len = Array.isArray(writedata) ? writedata.length : 1;

        switch (true)
        {
            case station === 0:
                result = "%EE";
                break;
            case station > 0 && station < 10:
                result = "%0" + station.toString();
                break;
            case station >= 10 && station <= this.MAX_STATION:
                result = "%" + station.toString();
                break;
            default:
                result = "%EE";
        }

        switch (datatype.toUpperCase())
        {
            case "DT":
                result += "#WDD";
                break;
            case "FL":
                result += "#WDF";
                break;
            case "LD":
                result += "#WDL";
                break;
            default:
                result += "#WDD";
        }

        let reg = zeroes + startAddr.toString();
        result += reg.slice(reg.length - zeroes.length);
        reg = zeroes + (startAddr + len - 1).toString();
        result += reg.slice(reg.length - zeroes.length);

        let nip1, nip2, nip3, nip4, tmpVal;

        for (let i = 0; i < len; i++)
        {
            tmpVal = zeroes + this.d2h(Array.isArray(writedata) ? writedata[i] : writedata);
            tmpVal = tmpVal.slice(tmpVal.length - 4);

            nip1 = tmpVal.slice(2, 3);
            nip2 = tmpVal.slice(3, 4);
            nip3 = tmpVal.slice(0, 1);
            nip4 = tmpVal.slice(1, 2);

            tmpVal = nip1.toUpperCase() + nip2.toUpperCase() + nip3.toUpperCase() + nip4.toUpperCase();
            tmpVal = zeroes + tmpVal;

            result += tmpVal.slice(tmpVal.length - 4);
        }

        result += this.bcc(result) + String.fromCharCode(13);

        return result;
    }

    /***************************************************************************/
    /**
     * @brief GenerateWriteWords - Generate the Mewtocol write command
     *
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateWriteWords(startAddr = this.StartAddr, station = this.Station, datatype = this.datatype, writedata = this.writeData)
    {
        let zeroes = "00000";
        let result = "";
        let len = Array.isArray(writedata) ? writedata.length : 1;

        switch (true)
        {
            case station === 0:
                result = "%EE";
                break;
            case station > 0 && station < 10:
                result = "%0" + station.toString();
                break;
            case station >= 10 && station <= this.MAX_STATION:
                result = "%" + station.toString();
                break;
            default:
                result = "%EE";
        }

        switch (datatype.toUpperCase())
        {
            case "WR":
                result += "#WCCR";
                break;
            case "WY":
                result += "#WCCY";
                break;
            case "WL":
                result += "#WCCL";
                break;
            default:
                result += "#WCCR";
        }

        let reg = zeroes + startAddr.toString();
        result += reg.slice(reg.length - 4);
        reg = zeroes + (startAddr + len - 1).toString();
        result += reg.slice(reg.length - 4);

        let nip1, nip2, nip3, nip4, tmpVal;

        for (let i = 0; i < len; i++)
        {
            tmpVal = zeroes + this.d2h(Array.isArray(writedata) ? writedata[i] : writedata);
            tmpVal = tmpVal.slice(tmpVal.length - 4);

            nip1 = tmpVal.slice(2, 3);
            nip2 = tmpVal.slice(3, 4);
            nip3 = tmpVal.slice(0, 1);
            nip4 = tmpVal.slice(1, 2);

            tmpVal = nip1.toUpperCase() + nip2.toUpperCase() + nip3.toUpperCase() + nip4.toUpperCase();
            tmpVal = zeroes + tmpVal;

            result += tmpVal.slice(tmpVal.length - 4);
        }

        result += this.bcc(result) + String.fromCharCode(13);

        return result;
    }

    /***************************************************************************/
    /**
     * @brief GenerateWriteSingleBit - Generate the Mewtocol write command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateWriteSingleBit(startAddr = this.StartAddr, station = this.Station, datatype = this.datatype, writebit = this.writeBit)
    {
        let zeroes = "0000";
        let result = "";

        switch (true)
        {
            case station === 0:
                result = "%EE";
                break;
            case station > 0 && station < 10:
                result = "%0" + station.toString();
                break;
            case station >= 10 && station <= this.MAX_STATION:
                result = "%" + station.toString();
                break;
            default:
                result = "%EE";
        }

        switch (datatype.toUpperCase())
        {
            case "R":
                result += "#WCSR";
                break;
            case "Y":
                result += "#WCSY";
                break;
            case "L":
                result += "#WCSL";
                break;
            default:
                result += "#WCSR";
        }

        let reg = zeroes + startAddr.toString();
        result += reg.slice(reg.length - zeroes.length);
        
        result += writebit.toString();
        result += this.bcc(result) + String.fromCharCode(13);

        return result;
    }

    /***************************************************************************/
    /**
     * @brief GeneratReadWords - Generate the Mewtocol read command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GeneratReadWords(startAddr = this.StartAddr, numbOfRegs = this.NumberOfRegs, station = this.Station, datatype = this.datatype)
    {
        let zeroes = "0000";
        let result = "";

        switch (true)
        {
            case station === 0:
                result = "%EE";
                break;
            case station > 0 && station < 10:
                result = "%0" + station.toString();
                break;
            case station >= 10 && station <= this.MAX_STATION:
                result = "%" + station.toString();
                break;
            default:
                result = "%EE";
        }

        switch (datatype.toUpperCase())
        {
            case "WX": result += "#RCCX";
                break;
            case "WY": result += "#RCCY";
                break;
            case "WR": result += "#RCCR";
                break;
            case "WL": result += "#RCCL";
                break;
            default:
                result += "#RCCR";
        }

        let reg = zeroes + startAddr.toString();
        result += reg.slice(reg.length - zeroes.length);
        reg = zeroes + (startAddr + numbOfRegs - 1);
        result += reg.slice(reg.length - zeroes.length);
        result += this.bcc(result) + String.fromCharCode(13);
        return (result);
    }

    /***************************************************************************/
    /**
     * @brief GenerateReadSingleBit - Generate the Mewtocol read command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateReadSingleBit(startAddr = this.StartAddr, station = this.Station, datatype = this.datatype)
    {
        let zeroes = "0000";
        let result = "";

        switch (true)
        {
            case station === 0:
                result = "%EE";
                break;
            case station > 0 && station < 10:
                result = "%0" + station.toString();
                break;
            case this.Station >= 10 && station <= this.MAX_STATION:
                result = "%" + station.toString();
                break;
            default:
                result = "%EE";
        }

        switch (datatype.toUpperCase())
        {
            case "X": result += "#RCSX";
                break;
            case "Y": result += "#RCSY";
                break;
            case "R": result += "#RCSR";
                break;
            case "L": result += "#RCSL";
                break;
            default:
                result += "#RCSR";
        }

        let reg = zeroes + startAddr.toString();
        result += reg.slice(reg.length - zeroes.length);
        result += this.bcc(result) + String.fromCharCode(13);
        return result;
    }

    /***************************************************************************/
    /**
     * @brief GenerateReadBits - Generate the Mewtocol read command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateReadBits(startAddr = this.StartAddr, station = this.Station, numbOfRegs = this.NumberOfRegs, datatype = this.datatype)
    {
        let zeroes = "0000";
        let result = "";

        switch (true)
        {
            case station === 0:
                result = "%EE";
                break;
            case station > 0 && station < 10:
                result = "%0" + station.toString();
                break;
            case this.Station >= 10 && station <= this.MAX_STATION:
                result = "%" + station.toString();
                break;
            default:
                result = "%EE";
        }

        result += `#RCP${numbOfRegs > this.MAX_READ_BITS ? this.MAX_READ_BITS.toString() : numbOfRegs.toString()}${datatype.toUpperCase()}`;        

        let reg, tmp = startAddr;

        for (let i = 0; i < numbOfRegs; i++)
        {
            if (i === 0)
            {
                reg = zeroes + startAddr.toString();
                result += reg.slice(reg.length - zeroes.length);
            }
            else
            {
                tmp = this.IncrementMewBoolAddress(tmp);
                reg = zeroes + tmp;
                result += datatype.toUpperCase() + reg.slice(reg.length - zeroes.length);
            }
        }

        result += this.bcc(result) + String.fromCharCode(13);
        return result;
    }

    /***************************************************************************/
    /**
     * @brief DecodeReadRegisters - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns an object with decoded values
     ****************************************************************************/
    DecodeReadRegisters(response)
    {
        let responseASCII = response.toString();
        let result = {
            err: false,
            err_code: 0,
            int: [],
            uint: [],
            udint: [],
            dint: [],
            hex: [],
            real: [],
            string: ""
        }

        let i, j;
        
        if (responseASCII.charAt(3) != '$')
        {
            result.err = true;
            result.err_code = responseASCII.substr(4, 2);
            return result;
        }

        let data = responseASCII.substring(6, responseASCII.length-3);
        let dataCount = data.length / 4;
        let SingleData;

        for (i = 0; i < dataCount; i++)
        {
            //Extract one word. Save as ASCII hex string
            j = (i * 4);
            SingleData = data.substring(j + 2, j + 4);
            //Swap bytes
            SingleData += data.substring(j, j + 2);
            result.uint.push(this.h2d(SingleData));
            result.int.push(this.HexStringToINT(SingleData));
            result.hex.push(SingleData);
            result.string += String.fromCharCode(this.h2d(data.substring(j, j + 2)));
            result.string += String.fromCharCode(this.h2d(data.substring(j + 2, j + 4)));
        }

        //combine DINT and UDINT
        let len = result.uint.length;
        for (i = 0; i < len; i=i+2)
        {
            if (i + 1 !== len)
            {
                let tmp = result.hex[i + 1] + result.hex[i];
                let udint = this.h2d(tmp);
                result.udint.push(udint);
                result.dint.push(this.HexStringToDINT(tmp));
                result.real.push(this.CalculateIEEEReal(udint));
            }
        }

        return result;
    }

    /***************************************************************************/
    /**
     * @brief DecodeSingleBit - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns an object with decoded values
     ****************************************************************************/
    DecodeSingleBit(response)
    {
        let responseASCII = response.toString();
        let result = {
            err: false,
            err_code: 0,
            state: 0
        }

        if (responseASCII.charAt(3) != '$')
        {
            result.err = true;
            result.err_code = responseASCII.substr(4, 2);
            return result;
        }

        let data = responseASCII.substr(6, 1);
        result.state = data === "0" ? 0 : 1;

        return result;
    }

    /***************************************************************************/
    /**
     * @brief DecodeReadBits - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns an object with decoded values
     ****************************************************************************/
    DecodeReadBits(response, iNumberOfContacts = this.NumberOfRegs)
    {
        let responseASCII = response.toString();
        let result = {
            err: false,
            err_code: 0,
            state: []
        }

        if (responseASCII.charAt(3) != '$')
        {
            result.err = true;
            result.err_code = responseASCII.substr(4, 2);
            return result;
        }

        let data = responseASCII.substring(6);

        for (let i = 0; i < iNumberOfContacts; i++)
        {
            data[i] === "0" ? result.state.push(0) : result.state.push(1);
        }

        return result;
    }

    /***************************************************************************/
    /**
     * @brief WriteResponseOK - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns true when the response was ok, otherwise false
     ****************************************************************************/
    WriteResponseOK(response)
    {
        let responseASCII = response.toString();
        let result = {
            err: false,
            err_code: 0,
            err_msg: ""
        };

        if (responseASCII.charAt(3) != '$')
        {
            result.err = true;
            result.err_code = responseASCII.substr(4, 2);
            result.err_msg = "Unexpected respond"
        }

        return result;
    }
}

exports.Mewtocol = Mewtocol;