#!/bin/sh
#
# Called from udev
#
# Attempt to mount any added block devices and umount any removed devices

# Avoid mounting eMMC partitions that are already mounted in rcS
# if EPAD is not yet running
if echo $DEVNAME | grep -q '^/dev/mmcblk1'; then
	pidof EPAD >/dev/null || exit 0
fi

MOUNT="/bin/mount"
PMOUNT="/usr/bin/pmount"
UMOUNT="/bin/umount"
FSCK="/sbin/fsck"
BLOCKDEV="/sbin/blockdev"

# SD card automount can be disabled
if [ "$ACTION" = "add" ] && echo $myname | grep -q '^sdcard'; then
	[ "$(sys_params -l factory/options/fs/sdcard/noAuto)" == "true" ] && exit 0
fi

# Device-specific options
if echo $myname | grep -q '^usbmemory\|^sdcard'; then
	MOUNT_ARGS="-o gid=data,umask=002"  # writable by data group
	EXTERNAL_MEDIA=1
fi

# Otherwise fsck won't find some executables
PATH="/usr/gnu/bin:/usr/local/bin:/bin:/usr/bin:.:/usr/sbin"
export PATH

for line in `grep -v ^# /etc/udev/mount.blacklist`
do
	if [ ` expr match "$DEVNAME" "$line" ` -gt 0 ];
	then
		logger "udev/mount.sh" "[$DEVNAME] is blacklisted, ignoring"
		exit 0
	fi
done

. /etc/default/rcS

if [ -z "$myname" ]; then
	if [ "$DEVNAME" == "$CONFIGOSPARTITION" ]; then
		name="configos"
	elif [ "$DEVNAME" == "$MAINOSPARTITION" ]; then
		name="mainos"
	elif [ -z "$ID_FS_LABEL" ]; then
		name="`basename "$DEVNAME"`"
	else
		name="`basename "$ID_FS_LABEL"`"
		[ "$name" == "data" ] && ( df / | sed -n 2p | grep -qv '^/dev' ) && name=data-mmc
	fi
else
	name="$myname"
fi

name=${name,,}

mount_dir="/mnt/$name"
[ -n "$EXTERNAL_MEDIA" ] &&  mount_dir="/mnt/media/$name"

automount() {

	! test -d "$mount_dir" && mkdir -p "$mount_dir"

	# Silent util-linux's version of mounting auto
	if [ "x`readlink $MOUNT`" = "x/bin/mount.util-linux" ] ;
	then
		MOUNT_ARGS="$MOUNT_ARGS -o silent"
	fi

	[ -n "$opts" ] && MOUNT_ARGS="$MOUNT_ARGS -o $opts"

	if ! $MOUNT -t auto $MOUNT_ARGS $DEVNAME "$mount_dir"
	then
		rm_dir "$mount_dir"
	else
		logger "mount.sh/automount" "Auto-mount of [/mnt/$name] successful"
		if [ -n "$EXTERNAL_MEDIA" ]; then
			[ -e /mnt/$name ] && rm -rf /mnt/$name
			ln -sf $mount_dir /mnt/$name
		fi
		touch "/tmp/.automount-$name"
	fi
}

rm_dir() {
	# We do not want to rm -r populated directories
	if test "`find "$1" | wc -l | tr -d " "`" -lt 2 -a -d "$1"
	then
		! test -z "$1" && rm -r "$1"
	else
		logger "mount.sh/automount" "Not removing non-empty directory [$1]"
	fi
}

if [ "$ACTION" = "add" ] && [ -n "$DEVNAME" ] && [ -n "$ID_FS_TYPE" ]; then

	if [ "$SECURE_BOOT" = "yes" ] && [ "$name" = "mainos" -o "$name" = "configos" ]; then
		# Make sure verity devices don't get altered
		$BLOCKDEV --setro $DEVNAME
	else
		# Perform fsck if device is not mounted
		[ -z "$( mount | grep "$DEVNAME" )" ] && [[ "$DEVNAME" == *"mmc"* ]] && $FSCK -a $DEVNAME
	fi

	if [ -x "$PMOUNT" ]; then
		$PMOUNT $MOUNT_ARGS $DEVNAME 2> /dev/null
	elif [ -x $MOUNT ]; then
		$MOUNT $MOUNT_ARGS $DEVNAME 2> /dev/null
	fi

	# If the device isn't mounted at this point, it isn't
	# configured in fstab (note the root filesystem can show up as
	# /dev/root in /proc/mounts, so check the device number too)
	if expr $MAJOR "*" 256 + $MINOR != `stat -c %d /`; then
		grep -q "^$DEVNAME " /proc/mounts || automount
	fi
fi

if [ "$ACTION" = "remove" ] && [ -x "$UMOUNT" ] && [ -n "$DEVNAME" ]; then
	for mnt in `cat /proc/mounts | grep "$DEVNAME" | cut -f 2 -d " " `
	do
		$UMOUNT -l $mnt # lazy umount in case files are in use
	done

	# Remove empty directories from auto-mounter
	test -e "/tmp/.automount-$name" && rm_dir "$mount_dir"
	[ -n "$EXTERNAL_MEDIA" ] && rm /mnt/$name
fi

