﻿const fs = require('fs');
const os = require('os');
const exec = require('child_process').exec;
const execSync = require('child_process').execSync;
const path = require('path');
const JSZip = require('jszip');
const md5File = require('md5-file');
const net = require('net');
const crc32 = require('crc-32');
const ncp = require('ncp').ncp;
const rimraf = require("rimraf");
const Mew = require('./mewtocol');
const Mew7 = require('./mewtocol7');
const Modbus = require('./modbus');
const base = require('./base');
const crypto = require('crypto');
const glob = require("glob");
const { isBinaryFileSync } = require("isbinaryfile");

/***************************************************************************
 * @brief Some constants for server usage
 * 
 * Set LOCALTEST to false for deploy!!!
 * Set DEBUG to true when you want to see debug output information. The 
 * information will be written to /home/admin/logs
 ***************************************************************************/

//Do not set LOCALTEST to true
const LOCALTEST = false;
//Set DEBUG to true when debug is wished
const DEBUG = false;

/***************************************************************************
 * @brief Class constructor
 ***************************************************************************/
class pewModule extends base
{
    constructor()
    {
        super();
        this.Constants = {
            isCustomized: false,
            httpStatus: {
                OK: 200,
                MOVED: 301,
                BAD_REQUEST: 400,
                UNAUTHORIZED: 401,
                FORBIDDEN: 403,
                NOT_FOUND: 404,
                METHOD_NOT_ALLOWED: 405,
                SERVER_ERR: 500
            },
            requests: {
                READ_INTERFACE: "/pew",
                READ_PORTS: "/pew_redirect",
                READ_DATALOGGER: "/pew_log",
                READ_MQTT: "/pew_mqtt",
                READ_TIME: "/pew_rtc",
                READ_FTPC: "/pew_ftpc",
                READ_SCRIPT: "/pew_script",
                READ_SQL: "/pew_sql",
                READ_NOSQL: "/pew_nosql",
                READ_60870: "/pew_60870",
                READ_HTTPC: "/pew_httpc",
                READ_API: "/pew_api",
                READ_SMTP: "/pew_email",
                READ_TLS: "/pew_tls",
                GETINDEX: "/",
                UPLOAD: LOCALTEST ? "/fp_config/upload" : "/upload",
                DELETE: "/delete",
                DELETE_LOG_FILE: "/delete_logfile",
                DELETE_CERT: "/delete_cert",
                DELETE_USER_WEB_FILES: "/delete_user_webfiles",
                GET_FORMATFILES: "/get_cfg",
                GET_FILES: "/download",
                WRITE: "/writefile",
                BACKUP: "/get_backup",
                RESTORE: "/restore",
                RESET: "/get_reset",
                PLCTEST: "/plctest",
                SQLTEST: "/get_sqltest",
                NOSQLTEST: "/get_nosqltest",
                GETVER: "/get_ver",
                GET_DEFAULT_SCRIPT: "/get_defscript",
                GET_SYS_LOGS: "/get_syslogs",
                GET: "/get",
                GET_HTTP_CONTENT: "/get_http_content",
                GETFILELIST_INTERN: "/get_files_intern",
                GETFILELIST_USB: "/get_files_usb",
                GETFILELIST_TEMP: "/get_files_temp",
                GET_UPDATE_STATUS: "/get_update_status",
                PLC_POST: "/plcpost",
                PLC_GET: "/plcget",
                API_TEST: "/apitest",
                //Customized specific
                SET: "/set",
                SET_BIT: "/setbit",
                SET_PLC: "/setplc",
                SET_PLC_DEFAULT: "/setplcdefault",
                GET_DEVICES: "/pew_pdc",
                GET_COUNT_VALUES: "/get_counts",
                GET_ERR_LOGS: "/get_errorlogs",
                GET_PLC_BACKUP: "/get_plcbackup",
            },

            configFiles: ["pew.json", "pew_redirect.json", "pew_log.json", "pew_60870.json", "pew_api.json", "pew_email.json", "pew_ftpc.json", "pew_httpc.json", "pew_mqtt.json", "pew_rtc.json", "pew_script.json", "pew_sql.json", "pew_tls.json", "pew_nosql.json"],

            configFilesWithPW: ["pew_email.json", "pew_ftpc.json", "pew_sql.json", "pew_httpc.json", "pew_nosql.json"],

            sqlConfigFile: "pew_sql.json",
            nosqlConfigFile: "pew_nosql.json",
            mqttConfigFile: "pew_mqtt.json",

            apps: ["api_server", "tlsClient", "tlsServer", "sqlClient", "nosequalClient", "rs232transparent", "rs485transparent"],

            certTypes: {
                crt: ".CRT",
                pem: ".PEM",
                csr: ".CSR",
                key: ".KEY",
                der: ".DER",
                cer: ".CER"
            },

            certFilename: {
                cacert: "ca",
                client: "client",
                key: "key",
                cacert_ftps: "ca_ftps",
                client_ftps: "client_ftps",
                key_ftps: "key_ftps",
                cacert_https: "ca_https",
                client_https: "client_https",
                key_https: "key_https",
                cacert_smtp: 'ca_smtp',
                client_smtp: 'client_smtp',
                key_smtp: 'key_smtp',
                cacert_tlsclient: "tlsclientca",
                client_tlsclient: "tlsclientcert",
                key_tlsclient: "tlsclientkey",
                cacert_tlsserver: "tlsserverca",
                cert_tlsserver: "tlsservercert",
                key_tlsserver: "tlsserverkey",
                slqclientca: "ca_sql",
                sqlclientcert: "client_sql",
                sqlclientkey: "key_sql",
                noslqclientca: "ca_nosql",
                nosqlclientcert: "client_nosql",
                nosqlclientkey: "key_nosql"
            },

            firmwareUpdateFiles: {
                package: "updatePackage",
                md5: "updateMD5"
            },

            firmwareFileTypes: {
                package: ".ZIP",
                md5: ".MD5"
            },

            firmwareFixedNames: {
                package: "package.zip",
                md5: "data.md5"
            },

            configSubfolders: {
                mqtt: "config/mqtt/",
                ftpc: "config/ftpc/",
                log: "config/log/",
                script: "config/script/",
                http: "config/httpc/",
                smtp: "config/smtp",
                http_server: "config/http",
                tls: "config/tls",
                sql: "config/sql",
                nosql: "config/nosql"
            },

            MEWCommands: {
                MewtocolStatus: "%EE#EX00RT00**\r",
                Mewtocol7Status: ">@EEE00#30STRD0043F1\r",
                MewUpload: "<EE#0102E\r",
                MultiframeResponse: "<EE3C&\r",
                MewSetToProg: "%EE#RMP49\r",
                MewSetToRun: "%EE#RMR4B\r",
                MewStartDownload: "<EE#020",
                MewDownloadLastResponse: "<EE$"
            },

            PROTOCOLS: {
                mewtocol: "MEWTOCOL",
                mewtocol7: "MEWTOCOL7",
                modbus: "MODBUS",
                transparent: "TRANSPARENT"
            },

            CRCMew7: ["43F1", "0DE3", "8A0D", "0F58", "8DC0", "0895", "8F7B", "0A2E", "825A", "070F",
                "8D1B", "084E", "8FA0", "0AF5", "886D", "0D38", "8AD6", "0F83", "87F7", "02A2",
                "83EC", "06B9", "8157", "0402", "869A", "03CF", "8421", "0174", "8900", "0C55",
                "8641", "0314", "84FA", "01AF", "8337", "0662", "818C", "04D9", "8CAD", "09F8",
                "9E02", "1B57", "9CB9", "19EC", "9B74", "1E21", "99CF", "1C9A", "94EE", "11BB",
                "9BAF", "1EFA", "9914", "1C41", "9ED9", "1B8C", "9C62", "1937", "9143", "1416",
                "9558", "100D", "97E3", "12B6", "902E", "157B", "9295", "17C0", "9FB4", "1AE1",
                "90F5", "15A0", "924E", "171B", "9583", "10D6", "9738", "126D", "9A19", "1F4C",
                "A5DE", "208B", "A765", "2230", "A0A8", "25FD", "A213", "2746", "AF32", "2A67",
                "A073", "2526", "A2C8", "279D", "A505", "2050", "A7BE", "22EB", "AA9F", "2FCA"],
            cipher: "aes-256-cbc",
            pwPlaceHolder: "************",
            pwkey: "",
            localtestkey: "abcdefghijklmno",

            allowedDataAreas: ["DT", "DDT", "FL", "DFL", "LD", "DLD", "WR", "WL", "WY", "WX", "R", "X", "Y", "L", "A", "I"],
            allowedPostAreas: ["DT", "FL", "LD", "WR", "WL", "WY", "WX", "R", "X", "Y", "L"],
            allowedHexAddresses: ["A", "B", "C", "D", "E", "F"],
            dataAreas: {
                DT: { name: "DT", restrictVal: 0 },
                DDT: { name: "DDT", restrictVal: 0 },
                FL: { name: "FL", restrictVal: 1 },
                DFL: { name: "DFL", restrictVal: 1 },
                LD: { name: "LD", restrictVal: 2 },
                DLD: { name: "DLD", restrictVal: 2 },
                WR: { name: "WR", restrictVal: 3 },
                WL: { name: "WL", restrictVal: 4 },
                WY: { name: "WY", restrictVal: 5 },
                WX: { name: "WX", restrictVal: 6 },
                R: { name: "R", restrictVal: 3 },
                X: { name: "X", restrictVal: 6 },
                Y: { name: "Y", restrictVal: 5 },
                L: { name: "L", restrictVal: 4 },
                station: "A",
                interface: "I"
            },
            datatype: {
                bool: { name: "BOOL", value: 0 },
                int: { name: "INT", value: 1 },
                uint: { name: "UINT", value: 2 },
                dint: { name: "DINT", value: 3 },
                udint: { name: "UDINT", value: 4 },
                real: { name: "REAL", value: 5 },
                string: { name: "STRING", value: 6 },
                any: {name: "ANY", value: 7}
            },
            allowedPostReqTypes: ["INT", "UINT", "DINT", "UDINT", "BOOL", "ANY", "STRING", "REAL"],
            //FPG16k, FPG32k, FP0R16k, FP0RC32, FP0RT32, FP0RF32, 3x FPX16k, 3x FPX32k
            plcTypes: ["43", "44", "46", "47", "48", "49", "70", "73", "76", "71", "74", "77"],
            //Some paths and file names constants
            tempFolder: "temp",
            configFolder: "config/",
            configDefault: "config_default/",
            configDefaultFolder: "config_default/script/",
            JSONFILE: ".json",
            PEWMAIN: "pew_main",
            API_SERVER: "api_server",
            TLS_CLIENT: "tlsClient",
            TLS_SERVER: "tlsServer",
            SQL_CLIENT: "sqlClient",
            RS232_TRANSPARENT: "rs232transparent",
            RS485_TRANSPARENT: "rs485transparent",
            NOSQL_CLIENT: "nosequalClient",
            LOG_FORMAT_FILE_NAME: "formatfile",
            LOG_FORMAT_PARAMETER: "user_formatfile",
            USER_WEB_FILES: "httpfiles",
            USER_WEB_FILE_NAME: "user_web_files.zip",
            LOG_FILE_FORMAT: ".CFG",
            SCRIPT_UPLOAD_NAME: "script_file",
            SCRIPT_FILE_NAME: "pew_script.cfg",
            VERSION_FILE: "config_default/pew_version.json",
            PDC_PLC_FILE: "config_default/pdc.plc",
            FULL_APP_PATH: "/mnt/data/hmi/pew/deploy",
            API_APP_PATH: "/api_server.js",
            TLS_CLIENT_PATH: "/tlsClient.js",
            TLS_SERVER_PATH: "/tlsServer.js",            
            SQL_APP_PATH: "/sqlClient.js",
            NOSQL_APP_PATH: "/nosequalClient.js",
            RS232TRANSPARENT_APP_PATH: "/rs232transparent.js",
            RS485TRANSPARENT_APP_PATH: "/rs485transparent.js",
            PDC_CONFIG_FILE: "pew_pdc.json",
            configTempFolderBefore: "/mnt/data/.installrt",
            configTempFolder: "/mnt/data/.installrt/pewconfig",
            NODE_V16_PATH: "/mnt/data/hmi/pew/deploy/node_v16/bin/node",
            NODEJS_SERVER_FILES_PATH: "/mnt/data/hmi/pew/deploy/node",

            MAX_UPLOAD_SIZE: 100000000, //max upload size in bytes

            //Some constant strings

            //Error Messages
            WRONG_FILE_FORMAT: "Wrong file format",
            DELETION_NOK: "Failed to delete",
            BACKUP_FAILED: "Backup failed",
            NOT_ALLOWED: "Not allowed",
            SAVE_OK_API_FAILED: "Settings saved but api failed to start",
            MD5_CHECK_FAILED: "Error: MD5 check failed",
            FILE_MISSING: "A file is missing. Please upload both (package and md5) files.",
            FAILED: "Something went wrong, operation failed",
            NO_FILE_SELECTED: "No file selected!",
            NO_FILENAME: "No filename specified",
            FILE_CHECK_FAILED: "The backup file is corrupted or not correct",
            NO_REPONSE: "No response from device",
            API_REQUEST_SYNTAX_ERR: "Syntax Error, please check the request syntax",
            INTERFACE_OUT_OF_RANGE: "Interface out of range.",
            STATION_OUT_OF_RANGE: "Station number out of range.",
            COM_NOT_ENABLED: "COM Port is not enabled",
            TCP_PORT_DISABLED: "No TCP Port has been opened for the COM port",
            DIR_NOT_FOUND: "Directory not found",
            FILE_NOT_FOUND: "File not found",
            LIST_FILE_FAILED: "Failed to list files",
            ZIP_FILES_FAILED: "Failed to zip files",
            READ_CFG_FAILED: "Failed to read CFG files",
            FILE_TO_DELETE_DOES_NOT_EXIST: "The file you are trying to delete does not exists",
            DELETE_FAILED: "Failed to delete file(s)",
            WRONG_FORMAT: "Wrong format received",
            REWRITE_FAILED: "Failed to rewrite a parameter",
            CERT_FILE_NOT_AVAILABLE: "Cert file not available",
            WRITE_CONFIG_FAILED: "Failed to write the config file",
            WRITE_SCRIPT_FILE_FAILED: "Failed to write to script file",
            CRC_FAILED: "Failed to check CRC",
            RESTORE_UNZIP_FAILED: "Failed to unzip or restore the settings",
            RESTART_APP_FAILED: "Failed to restart app by Signal USR1",
            TIMEOUT: "Timeout",
            BUILD_CMD_FAILED: "Failed to build the status command",
            PLC_COM_ERR: "PLC communication error",
            VER_READ_FAILED: "Failed to read the Version number",
            LOAD_DEFAULT_SCRIPT_FAILED: "Failed to load the default script file",
            API_COULD_NOT_START: "API app could not be started",
            PLC_GET_FAILED: "PLC GET request failed",
            PLC_POST_FAILED: "PLC POST request failed",
            EXTRACT_GET_REQ_FAILED: "Extract PLC GET request failed",
            INPUT_WRITING_FORBIDDEN: "Input writing is not permitted (X or WX)",
            SET_INTERFACE_FAILED: "Set interface failed for PLC GET request",
            MEW7_NO_FL: "Mewtocol7 does not have FL registers. Please check your request",
            ADDR_OUT_OF_RANGE: "Address out of range",
            VAL_OUT_OF_RANGE: "Value out of range",
            READ_FILE_FAILED: "Failed to read file",
            FIRMWARE_FILE_NOT_FOUND: "File(s) missing for firmware update. Process cancelled.",
            SAVE_FAILED: "Save setting failed",
            PID_NOT_FOUND: "PID of application could not be found",
            USR_SIG1_FAILED: "USR1 Signal for restarting application failed",
            KILL_FAILED: "Failed to kill application",
            WRITE_TO_SCRIPT_FAILED: "Failed to write content to script file",
            RESTORE_FAILED: "Failed to restore settings",
            PLC_TEST_FAIL: "PLC connection test failed",
            FILE_UPLOAD_FAIL: "Failed to upload file",
            REQ_SYNTAX_FORMAT_ERR: "Request syntax or format error.",
            REQ_MISSING_PARAMS_ERR: "Missing parameter(s)",
            REQ_COUNT_MISSING: "Count parameter must be specified for all types except string",
            REQ_COUNT_LIMIT_EXCEED: "Count max limit has been exceeded",
            REQ_STRING_LIMIT_EXCEED: "Max. string length exceeded",
            REQ_WRONG_AREA_OR_TYPE: "Wrong area or type specified",
            REQ_CNT_VAL_NO_MATCH: "Count and number of values to write does not match",
            REQ_TYPE_MISMATCH: "Type mismatch, write value and data type does not match",
            REQ_IFACE_OR_STATION_OUT_OF_RANGE: "Interface or station number out of range",
            REQ_START_ADDR_TYPE_MISMATCH: "Start address type mismatch",
            REQ_ANY_NOT_ALLOWED_BOOL: "The type ANY does not support data area of Bools",
            MISSING_PARAM: "Missing parameter",
            MEW_RESPONSE_ERR: "Mewtocol response error",
            WRONG_PLC_TYPE: "Connected PLC type is not supported",
            PLC_TYPE_NOT_MATCH: "Connected PLC type does not match with the project",
            SET_PROG_FAILED: "Failed to set PLC to prog mode",
            SET_RUN_FAILED: "Failed to set PLC to run mode",
            MAX_POST_REQUEST_EXCEEDED: "Maximum number of requests has been exceeded",
            EXTRACT_WEB_FILES_FAILED: "Extract user web files failed",
            TCP_PORT_OCCUPIED: "Port is already in use",
            CERT_FILE_MISSING: "Certificate file(s) missing",
            EMPTY_STRING_NOT_ALLOWED: "Empty string not allowed",
            DATA_PROPERTY_MISSING: "Data property in object missing",
            INFLUX_PROPERTY_NOK: "Data type, name or value missing (or not valid)",
            KEY_READ_ERROR: "Read key failed",
            //Messages and log messages
            SAVE_OK: "Successful saved",
            SAVE_OK_API_STARTED: "Successfully saved and started api",
            DELETE_OK: "Successful deleted",
            POST_TRY_FROM_OUTSIDE: "Post try from outside of localhost",
            UPLOAD_OK: "File(s) successful uploaded!",
            UPDATE_SUCCESS: "Update successful",
            API_TEST_SUCCESS: "Test successful",
            FILE_DELETED: "File(s) has been deleted",
            LOG_FILES_DELETED: "Logfile(s) has been deleted",
            BACKUP_REQ: "Backup file was requested",
            PLC_TEST_REQ: "PLC connection test request was requested for Port",
            SEND_MEW_REQ: "Sending standard Mewtocol status request",
            SEND_MEW7_REQ: "Sending Mewtocol7 status request",
            LOAD_DEFAULT_SCRIPT_REQ: "Load the default pew_script file requested",
            API_AUTO_START: "Autostart API app",
            PLC_GET_REQ: "PLC GET request",
            PLC_POST_REQ: "PLC POST request",
            ACCESS_DENIED: "Access from the IP has been denied",
            SINGLE_FILE_DOWNLOAD_REQ: "File download requested",
            FIRMWARE_UPDATE_REQ: "Firmware update requested",
            START_FIRMWARE_UPDATE: "Start FP-I4C Firmware update process...",
            SAVE_REQ: "Save setting request",
            SEND_USR_SIG: "Send a request to restart application",
            FORBIDDEN_REQ: "A request from external was received which has been blocked",
            DELETE_CERT_FILE_REQ: "Delete cert file request",
            WRITE_TO_SCRIPT_REQ: "Write to script request",
            FILE_DOWNLOAD_REQ: "Download all files requested",
            SETTING_RESTORE_REQ: "Settings restore request",
            RESTORE_FILE_EMPTY: "Restore file was empty, no action taken.",
            FILE_UPLOAD_REQ: "File upload request",
            NODE_APP_START: "Node application (fp_config) started at port",
            OPERATION_SUCCESSFUL: "Operation successful",
            OK: "OK",
            //Fix Strings for checks
            LOCAL_HOST: "127.0.0.1",
            LOCAL_HOST_v6: "::1",
            ALL_IP: "0.0.0.0",
            ALLOW_ALL_IPS: "255.255.255.255",
            INTERNAL_MEMORY_LINK: "__INTERNAL",
            USB_MEMORY_LINK: "__USB_STICK",
            TEMP_MEMORY_LINK: "__TEMP",
            GET_ALL_FILES: "all_logfiles",
            EMPTY_STRING: "",
            UNDEFINED: "undefined",
            REMOTE_IP: "192.168.188.118",


            SUCCESS: 0,
            URL_LEVEL_CFG_FILES: 3,
            Error_Position_MewStatus: 3,
            OK_Position_MewStatus: 3,
            Error_Position_Mew7Status: 7,
            OK_Position_Mew7Status: 7,
            Timeout_Status_request: 5000,
            MAX_MEW_ADDR: 99999,
            MAX_MEW7_ADDR: 999999,
            MAX_MODBUS_ADDR: 65535,
            MAX_FL_ADDR: 32767,
            MAX_WR_ADDR: 2048,
            MAX_WL_ADDR: 1024,
            MAX_WXWY_ADDR: 512,
            MAX_LD_ADDR: 16384,
            MAX_BOOL_ADDR_LEN: 5,
            MAX_FRAME_DATA_SIZE: 2000,
            TIMEOUT_GETUPDATE_STATUS: 200000,
            MAX_POST_REQUESTS: 20,
            MAX_COUNT_PER_REQUEST: 100,
            MAX_STRING_LEN_REQUEST: 1000,
            MAX_BITS_WRITE: 50,
            MIN_IP: 0, //0.0.0.0
            MAX_IP: 4294967295, //255.255.255.255
            IDLE_TIMEOUT: 60000,
            MAX_SELECT_VALUES: 2000,
            KEY_LEN: 32,

            //Error codes
            errorCodes: {
                noErr: 0,
                unknown: -999
            },

            port: LOCALTEST ? 8080 : 8081,
            keyServicePort: 497,

            Storage:
            {
                INTERN: { path: "/home/log/log", value: 0 },
                USBSTICK: { path: "/mnt/media/usbmemory/log", value: 1 },
                TEMP: { path: "/var/volatile/pew/log", value: 2 }
            },

            Interfaces: {
                RS232: { service: 0, interface: 0, name: "COM1" },
                RS485: { service: 3, interface: 3, name: "COM2" },
                USB: { service: 1, interface: 1, name: "COM3" },
                INTERN: { service: 4, interface: 4, name: "INTERN" },
                ETHERNET: { service: 5, interface: 5, name: "ETH" }
            },

            writeDataTypes: {
                _16bit: 0,
                _32bit: 1,
                _32bitReal: 2
            },

            //Customized specific
            startAddrPLCVersion: 3001,
            startAddrCountValues: 3232,
            numberOfRegsCountValues: 96,
            startAddrPLCDateTime: 3090,
            startAddrStatus: 3032,
            startAddrErrorLogs: 31505,
            numberOfRegsErrorLogs: 490,
            defaultPort: 9094,

            DebugLog: function (message)
            {
                if (DEBUG)
                {
                    var DEBUG_PATH = "/home/admin/logs";
                    var NODE_DEBUG_FILE = "node_logs.txt";
                    var dt = new Date()
                    var year = dt.getFullYear().toString()
                    var month = dt.getMonth() + 1 < 10 ? "0" + (dt.getMonth() + 1).toString() : (dt.getMonth() + 1).toString()
                    var day = dt.getDate() < 10 ? "0" + dt.getDate().toString() : dt.getDate().toString()
                    var hour = dt.getHours() < 10 ? "0" + dt.getHours().toString() : dt.getHours().toString()
                    var min = dt.getMinutes() < 10 ? "0" + dt.getMinutes().toString() : dt.getMinutes().toString()
                    var secs = dt.getSeconds() < 10 ? "0" + dt.getSeconds().toString() : dt.getSeconds().toString()

                    var logmessage = year + "-" + month + "-" + day + " " + hour + ":" + min + ":" + secs + " : " + message + "\r";
                    try
                    {
                        if (!fs.existsSync(DEBUG_PATH))
                        {
                            fs.mkdirSync(DEBUG_PATH)
                            fs.appendFileSync(path.join(DEBUG_PATH, NODE_DEBUG_FILE), logmessage)
                        }
                        else
                        {
                            fs.appendFileSync(path.join(DEBUG_PATH, NODE_DEBUG_FILE), logmessage)
                        }
                    }
                    catch (ex)
                    {
                        console.log(ex.message)
                    }
                }
            }
        }
    }
    /***************************************************************************/
    //End constructor
    /***************************************************************************/

    /***************************************************************************/
    /**
     * @brief getListOfFilesInDir - Returns all files as an array within a directory
     * 
     * @param dir: The directory to read
     ****************************************************************************/
    getListOfFilesInDir(dir)
    {
        let LocalConstants = this.Constants;
        var tempdir = dir;

        return new Promise((res, rej) =>
        {
            try 
            {
                fs.exists(tempdir, exists =>
                {
                    if (!exists) 
                    {
                        return rej(LocalConstants.DIR_NOT_FOUND);
                    }

                    fs.readdir(tempdir, (err, files) =>
                    {
                        if (err)
                        {
                           return rej(LocalConstants.FILE_NOT_FOUND);
                        }

                        res(JSON.stringify(files));
                    })

                })
            }
            catch (ex) 
            {
                this.Constants.DebugLog(`${LocalConstants.LIST_FILE_FAILED}: ${ex.message}`)
                return rej(ex.message);
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief zipLogFilesAndProvideForDownload - zip log files and provide them for 
     * download
     * 
     * @param logfilePath: The path where all files are stored
     ****************************************************************************/
    zipLogFilesAndProvideForDownload(logfilePath) 
    {
        return new Promise((res, rej) =>
        {
            try 
            {
                var zip = new JSZip()
                var tmpPath = logfilePath;

                this.Constants.DebugLog(`${this.Constants.FILE_DOWNLOAD_REQ}: ${tmpPath}`)

                fs.readdir(tmpPath, (err, files) =>
                {
                    if (err)
                    {
                        return rej(err)
                    }

                    for (let file of files) 
                    {
                        zip.file(file, fs.readFileSync(path.join(tmpPath, file)))
                    }

                    zip.generateAsync({ type: "base64" }).then(content =>
                    {
                        res(content);
                    })
                })
            }
            catch (ex)
            {
                //Show Debug message
                this.Constants.DebugLog(`${this.Constants.ZIP_FILES_FAILED}: ${ex.message}`);
                return rej(ex.message);
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief getCFGFiles - Returns all available cfg files
     * 
     * @param req: the request of the client
     ****************************************************************************/
    getCFGFiles() 
    {
        //Constants for asynchronous tasks
        var LocalConstants = this.Constants;
        return new Promise((res, rej) =>
        {
            try 
            {
                fs.readdir(path.join(__dirname, LocalConstants.configSubfolders.log), function (err, files) 
                {
                    if (err) 
                    {
                        LocalConstants.DebugLog(`${LocalConstants.READ_CFG_FAILED}: ${err.message}`);
                        rej(err);
                    }
                    res(files);
                });
            }
            catch (ex) 
            {
                LocalConstants.DebugLog(`${LocalConstants.READ_CFG_FAILED}: ${ex.message}`);
                rej(ex.message);
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief deleteCertFile - Delete cert files
     * 
     * @param content: The content consists of filename and the field (ca, client or key)
     ****************************************************************************/
    deleteCertFile(content)
    {
        var filename = content.filename;
        var type, file, confFile, parameter;
        //Constants for asynchronous tasks
        type = content.field;
        var LocalModule = this;
        const { Constants, Constants: { JSONFILE, certFilename, requests, DebugLog, configSubfolders, configFolder } } = this;

        return new Promise((res, rej) =>
        {
            switch (type) 
            {
                case certFilename.cacert:
                    file = path.join(__dirname, configSubfolders.mqtt, filename);
                    confFile = `${configFolder}${requests.READ_MQTT}${JSONFILE}`;
                    parameter = certFilename.cacert;
                    break;

                case certFilename.client:
                    file = path.join(__dirname, configSubfolders.mqtt, filename);
                    confFile = `${configFolder}${requests.READ_MQTT}${JSONFILE}`;
                    parameter = certFilename.client;
                    break;

                case certFilename.key:
                    file = path.join(__dirname, configSubfolders.mqtt, filename);
                    confFile = `${configFolder}${requests.READ_MQTT}${JSONFILE}`;
                    parameter = certFilename.key;
                    break;

                case certFilename.cacert_ftps:
                    file = path.join(__dirname, configSubfolders.ftpc, filename);
                    confFile = `${configFolder}${requests.READ_FTPC}${JSONFILE}`;
                    parameter = certFilename.cacert;
                    break;

                case certFilename.client_ftps:
                    file = path.join(__dirname, configSubfolders.ftpc, filename);
                    confFile = `${configFolder}${requests.READ_FTPC}${JSONFILE}`;
                    parameter = certFilename.client;
                    break;

                case certFilename.key_ftps:
                    file = path.join(__dirname, configSubfolders.ftpc, filename);
                    confFile = `${configFolder}${requests.READ_FTPC}${JSONFILE}`;
                    parameter = certFilename.key;
                    break;

                case certFilename.cacert_https:
                    file = path.join(__dirname, configSubfolders.http, filename);
                    confFile = `${configFolder}${requests.READ_HTTPC}${JSONFILE}`;
                    parameter = certFilename.cacert;
                    break;

                case certFilename.client_https:
                    file = path.join(__dirname, configSubfolders.http, filename);
                    confFile = `${configFolder}${requests.READ_HTTPC}${JSONFILE}`;
                    parameter = certFilename.client;
                    break;

                case certFilename.key_https:
                    file = path.join(__dirname, configSubfolders.http, filename);
                    confFile = `${configFolder}${requests.READ_HTTPC}${JSONFILE}`;
                    parameter = certFilename.key;
                    break;

                case certFilename.cacert_smtp:
                    file = path.join(__dirname, configSubfolders.smtp, filename);
                    confFile = `${configFolder}${requests.READ_SMTP}${JSONFILE}`;
                    parameter = certFilename.cacert;
                    break;

                case certFilename.client_smtp:
                    file = path.join(__dirname, configSubfolders.smtp, filename);
                    confFile = `${configFolder}${requests.READ_SMTP}${JSONFILE}`;
                    parameter = certFilename.client;
                    break;

                case certFilename.key_smtp:
                    file = path.join(__dirname, configSubfolders.smtp, filename);
                    confFile = `${configFolder}${requests.READ_SMTP}${JSONFILE}`;
                    parameter = certFilename.key;
                    break;

                case certFilename.slqclientca:
                    file = path.join(__dirname, configSubfolders.sql, filename);
                    confFile = `${configFolder}${requests.READ_SQL}${JSONFILE}`;
                    parameter = certFilename.cacert;
                    break;

                case certFilename.sqlclientcert:
                    file = path.join(__dirname, configSubfolders.sql, filename);
                    confFile = `${configFolder}${requests.READ_SQL}${JSONFILE}`;
                    parameter = certFilename.client;
                    break;

                case certFilename.sqlclientkey:
                    file = path.join(__dirname, configSubfolders.sql, filename);
                    confFile = `${configFolder}${requests.READ_SQL}${JSONFILE}`;
                    parameter = certFilename.key;
                    break;

                case certFilename.noslqclientca:
                    file = path.join(__dirname, configSubfolders.nosql, filename);
                    confFile = `${configFolder}${requests.READ_NOSQL}${JSONFILE}`;
                    parameter = certFilename.cacert;
                    break;

                case certFilename.nosqlclientcert:
                    file = path.join(__dirname, configSubfolders.nosql, filename);
                    confFile = `${configFolder}${requests.READ_NOSQL}${JSONFILE}`;
                    parameter = certFilename.client;
                    break;

                case certFilename.nosqlclientkey:
                    file = path.join(__dirname, configSubfolders.nosql, filename);
                    confFile = `${configFolder}${requests.READ_NOSQL}${JSONFILE}`;
                    parameter = certFilename.key;
                    break;

                case certFilename.cacert_tlsclient:
                case certFilename.client_tlsclient:
                case certFilename.key_tlsclient:
                case certFilename.cacert_tlsserver:
                case certFilename.cert_tlsserver:
                case certFilename.key_tlsserver:
                    file = path.join(__dirname, configSubfolders.tls, filename);
                    confFile = `${configFolder}${requests.READ_TLS}${JSONFILE}`;
                    parameter = type;
                    break;

                case Constants.LOG_FORMAT_FILE_NAME:
                    file = path.join(__dirname, configSubfolders.log, filename);
                    confFile = `${configFolder}${requests.READ_DATALOGGER}${JSONFILE}`;
                    parameter = Constants.LOG_FORMAT_FILE_NAME;
                    break;

                case Constants.SCRIPT_UPLOAD_NAME:
                    file = path.join(__dirname, Constants.configSubfolders.script, filename);
                    confFile = `${configFolder}${requests.READ_SCRIPT}${JSONFILE}`;
                    parameter = Constants.SCRIPT_UPLOAD_NAME;
                    break;

                default:
                    DebugLog(Constants.FILE_TO_DELETE_DOES_NOT_EXIST)
                    rej(Constants.FAILED);
                    return;
            }

            if (typeof filename !== Constants.UNDEFINED)
            {
                try 
                {
                    var bError;
                    //Check first if the file is available
                    fs.exists(file, exists =>
                    {
                        if (!exists)
                        {
                            //File not available but in the config file the entry exists -> devare the entry of conf file
                            bError = LocalModule.rewriteConfFile(confFile, parameter, Constants.EMPTY_STRING);
                            DebugLog(Constants.FILE_TO_DELETE_DOES_NOT_EXIST);
                            rej(bError)
                        }
                        else 
                        {
                            //File available -> delete it
                            fs.unlink(file, err =>
                            {
                                if (err)
                                {
                                    DebugLog(`${Constants.DELETE_FAILED}: ${err.message}`);
                                    return rej(err);
                                }

                                bError = LocalModule.rewriteConfFile(confFile, parameter, Constants.EMPTY_STRING)
                                DebugLog(`${Constants.FILE_DELETED}: ${file}`)
                                bError ? rej() : res();
                            })
                        }
                    })
                }
                catch (err)
                {
                    DebugLog(`${Constants.DELETE_FAILED}: ${err.message}`);
                    return rej(err);
                }
            }
            else
            {
                return rej(Constants.NO_FILENAME);
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief deleteLogFile - Delete log files
     * 
     * @param content: The content consists of filenames and the storage
     ****************************************************************************/
    deleteLogFile(content)
    {
        var errCount = 0;
        var files = content.files;
        var storage = content.storage;
        var deleteAll = content.delete_all;

        return new Promise((res, rej) =>
        {
            var fileWithPath = this.Constants.Storage.TEMP.path;
            for (let val of Object.keys(this.Constants.Storage)) {
                if (this.Constants.Storage[val].value === storage) {
                    fileWithPath = this.Constants.Storage[val].path;
                    break;
                }
            }

            if (this.isUndefined(files) || this.isUndefined(storage) || this.isUndefined(deleteAll)) {
                return rej(this.Constants.WRONG_FORMAT);
            }

            //Delete files
            let filesToRemove = deleteAll ? fs.readdirSync(fileWithPath) : files;
            for (let file of filesToRemove) {
                let filepath = path.join(fileWithPath, file);
                try {
                    fs.unlinkSync(filepath);
                }
                catch (ex) {
                    errCount++;
                    this.Constants.DebugLog(`${this.Constants.DELETE_FAILED}: ${ex.message}`);
                }
            }

            if (errCount > 0) {
                this.Constants.DebugLog(`${this.Constants.DELETE_FAILED}. Number of failed: ${ErrorCounts.toString()}`);
                return rej(this.Constants.DELETE_FAILED);
            }

            this.Constants.DebugLog(this.Constants.LOG_FILES_DELETED);
            res();
        })
    }

    /***************************************************************************/
    /**
     * @brief rewriteConfFile - rewrite the conf file with specified value
     * e.g. MQTT if file has been devared rewrite the config file 
     * 
     * @param confFile: the config file where the parameter should be written to
     * @param parameter: the JSON parameter which should be written to
     * @param value: the value which should be written to the parameter
     * Returns true if error otherwise false
     ****************************************************************************/
    rewriteConfFile(confFile, parameter, value) 
    {
        try
        {
            var data = fs.readFileSync(confFile);
            var jsonData;
            const { Constants } = this;

            try 
            {
                jsonData = JSON.parse(data);
                var LogConf = `${Constants.configFolder}${Constants.requests.READ_DATALOGGER}${Constants.JSONFILE}`;

                //For data logging if file is deleted also check the setting of user selected file. If it is the user defined one delete the entry
                if (LogConf === confFile) 
                {
                    if (jsonData[Constants.LOG_FORMAT_FILE_NAME] === jsonData[parameter]) 
                    {
                        jsonData[Constants.LOG_FORMAT_FILE_NAME] = value;
                    }
                }
                jsonData[parameter] = value;
                //Now write the new file
                fs.writeFileSync(confFile, JSON.stringify(jsonData))
                return false;
            }
            catch (ex) 
            {
                Constants.DebugLog(`${Constants.REWRITE_FAILED}:  ${confFile} - ${parameter} -- ${ex.message}`);
                return true;
            }
        }
        catch (exception)
        {
            Constants.DebugLog(`${Constants.REWRITE_FAILED}:  ${confFile} - ${parameter} -- ${exception.message}`);
            return true;
        }

    }

    /***************************************************************************/
    /**
     * @brief writeCertFile - Write Certificate name into the json config
     * file after the file has been uploaded
     * 
     * @param file: The setting where the file has been uploaded and the json should be written
     ****************************************************************************/
    writeCertFile(file) 
    {
        var CertFilename;
        //Constants for asynchronous tasks
        var LocalConstants = this.Constants;

        //Set the correct filename (cert filenames are fixed ca, client and key but the file ending differs (e.g. crt, csr, pem))
        switch (file.fieldname) 
        {
            case LocalConstants.certFilename.cacert:
            case LocalConstants.certFilename.cacert_ftps:
            case LocalConstants.certFilename.cacert_https:
            case LocalConstants.certFilename.cacert_smtp:
            case LocalConstants.certFilename.slqclientca:
            case LocalConstants.certFilename.noslqclientca:
                CertFilename = LocalConstants.certFilename.cacert + path.extname(file.originalname);
                break;

            case LocalConstants.certFilename.client:
            case LocalConstants.certFilename.client_ftps:
            case LocalConstants.certFilename.client_https:
            case LocalConstants.certFilename.client_smtp:
            case LocalConstants.certFilename.sqlclientcert:
            case LocalConstants.certFilename.nosqlclientcert:
                CertFilename = LocalConstants.certFilename.client + path.extname(file.originalname);
                break;

            case LocalConstants.certFilename.key:
            case LocalConstants.certFilename.key_ftps:
            case LocalConstants.certFilename.key_https:
            case LocalConstants.certFilename.key_smtp:
            case LocalConstants.certFilename.sqlclientkey:
            case LocalConstants.certFilename.nosqlclientkey:
                CertFilename = LocalConstants.certFilename.key + path.extname(file.originalname);
                break;

            case LocalConstants.certFilename.cacert_tlsclient:
                CertFilename = LocalConstants.certFilename.cacert_tlsclient + path.extname(file.originalname);
                break;
            case LocalConstants.certFilename.client_tlsclient:
                CertFilename = LocalConstants.certFilename.client_tlsclient + path.extname(file.originalname);
                break;
            case LocalConstants.certFilename.key_tlsclient:
                CertFilename = LocalConstants.certFilename.key_tlsclient + path.extname(file.originalname);
                break;
            case LocalConstants.certFilename.cacert_tlsserver:
                CertFilename = LocalConstants.certFilename.cacert_tlsserver + path.extname(file.originalname);
                break;
            case LocalConstants.certFilename.cert_tlsserver:
                CertFilename = LocalConstants.certFilename.cert_tlsserver + path.extname(file.originalname);
                break;
            case LocalConstants.certFilename.key_tlsserver:
                CertFilename = LocalConstants.certFilename.key_tlsserver + path.extname(file.originalname);
                break;

            case LocalConstants.LOG_FORMAT_FILE_NAME:
                CertFilename = file.originalname;
                break;

            case LocalConstants.SCRIPT_UPLOAD_NAME:
                CertFilename = LocalConstants.SCRIPT_FILE_NAME;
                break;

            default:
                LocalConstants.DebugLog(LocalConstants.CERT_FILE_NOT_AVAILABLE);
                return false;
        }

        //Check if the file really exists, if so update the config json file
        try 
        {
            let data, jsonData, confFilename;
            //Read the current setting file and overwrite parameter of the certificate file
            switch (file.fieldname) 
            {
                case LocalConstants.certFilename.cacert:
                case LocalConstants.certFilename.client:
                case LocalConstants.certFilename.key:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_MQTT}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    //Add the filename of certificate to the config file
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_MQTT + LocalConstants.JSONFILE);
                    break;

                case LocalConstants.certFilename.cacert_ftps:
                case LocalConstants.certFilename.client_ftps:
                case LocalConstants.certFilename.key_ftps:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_FTPC}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_FTPC + LocalConstants.JSONFILE);
                    break;

                case LocalConstants.certFilename.cacert_tlsclient:
                case LocalConstants.certFilename.client_tlsclient:
                case LocalConstants.certFilename.key_tlsclient:
                case LocalConstants.certFilename.cacert_tlsserver:
                case LocalConstants.certFilename.cert_tlsserver:
                case LocalConstants.certFilename.key_tlsserver:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_TLS}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_TLS + LocalConstants.JSONFILE);
                    break;

                case LocalConstants.certFilename.cacert_https:
                case LocalConstants.certFilename.client_https:
                case LocalConstants.certFilename.key_https:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_HTTPC}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_HTTPC + LocalConstants.JSONFILE);
                    break;

                case LocalConstants.certFilename.cacert_smtp:
                case LocalConstants.certFilename.client_smtp:
                case LocalConstants.certFilename.key_smtp:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_SMTP}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_SMTP + LocalConstants.JSONFILE);
                    break;

                case LocalConstants.certFilename.slqclientca:
                case LocalConstants.certFilename.sqlclientcert:
                case LocalConstants.certFilename.sqlclientkey:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_SQL}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_SQL + LocalConstants.JSONFILE);
                    break;

                case LocalConstants.certFilename.noslqclientca:
                case LocalConstants.certFilename.nosqlclientcert:
                case LocalConstants.certFilename.nosqlclientkey:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_NOSQL}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_NOSQL + LocalConstants.JSONFILE);
                    break;

                case LocalConstants.LOG_FORMAT_FILE_NAME:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_DATALOGGER}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_DATALOGGER + LocalConstants.JSONFILE);
                    break;

                case LocalConstants.SCRIPT_UPLOAD_NAME:
                    data = fs.readFileSync(`${LocalConstants.configFolder}${LocalConstants.requests.READ_SCRIPT}${LocalConstants.JSONFILE}`);
                    jsonData = JSON.parse(data);
                    confFilename = path.join(__dirname, LocalConstants.configFolder, LocalConstants.requests.READ_SCRIPT + LocalConstants.JSONFILE);
                    break;
            }

            //Set the correct value
            switch (file.fieldname) 
            {
                case LocalConstants.certFilename.cacert:
                case LocalConstants.certFilename.cacert_ftps:
                case LocalConstants.certFilename.cacert_https:
                case LocalConstants.certFilename.cacert_smtp:
                case LocalConstants.certFilename.noslqclientca:
                case LocalConstants.certFilename.slqclientca:
                    jsonData.ca = CertFilename;
                    break;

                case LocalConstants.certFilename.client:
                case LocalConstants.certFilename.client_ftps:
                case LocalConstants.certFilename.client_https:
                case LocalConstants.certFilename.client_smtp:
                case LocalConstants.certFilename.sqlclientcert:
                case LocalConstants.certFilename.nosqlclientcert:
                    jsonData.client = CertFilename;
                    break;

                case LocalConstants.certFilename.key:
                case LocalConstants.certFilename.key_ftps:
                case LocalConstants.certFilename.key_https:
                case LocalConstants.certFilename.key_smtp:
                case LocalConstants.certFilename.sqlclientkey:
                case LocalConstants.certFilename.nosqlclientkey:
                    jsonData.key = CertFilename;
                    break;

                case LocalConstants.certFilename.cacert_tlsclient:
                    jsonData.tlsclientca = CertFilename;
                    break;
                case LocalConstants.certFilename.client_tlsclient:
                    jsonData.tlsclientcert = CertFilename;
                    break;
                case LocalConstants.certFilename.key_tlsclient:
                    jsonData.tlsclientkey = CertFilename;
                    break;
                case LocalConstants.certFilename.cacert_tlsserver:
                    jsonData.tlsserverca = CertFilename;
                    break;
                case LocalConstants.certFilename.cert_tlsserver:
                    jsonData.tlsservercert = CertFilename;
                    break;
                case LocalConstants.certFilename.key_tlsserver:
                    jsonData.tlsserverkey = CertFilename;
                    break;

                case LocalConstants.LOG_FORMAT_FILE_NAME:
                    jsonData.user_formatfile = CertFilename;
                    jsonData.formatfile = CertFilename;
                    break;

                case LocalConstants.SCRIPT_UPLOAD_NAME:
                    jsonData.script_file = CertFilename;
                    break;

                default:
                    return false;
            }

            //Write the Config file
            fs.writeFileSync(confFilename, JSON.stringify(jsonData));
            return true;
        }
        catch (err) 
        {
            LocalConstants.DebugLog(`${LocalConstants.WRITE_CONFIG_FAILED}: ${err.message}`);
            return false;
        }
    }

    /***************************************************************************/
    /**
     * @brief writeContentToFile - Writes the content which was posted into a
     * specified file path. If a file exists it will be overwritten otherwise
     * a new file will be created
     * 
     * @param content: The content which was posted from the user
     ****************************************************************************/
    writeContentToFile(content)
    {
        var filename = content.filename;
        var filecontent = content.filecontent;

        //Constants for asynchronous tasks
        var LocalConstants = this.Constants;
        this.createFolderIfNotExists(path.join(LocalConstants.FULL_APP_PATH, "node", LocalConstants.configSubfolders.script));
        
        //Currently only script uses this functionality
        return new Promise((res, rej) => {
            try 
            {
                switch (filename) 
                {
                    case LocalConstants.SCRIPT_FILE_NAME:
                        fs.writeFile(path.join(LocalConstants.FULL_APP_PATH, "node", LocalConstants.configSubfolders.script, LocalConstants.SCRIPT_FILE_NAME), filecontent, err =>
                        {
                            if (err)
                            {
                                //Show Debug message
                                LocalConstants.DebugLog(`${LocalConstants.WRITE_SCRIPT_FILE_FAILED}: ${err}`);
                                rej();
                            }
                            res();
                        })
                        break

                    default:
                        rej();
                }
            }
            catch (ex)
            {
                LocalConstants.DebugLog(`${LocalConstants.WRITE_SCRIPT_FILE_FAILED}: ${ex.message}`);
                rej();
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief createBackup - Create a zip file with all setting files as backup
     * 
     * @param content: The content which was posted from the user
     ****************************************************************************/
    createBackup(pw)
    {
        var LocalModule = this;
        return new Promise((res, rej) =>
        {
            try 
            {
                //user has set pw but the key was not read! return error
                if (pw && pw !== "" && LocalModule.Constants.pwkey === "") {
                    return rej(LocalModule.Constants.KEY_READ_ERROR);
                }

                var zip = new JSZip();
                //Constants for asynchronous tasks                
                LocalModule.Constants.DebugLog(LocalModule.Constants.BACKUP_REQ);
                let filepath = path.join(__dirname, LocalModule.Constants.configFolder);
                glob(filepath + '/**/*', { nodir: true }, function (err, files) {
                    if (err) {
                        rej(err);
                        return;
                    }
                    let filesRelative = files.map(function (match) {
                        return path.relative(filepath, match);
                    })

                    for (let i = 0; i < files.length; i++) {
                        let fileContent = fs.readFileSync(files[i]);
                        let fileName = files[i].substring(files[i].lastIndexOf("/") + 1);
                        if (LocalModule.Constants.configFilesWithPW.includes(fileName.toLowerCase())) {
                            let newContent = JSON.parse(fileContent);
                            //SQL client password property is different to others
                            try {
                                if (!LocalModule.Constants.isCustomized) {
                                    if (fileName.toLowerCase() === LocalModule.Constants.sqlConfigFile || fileName.toLowerCase() === LocalModule.Constants.nosqlConfigFile) {
                                        if (newContent.pass !== "") {
                                            let changePass = LocalModule.decryptPW(newContent.pass);
                                            changePass = btoa(changePass);
                                            newContent.pass = changePass;
                                        }
                                    }
                                    else {
                                        if (newContent.server_password !== "") {
                                            let changePass = LocalModule.decryptPW(newContent.server_password);
                                            changePass = btoa(changePass);
                                            newContent.server_password = changePass;
                                        }
                                    }
                                }
                                else {
                                    if (fileName.toLowerCase() === LocalModule.Constants.mqttConfigFile) {
                                        if (newContent.broker_password !== "") {
                                            let changePass = LocalModule.decryptPW(newContent.broker_password);
                                            changePass = btoa(changePass);
                                            newContent.pass = changePass;
                                        }
                                    }
                                }
                                
                            }
                            catch (ex) {
                                console.log(ex.message)
                            }

                            zip.file(filesRelative[i], JSON.stringify(newContent));
                        }
                        else {
                            zip.file(filesRelative[i], fileContent);
                        }
                        
                    }

                    zip.generateAsync({ type: "base64" }).then(content =>
                    {
                        let crc = Math.abs(crc32.str(content)).toString(16);
                        //Add =crc32 to backup file
                        let resultValue = `${content}=${crc.toString()}`;

                        if (pw && pw !== "") {
                            let fullpw = LocalModule.fillWithChars(pw);
                            let iv = crypto.randomBytes(16);
                            resultValue = LocalModule.encryptPWWithKeyAndIV(resultValue, fullpw, iv);
                            resultValue += iv.toString("hex");
                        }
                        res(resultValue);
                    });
                })

            }
            catch (ex) 
            {
                //Show Debug message
                LocalModule.Constants.DebugLog(`${LocalModule.Constants.BACKUP_FAILED}: ${ex.message}`);
                return rej(ex.message);
            }
        })
    }

    
    fillWithChars(value) {
        let result = value;
        if (value.length < this.Constants.KEY_LEN) {
            let diff = this.Constants.KEY_LEN - value.length;

            for (let i = 0; i < diff; i++) {
                result += "*";
            }
        }
        return result;
    }

    /***************************************************************************/
    /**
     * @brief createPLCBackup - Create a backup file from PLC
     ****************************************************************************/
    createPLCBackup()
    {
        let lastSendCmd = "";
        let resultString = "";
        let plcType = "";
        let endCut = false;
        //Constants for asynchronous tasks
        var LocalConstants = this.Constants;
        let target = LOCALTEST ? LocalConstants.REMOTE_IP : LocalConstants.LOCAL_HOST;
        var client = new net.Socket();

        //Multi frame packages are sometimes split into multiple frames -> handle it with timeout
        client.setTimeout(LocalConstants.Timeout_Status_request, () => client.destroy());
        return new Promise((res, rej) =>
        {
            client.connect(LocalConstants.defaultPort, target, function () 
            {
                lastSendCmd = LocalConstants.MEWCommands.MewtocolStatus;
                //Show Debug message
                LocalConstants.DebugLog(`${LocalConstants.SEND_MEW_REQ}: ${LocalConstants.MEWCommands.MewtocolStatus}`);
                client.write(LocalConstants.MEWCommands.MewtocolStatus);
            });
            
            //Attention!! Answers are sometimes split into multiple frames
            client.on("data", function (data)
            {
                let response = data.toString();
                //Check plc response for status request
                if (lastSendCmd === LocalConstants.MEWCommands.MewtocolStatus)
                {
                    //General response ok?
                    if (response[3] !== "$")
                    {
                        client.destroy();
                        rej(LocalConstants.MEW_RESPONSE_ERR);
                        return;
                    }
                    plcType = response.substr(14, 2);
                    //PLC type supported?
                    if (!LocalConstants.plcTypes.includes(plcType))
                    {
                        client.destroy();
                        rej(LocalConstants.MEW_RESPONSE_ERR + response);
                        return;
                    }
                    lastSendCmd = LocalConstants.MEWCommands.MewUpload;
                    client.write(LocalConstants.MEWCommands.MewUpload);
                    return;
                }

                //Continue with multiframe upload, first response of the start upload command consist of some
                //header bytes!! -> <EE$01
                if (lastSendCmd === LocalConstants.MEWCommands.MewUpload)
                {
                    //First response in one frame
                    if (response.indexOf("&") !== -1 && response.indexOf("$") !== -1)
                    {
                        //Extract the content (ending has 2bytes of crc, 1byte of the dollar sign and 1byte CR)
                        //First response contains also a header of 4bytes (steps?)
                        resultString += response.slice(10, response.length - 4);
                        lastSendCmd = LocalConstants.MEWCommands.MultiframeResponse;
                        client.write(LocalConstants.MEWCommands.MultiframeResponse);
                        return;
                    }

                    //First frame of the splitted response
                    if (response.indexOf("$") !== -1 && response.indexOf("&") === -1)
                    {
                        resultString += response.slice(10, response.length);
                        return;
                    }

                    //last frame of the splitted response
                    if (response.indexOf("$") === -1 && response.indexOf("&") !== -1)
                    {
                        resultString += response.slice(0, response.length - 4);
                        lastSendCmd = LocalConstants.MEWCommands.MultiframeResponse;
                        client.write(LocalConstants.MEWCommands.MultiframeResponse);
                        return;
                    }
                }

                //A continueing multiframe message (can be splitted)
                if (lastSendCmd === LocalConstants.MEWCommands.MultiframeResponse)
                {
                    //Check if the multiframe char is available
                    if (response.indexOf("&") !== -1)
                    {
                        //Check if it is a split frame
                        if (response.indexOf("<EE") !== -1)
                        {
                            //Ending frame -> don't cut the header, only cut the end
                            resultString += response.slice(3, response.length - 4);
                        }
                        else
                        {
                            resultString += response.slice(0, response.length - 4);
                        }
                        //continue
                        client.write(LocalConstants.MEWCommands.MultiframeResponse);
                        return;
                    }
                    //Last message or split message
                    else
                    {
                        //No multiframe char and no header -> split frame, no crc and CR -> dont cut end
                        if (response.indexOf("<EE") !== -1)
                        {
                            //Can't determine if its the last message or not since it has the same conditions
                            resultString += response.slice(3, response.length);
                        }
                        //if the last message is split then this is definately the last frame
                        else
                        {
                            resultString += response.slice(0, response.length - 3);
                            endCut = true;
                        }
                        return;
                    }
                }
            });

            client.on("close", function ()
            {
                if (resultString.length === 0)
                {
                    rej(LocalConstants.TIMEOUT);
                    return;
                }
                //Make sure that the last message has been cut crc and CR
                if (!endCut)
                {
                    resultString = resultString.slice(0, resultString.length - 3);
                }

                //Add plc type to code
                resultString += plcType;
                resultString += crypto.createHash('md5').update(resultString).digest("hex").toUpperCase();
                res(resultString);
            });

            client.on("error", function ()
            {
                rej("error");
                client.destroy();
            });
        })
    }

    /***************************************************************************/
    /**
     * @brief updatePLCProgram - Update the plc program
     ****************************************************************************/
    updatePLCProgram(content)
    {
        var LocalConstants = this.Constants;
        let plcCodeIncludeMd5 = content.plcCode;
        let plcType = "";
        let lastSendCmd = "";
        var bccCalc = this.bcc;
        var client = new net.Socket();
        
        return new Promise((res, rej) =>
        {
            //First check if the code is not empty
            if (!plcCodeIncludeMd5 || plcCodeIncludeMd5.length === 0)
            {
                rej(LocalConstants.MISSING_PARAM);
                return;
            }

            //Extract the md5 (structure is: PLC data -> 2bytes plcType, 32bytes md5)
            let plcCode = plcCodeIncludeMd5.slice(0, plcCodeIncludeMd5.length - 34);
            let md5 = plcCodeIncludeMd5.slice(plcCodeIncludeMd5.length - 32);
            plcType = plcCodeIncludeMd5.slice(plcCodeIncludeMd5.length - 34, plcCodeIncludeMd5.length - 32);

            if (crypto.createHash('md5').update(plcCode + plcType).digest("hex").toUpperCase() !== md5)
            {
                //MD5 check failed
                rej(LocalConstants.MD5_CHECK_FAILED);
                return;
            }

            client.setTimeout(LocalConstants.Timeout_Status_request, () =>
            {
                client.destroy();
                rej(LocalConstants.TIMEOUT);
            });
            //seems like everything is ok
            if (LOCALTEST)
            {
                client.connect(LocalConstants.defaultPort, LocalConstants.REMOTE_IP, function ()
                {
                    lastSendCmd = LocalConstants.MEWCommands.MewtocolStatus;
                    client.write(LocalConstants.MEWCommands.MewtocolStatus);
                });
            }
            else 
            {
                client.connect(LocalConstants.defaultPort, LocalConstants.LOCAL_HOST, function () 
                {
                    lastSendCmd = LocalConstants.MEWCommands.MewtocolStatus;
                    //Show Debug message
                    LocalConstants.DebugLog(`${LocalConstants.SEND_MEW_REQ}: ${LocalConstants.MEWCommands.MewtocolStatus}`);
                    client.write(LocalConstants.MEWCommands.MewtocolStatus);
                });
            }

            client.on("data", function (data)
            {
                let response = data.toString();
                //Check plc response for status request
                if (lastSendCmd === LocalConstants.MEWCommands.MewtocolStatus)
                {
                    //General response ok?
                    if (response[3] !== "$")
                    {
                        client.destroy();
                        rej(LocalConstants.MEW_RESPONSE_ERR);
                        return;
                    }
                    let currentPLCType = response.substr(14, 2);
                    //Is it the same plc type as the project??
                    if (currentPLCType !== plcType)
                    {
                        client.destroy();
                        rej(LocalConstants.PLC_TYPE_NOT_MATCH);
                        return;
                    }

                    //Now set the PLC into Prog mode
                    lastSendCmd = LocalConstants.MEWCommands.MewSetToProg;
                    client.write(LocalConstants.MEWCommands.MewSetToProg);
                    return;
                }

                //Is the plc in prog mode?
                if (lastSendCmd === LocalConstants.MEWCommands.MewSetToProg)
                {
                    //General response ok?
                    if (response[3] !== "$")
                    {
                        client.destroy();
                        rej(LocalConstants.SET_PROG_FAILED);
                        return;
                    }

                    //Now start the download
                    lastSendCmd = LocalConstants.MEWCommands.MewStartDownload;
                    //The data package can have a max. size of 2000 bytes. get the first 2k bytes
                    
                    let writedata = plcCode.length > LocalConstants.MAX_FRAME_DATA_SIZE ?
                        `${LocalConstants.MEWCommands.MewStartDownload}${plcCode.slice(0, 2000)}${bccCalc(LocalConstants.MEWCommands.MewStartDownload + plcCode.slice(0, 2000)).toUpperCase()}&\r` :
                        `${LocalConstants.MEWCommands.MewStartDownload}${plcCode}${bccCalc(LocalConstants.MEWCommands.MewStartDownload + plcCode).toUpperCase()}\r`;
                    plcCode = plcCode.slice(LocalConstants.MAX_FRAME_DATA_SIZE);
                    client.write(writedata);
                    return;
                }

                //Download response
                if (lastSendCmd === LocalConstants.MEWCommands.MewStartDownload)
                {
                    //Response ok then continue send frames
                    if (response === LocalConstants.MEWCommands.MultiframeResponse || response.includes(LocalConstants.MEWCommands.MewDownloadLastResponse))
                    {
                        //Still data to send -> continue
                        if (plcCode.length > 0)
                        {
                            let writedata = plcCode.length > LocalConstants.MAX_FRAME_DATA_SIZE ?
                                `<EE${plcCode.slice(0, 2000)}${bccCalc("<EE" + plcCode.slice(0, 2000)).toUpperCase()}&\r` : `<EE${plcCode}${bccCalc("<EE" + plcCode).toUpperCase()}\r`;
                            plcCode = plcCode.slice(LocalConstants.MAX_FRAME_DATA_SIZE);
                            client.write(writedata);
                            return;
                        }
                        else
                        {
                            //All data send -> Now set the PLC back to run mode
                            lastSendCmd = LocalConstants.MEWCommands.MewSetToRun;
                            client.write(LocalConstants.MEWCommands.MewSetToRun);
                            return;
                        }
                    }
                    //Received something wrong
                    else
                    {
                        //Set PLC back to run mode
                        client.destroy();
                        rej(LocalConstants.MEW_RESPONSE_ERR);
                        return;
                    }
                }

                if (lastSendCmd === LocalConstants.MEWCommands.MewSetToRun)
                {
                    //General response ok?
                    if (response[3] !== "$")
                    {
                        client.destroy();
                        rej(LocalConstants.SET_RUN_FAILED);
                        return;
                    }

                    client.destroy();
                    res(LocalConstants.UPDATE_SUCCESS);
                }
            })

            client.on("error", function ()
            {
                rej("error");
                client.destroy();
            });
        })
    }

    /***************************************************************************/
    /**
     * @brief setPLCToDefault - Set the plc program to default
     ****************************************************************************/
    setPLCToDefault()
    {
        var LocalModule = this;

        return new Promise((res, rej) =>
        {
            //try to read the default pdc.plc file
            try
            {
                fs.readFile(path.join(__dirname, "/../", LocalModule.Constants.PDC_PLC_FILE), function (err, data) 
                {
                    if (err) 
                    {
                        LocalModule.Constants.DebugLog(err.message);
                        rej(err.message);
                        return;
                    }

                    LocalModule.updatePLCProgram({ plcCode: data.toString()}).then(result =>
                    {
                        res(result);
                    }).catch(error =>
                    {
                        rej(error);
                    })
                })
            }
            catch (ex)
            {
                rej(ex);
                return;
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief restoreSettings - Restore all settings with a backup file
     * 
     * @param filecontent: The backup file content
     ****************************************************************************/
    restoreSettings(content)
    {
        let filecontent = content.restorefile;
        let restorePW = content.restorePW;
        var LocalModule = this;
       
        return new Promise((res, rej) =>
        {
            try 
            {
                //Check first if password is available
                if (restorePW && restorePW !== "") {
                    restorePW = LocalModule.fillWithChars(restorePW);
                    let iv = filecontent.substring(filecontent.length - 32);
                    filecontent = filecontent.substring(0, filecontent.length - 32);
                    let zipContent = LocalModule.decryptPWWithKeyAndIV(filecontent, restorePW, iv);
                    filecontent = zipContent;
                }

                //Extract the CRC32 first
                var pos = filecontent.lastIndexOf("=") + 1;
                //The crc starts after "="
                var crc = filecontent.substr(pos, filecontent.length - pos);
                //Don't add the "=" char
                var content = filecontent.substr(0, pos - 1);

                var readzip = new JSZip();

                readzip.loadAsync(content, { base64: true, checkCRC32: true }).then(function (zip) 
                {
                    var crcCalc = Math.abs(crc32.str(content)).toString(16);
                    //Check the CRC32
                    if (crcCalc !== crc) 
                    {
                        //Show Debug message
                        LocalModule.Constants.DebugLog(LocalModule.Constants.CRC_FAILED);
                        return rej(LocalModule.Constants.FILE_CHECK_FAILED);
                    }

                    //Delete all files from subfolders like certificates, keys, script files etc.
                    exec(`rm -r ${path.join(__dirname, LocalModule.Constants.configFolder)}*`, () =>
                    {
                        //unzip the backup file and restore the settings
                        Object.keys(zip.files).forEach(filename =>
                        {
                            try 
                            {
                                let fileContent;
                                var file = filename;
                                var isSubFolder = file.substring(filename.length - 1) === "/" ? true : false;

                                //Check if it is a foldername, if so dont do anything (our node version doesnt support "continue" )
                                if (!isSubFolder)
                                {
                                    fileContent = typeof (zip.files[filename]._data.compressedContent) === LocalModule.Constants.UNDEFINED ? LocalModule.Constants.EMPTY_STRING : LocalModule.Utf8ArrayToStr(zip.files[filename]._data.compressedContent);
                                    try {
                                        //Check and encrypt pw if needed
                                        if (!LocalModule.Constants.isCustomized) {
                                            if (LocalModule.Constants.configFilesWithPW.includes(filename.toLowerCase())) {
                                                if (filename.toLowerCase() === LocalModule.Constants.sqlConfigFile || filename.toLowerCase() === LocalModule.Constants.nosqlConfigFile) {
                                                    let newContent = JSON.parse(fileContent);
                                                    if (newContent.pass !== "") {
                                                        let entryptedPW = LocalModule.encryptPW(atob(newContent.pass));
                                                        newContent.pass = entryptedPW;
                                                        fileContent = JSON.stringify(newContent);
                                                    }
                                                }
                                                else {
                                                    let newContent = JSON.parse(fileContent);
                                                    if (newContent.server_password !== "") {
                                                        let entryptedPW = LocalModule.encryptPW(atob(newContent.server_password));
                                                        newContent.server_password = entryptedPW;
                                                        fileContent = JSON.stringify(newContent);
                                                    }
                                                }
                                            }
                                        }
                                        else {
                                            if (filename.toLowerCase() === LocalModule.Constants.mqttConfigFile) {
                                                let newContent = JSON.parse(fileContent);
                                                if (newContent.broker_password !== "") {
                                                    let entryptedPW = LocalModule.encryptPW(atob(newContent.broker_password));
                                                    newContent.broker_password = entryptedPW;
                                                    fileContent = JSON.stringify(newContent);
                                                }
                                            }
                                        }
                                    }
                                    catch (ex) {
                                        console.log(ex.message);
                                    }

                                    //Check if the file is in a sub directory. If so we need to check if the directory exists, if not create it.
                                    if (filename.indexOf("\\") !== -1) 
                                    {
                                        var subdirFile = filename.split("\\");
                                        var subsubdir = LocalModule.Constants.configFolder;
                                        file = path.join(LocalModule.Constants.configFolder, file);
                                        //In case the subdirectory has more subfolders.....
                                        for (let i = 0; i < subdirFile.length - 1; i++) 
                                        {
                                            subsubdir += i !== subdirFile.length - 1 ? `${subdirFile[i]}\\` : subdirFile[i];
                                            LocalModule.createFolderIfNotExists(subsubdir);                                            
                                        }

                                        fs.writeFileSync(file, fileContent);
                                    }
                                    else
                                    {
                                        fs.writeFileSync(path.join(LocalModule.Constants.configFolder, file), fileContent);
                                    }
                                }
                                else
                                {
                                    //Check if folder already exists
                                    let subFolderName = file.substr(0, file.length - 1);
                                    LocalModule.createFolderIfNotExists(path.join(LocalModule.Constants.configFolder, subFolderName));               
                                }
                            }
                            catch (ex)
                            {
                                //Show Debug message
                                LocalModule.Constants.DebugLog(`${LocalModule.Constants.RESTORE_UNZIP_FAILED}: ${ex.message}`);
                                return rej(ex.message);
                            }
                        })

                        //Check if any config files is missing, if so copy from default folder
                        LocalModule.copyDefaultConfigIfNoConfigExists();
                        //We also need to shut down all nodejs apps which are active
                        LocalModule.restartApplications();
                        
                        //Everything ok -> send a message to pew_main so that it can reload all the settings
                        res(LocalModule.Constants.OPERATION_SUCCESSFUL);
                        LocalModule.sendSIGToEmbeddedApp(LocalModule.Constants.PEWMAIN);
                    });
                })
            }
            catch (ex) 
            {
                //Show Debug message
                LocalModule.Constants.DebugLog(`${LocalModule.Constants.RESTORE_UNZIP_FAILED}: ${ex.message}`);
                if (ex.code === "ERR_OSSL_EVP_BAD_DECRYPT") {
                    rej(ex.code);
                }
                else
                    rej(ex.message);
            }
        })        
    }

    /***************************************************************************/
    /**
     * @brief restartApplications - In some cases the nodejs applications (e.g. api_server, sqlClient etc.)
     * must be restarted. For example when the user restores all applications settings
     * or if the user resets application settings
     ****************************************************************************/
    restartApplications() {
        var LocalModule = this;
        let killProcceses = [];
        try {
            for (let processName of LocalModule.Constants.apps) {
                killProcceses.push(LocalModule.killProcessByName(processName));
            }

            Promise.all(killProcceses).catch().finally(() => {
                LocalModule.autoStartApps();
            })
        }
        catch{
            //Nothing to do here
        }
    }

    /***************************************************************************/
    /**
     * @brief copyDefaultConfigIfNoConfigExists - After restoring a backup file
     * it could happen that the backup file has older version than the current 
     * Firmware. If the current Firmware has new config files, these wont exist in
     * the backup file. In this case we need to bring the default file to the 
     * system.
     ****************************************************************************/
    copyDefaultConfigIfNoConfigExists() {
        try {
            for (let config of this.Constants.configFiles) {
                if (!fs.existsSync(path.join(__dirname, this.Constants.configFolder, config))){
                    fs.copyFileSync(path.join(__dirname, "../", this.Constants.configDefault, config), path.join(__dirname, this.Constants.configFolder, config));
                }
            }
        }
        catch {
            //Nothing needed to do here
        }
    }

    /***************************************************************************/
    /**
     * @brief plcCommunicationTest - Test if the PLC communication is established
     * 
     * @param postcontent: content of the request
     * @param returns a string with the response of PLC in the first line. The second line is the time needed in ms
     ****************************************************************************/
    plcCommunicationTest(postcontent)
    {
        var client = new net.Socket();
        var port = postcontent.port;
        var station = postcontent.PLCStation;
        var statusCommand;
        //Remember start trigger time
        var startTime = Date.now();
        var endTime;
        //Constants for asynchronous tasks
        var LocalModule = this;
        let target = LOCALTEST ? LocalModule.Constants.REMOTE_IP : LocalModule.Constants.LOCAL_HOST;        
        var setTime;
        var DataReceived = false;

        return new Promise((res, rej) =>
        {
            try 
            {
                //Debug information
                LocalModule.Constants.DebugLog(`${LocalModule.Constants.PLC_TEST_REQ}: ${port.toString()}`);

                client.setTimeout(LocalModule.Constants.Timeout_Status_request, () =>
                {
                    LocalModule.Constants.DebugLog(LocalModule.Constants.TIMEOUT);
                    client.end();
                    rej(LocalModule.Constants.TIMEOUT);
                });

                client.connect(port, target, function () 
                {
                    statusCommand = `%${LocalModule.getStationNumberAsValidStringMew(station)}#EX00RT00**\r`;
                    //Show Debug message
                    LocalModule.Constants.DebugLog(`${LocalModule.Constants.SEND_MEW_REQ}: ${statusCommand}`);
                    client.write(statusCommand);
                });                

                client.on('data', function (data)
                {
                    var content = data;

                    //Status Error message looks like this %EE!4202. Check the position of !
                    if (content.toString().indexOf("!") === LocalModule.Constants.Error_Position_MewStatus)
                    {
                        //Reset the start time for the second try with Mew7 Status
                        startTime = Date.now();
                        client.destroy();
                        client.connect(port, target, () =>
                        {
                            statusCommand = `>@${LocalModule.getStationNumberAsValidStringMew7(station)}00#30STRD00${LocalModule.Constants.CRCMew7[station]}\r`;
                            //Show Debug message
                            LocalModule.Constants.DebugLog(`${LocalModule.Constants.SEND_MEW7_REQ}: ${statusCommand}`);
                            client.write(statusCommand);
                        });                        
                    }
                    else 
                    {
                        DataReceived = true;

                        if (content.toString().indexOf("$") !== LocalModule.Constants.OK_Position_MewStatus && content.toString().indexOf("$") !== LocalModule.Constants.OK_Position_Mew7Status)
                        {
                            client.destroy(); // kill client after server's response
                            return rej(LocalModule.Constants.FAILED);
                        }

                        endTime = Date.now();
                        var diff = endTime - startTime;
                        var ReturnString = `${content}\n${diff.toString()}`;

                        res(ReturnString);
                        client.destroy(); // kill client after server's response
                    }
                });

                client.on('error', err =>
                {
                    //Show Debug message
                    LocalModule.Constants.DebugLog(`${LocalModule.Constants.PLC_COM_ERR}: ${err}`);
                    rej(err.message);
                });

                client.on('close', () =>
                {
                    LocalModule.Constants.DebugLog('Connection closed');

                    if (!DataReceived) 
                    {
                        setTime = setTimeout(() =>
                        {
                            rej(LocalModule.Constants.NO_REPONSE)
                        }, LocalModule.Constants.Timeout_Status_request)
                    }
                    else 
                    {
                        typeof setTime !== LocalModule.Constants.UNDEFINED && clearTimeout(setTime);
                    }
                });
            }
            catch (ex) 
            {
                //returns true for error and a error message
                LocalModule.Constants.DebugLog(ex.message)
                rej(ex.message);
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief getVersion - returns the current version
     ****************************************************************************/
    getVersion()
    {
        var LocalConstants = this.Constants;
        return new Promise((res, rej) =>
        {
            try 
            {
                fs.readFile(`${__dirname}/../${LocalConstants.VERSION_FILE}`, (err, data) =>
                {
                    if (err)
                    {
                        LocalConstants.DebugLog(`${LocalConstants.VER_READ_FAILED}: ${err}`);
                        return rej(err);
                    }
                    res(data);
                });
            }
            catch (ex) 
            {
                //returns true for error and a error message
                LocalConstants.DebugLog(`${LocalConstants.VER_READ_FAILED}: ${ex.message}`);
                rej(ex.message);
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief getSysLogs - returns the last 100 lines of syslog
     ****************************************************************************/
    getSysLogs() {
        return new Promise((res, rej) => {
            try {
                let logs = execSync("tail -100 /var/log/messages");
                res(logs.toString());
            }
            catch (ex) {
                rej(ex);
           }       
        })                
    }

    /***************************************************************************/
    /**
     * @brief getDefaultScript - Read the default script file and overwrite it to the user's file
     ****************************************************************************/
    getDefaultScript()
    {
        var LocalConstants = this.Constants;
        var LocalModule = this;

        return new Promise((res, rej) =>
        {
            try 
            {
                LocalConstants.DebugLog(LocalConstants.LOAD_DEFAULT_SCRIPT_REQ);

                LocalModule.createFolderIfNotExists(`${path.join(LocalConstants.FULL_APP_PATH, "node", LocalConstants.configSubfolders.script)}`);
                exec(`cp ${path.join(LocalConstants.FULL_APP_PATH, LocalConstants.configDefaultFolder, LocalConstants.SCRIPT_FILE_NAME) + " " + path.join(LocalConstants.FULL_APP_PATH, "node", LocalConstants.configSubfolders.script)}`, cpErr =>
                {
                    if (cpErr)
                    {
                        return rej(cpErr);
                    }

                    if (LocalModule.rewriteConfFile(`${LocalModule.Constants.configFolder}${LocalModule.Constants.requests.READ_SCRIPT}${LocalModule.Constants.JSONFILE}`, LocalModule.Constants.SCRIPT_UPLOAD_NAME, LocalModule.Constants.SCRIPT_FILE_NAME)) 
                    {
                        return rej(LocalConstants.REWRITE_FAILED);
                    }
                    res(LocalModule.Constants.SUCCESS);
                    LocalModule.sendSIGToEmbeddedApp(LocalConstants.PEWMAIN);
                })
            }
            catch (ex)
            {
                //returns true for error and a error message
                LocalModule.Constants.DebugLog(`${LocalConstants.LOAD_DEFAULT_SCRIPT_FAILED}: ${ex.message}`);
                rej(ex.message);
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief updateFirmware - Update the firmware of pew
     ****************************************************************************/
    updateFirmware()
    {
        var LocalModule = this;
        var LocalConstants = this.Constants;
        LocalConstants.DebugLog(LocalConstants.FIRMWARE_UPDATE_REQ);

        return new Promise((res, rej) =>
        {
            //First check if both files are available
            if (!fs.existsSync(path.join(__dirname, LocalConstants.tempFolder, LocalConstants.firmwareFixedNames.package)) || !fs.existsSync(path.join(__dirname, LocalConstants.tempFolder, LocalConstants.firmwareFixedNames.md5)))
            {
                LocalConstants.DebugLog(LocalConstants.FIRMWARE_FILE_NOT_FOUND);
                return rej(LocalConstants.FILE_MISSING);
            }

            var hash = md5File.sync(path.join(__dirname, LocalConstants.tempFolder, LocalConstants.firmwareFixedNames.package));
            //Read the hash file from the md5 file
            var hashfile = fs.readFileSync(path.join(__dirname, LocalConstants.tempFolder, LocalConstants.firmwareFixedNames.md5)).toString();

            if (hashfile.indexOf('\n') !== -1)
            {
                //In case of carriage return, delete the character
                hashfile = hashfile.slice(0, hashfile.indexOf('\n'));
            }

            if (hashfile !== hash)
            {
                LocalConstants.DebugLog(LocalConstants.MD5_CHECK_FAILED);
                return rej(LocalConstants.MD5_CHECK_FAILED);
            }

            LocalConstants.DebugLog(LocalConstants.START_FIRMWARE_UPDATE);
            res();
            //Check if folder exists, if so delete it first
            rimraf(LocalConstants.configTempFolder, { disableGlob: true }, () =>
            {
                //Sometimes the .installrt folder doesnt exists. 
                LocalModule.createFolderIfNotExists(LocalConstants.configTempFolderBefore);
                LocalModule.createFolderIfNotExists(LocalConstants.configTempFolder);
                //Copy all config files to temp folder
                ncp(path.join(__dirname, LocalConstants.configFolder), LocalConstants.configTempFolder, () =>
                {
                    execSync(`cp ${path.join(__dirname, "../", LocalConstants.configDefault, "pew_version.json")} ${path.join(LocalConstants.configTempFolder,"pew_version.json")}`)
                    //Start the firmware update process
                    exec(`dbus-send --print-reply --system --dest=com.exor.JMLauncher '/' com.exor.JMLauncher.install string:'${path.join(__dirname, LocalConstants.tempFolder, LocalConstants.firmwareFixedNames.package)}' boolean:false int32:-1`);
                });
            });
        })
    }

    /***************************************************************************/
    /**
     * @brief updateStatus - get Update status
     ****************************************************************************/
    updateStatus()
    {
        return new Promise((res, rej) =>
        {
            if (LOCALTEST)
            {
                setTimeout(() =>
                {
                    res();
                }, 10000)
            }
            else
            {
                var command = "dbus-monitor --system type='signal',interface=com.exor.JMLauncher,member=installationFailed & dbus-monitor --system type='signal',interface=com.exor.JMLauncher,member=installationSuccess";
                var timeout = setTimeout(() =>
                {
                    rej();
                }, this.Constants.TIMEOUT_GETUPDATE_STATUS)
                var monitor = exec(command);
                monitor.stdout.on('data', data =>
                {
                    if (data.search("installationSuccess") !== -1)
                    {
                        clearTimeout(timeout);
                        res();
                    }

                    if (data.search("installationFailed") !== -1)
                    {
                        clearTimeout(timeout);
                        rej();
                    }
                })
            }
        })
    }

    /***************************************************************************/
    /**
     * @brief startApp - start any app
     ****************************************************************************/
    startApp(appNameParam) {
        exec(`${appNameParam} 2>&1 | logger`, () => {
            /* Just start the app */
        });
    }

    /***************************************************************************/
    /**
     * @brief autoStartApps - start apps at startup
     ****************************************************************************/
    autoStartApps()
    {
        //check if rest api has to be started...
        this.getConfFile(this.Constants.requests.READ_API).then(config => {
            if (typeof config.api_enable !== this.Constants.UNDEFINED) {
                if (config.api_enable || config.http_server) {
                    this.startApp(`${this.Constants.NODE_V16_PATH} ${this.Constants.NODEJS_SERVER_FILES_PATH}${this.Constants.API_APP_PATH} ${config.http_port}`);
                    this.Constants.DebugLog(`${this.Constants.API_AUTO_START}`);
                }
            }
        }).catch(err => {
            this.Constants.DebugLog(err);
        })

        //check if tlsclient has to be started...
        this.getConfFile(this.Constants.requests.READ_TLS).then(config => {
            if (typeof config.enable_tlsclient !== this.Constants.UNDEFINED) {
                if (config.enable_tlsclient) {
                    this.startApp(`${this.Constants.NODE_V16_PATH} ${this.Constants.NODEJS_SERVER_FILES_PATH}${this.Constants.TLS_CLIENT_PATH}`);
                }
            }

            if (typeof config.enable_tlsserver) {
                if (config.enable_tlsserver) {
                    this.startApp(`${this.Constants.NODE_V16_PATH} ${this.Constants.NODEJS_SERVER_FILES_PATH}${this.Constants.TLS_SERVER_PATH}`);
                }
            }
        }).catch(err => {
            this.Constants.DebugLog(err);
        })

        //Check if sql client has to be started...
        this.getConfFile(this.Constants.requests.READ_SQL).then(config => {
            if (typeof config.enable_sql !== this.Constants.UNDEFINED) {
                if (config.enable_sql) {
                    this.startApp(`${this.Constants.NODE_V16_PATH} ${this.Constants.NODEJS_SERVER_FILES_PATH}${this.Constants.SQL_APP_PATH}`);
                }
            }
        }).catch(err => {
            this.Constants.DebugLog(err);
        })

        //Check if nosql client has to be started...
        this.getConfFile(this.Constants.requests.READ_NOSQL).then(config => {
            if (typeof config.enable_nosql !== this.Constants.UNDEFINED) {
                if (config.enable_nosql) {
                    this.startApp(`${this.Constants.NODE_V16_PATH} ${this.Constants.NODEJS_SERVER_FILES_PATH}${this.Constants.NOSQL_APP_PATH}`);
                }
            }
        }).catch(err => {
            this.Constants.DebugLog(err);
        })

        //Check if transparent client has to be started...
        this.getConfFile(this.Constants.requests.READ_INTERFACE).then(config => {
            let rs232 = config.interface.filter(el => el.id === "/RS232");
            let rs485 = config.interface.filter(el => el.id === "/RS485/RS232");
            if (rs232.length !== 0 && rs232[0].enabled === 1 && rs232[0].protocol === this.Constants.PROTOCOLS.transparent) {
                this.startApp(`${this.Constants.NODE_V16_PATH} ${this.Constants.NODEJS_SERVER_FILES_PATH}${this.Constants.RS232TRANSPARENT_APP_PATH}`);
            }

            if (rs485.length !== 0 && rs485[0].enabled === 1 && rs485[0].protocol === this.Constants.PROTOCOLS.transparent) {
                this.startApp(`${this.Constants.NODE_V16_PATH} ${this.Constants.NODEJS_SERVER_FILES_PATH}${this.Constants.RS485TRANSPARENT_APP_PATH}`);
            }
        }).catch(err => {
            this.Constants.DebugLog(err);
        })


        //Get key
        this.getKeyContent().then(data => {
            this.Constants.pwkey = data;
        }).catch(() => {
            console.log("Read key failed!");
        });
    }

    /***************************************************************************/
    /**
     * @brief ExtractRequest - Extract the user get request
     ****************************************************************************/
    extractGetRequest(req)
    {
        let LocalConstants = this.Constants;

        let result = {
            error: false,
            message: "",
            request: []
        };

        let uri = req.url;
        LocalConstants.DebugLog(`${LocalConstants.PLC_GET_REQ}: ${uri}`);

        let pos = uri.indexOf("?");
        if (pos === -1)
        {
            result.error = true;
            result.message = LocalConstants.API_REQUEST_SYNTAX_ERR;
            LocalConstants.DebugLog(`${LocalConstants.PLC_GET_FAILED}: ${result.message}`);
            return result;
        }

        var request = uri.substring(pos + 1);
        var requests = request.split("&");
        var bErr = false;

        for (let singleReq of requests)
        {
            let len = singleReq.length;
            //At least length 2 e.g. R0
            bErr = len < 2;
            if (bErr)
                break;
            
            let datatype = "";
            let isBool = false;
            for (let j = 0; j < len; j++)
            {
                if (isNaN(singleReq[j]) && singleReq[j] !== "=") {
                    datatype += singleReq[j].toUpperCase();
                    //In case it's a bool the address could be also a non number like RF or YA
                    //For the special case L, it could be a Bool, but also could be a register if its LD
                    if (datatype === 'L') {
                        isBool = this.isBoolArea(datatype, singleReq[j+1].toUpperCase());
                    }
                    else {
                        isBool = this.isBoolArea(datatype);
                    }

                    if (isBool)
                        break;
                }
                else {
                    break;
                }
            }

            bErr = !LocalConstants.allowedDataAreas.includes(datatype);

            if (!bErr)
            {
                let params = singleReq.substring(datatype.length);
                if (params.indexOf("=") !== -1)
                {
                    let splitParam = params.split("=");
                    bErr = splitParam.length > 2 || (!this.isStationOrInterface(datatype) && (splitParam[0] === LocalConstants.EMPTY_STRING || splitParam[1] === LocalConstants.EMPTY_STRING));
                        
                    for (let val of splitParam)
                    {
                        if (isNaN(val) && !isBool)
                        {
                            bErr = true;
                            break;
                        }
                    }

                    let regAddr = parseInt(splitParam[0]);
                    //In case of bool check if the address syntax is correct (e.g. R1A and NOT RA1!)
                    if (isBool && isNaN(regAddr))
                    {
                        bErr = !this.isValidBoolAddress(splitParam[0]);                                                
                        //Delete leading zeroes in case there are some
                        regAddr = this.removeLeadingZeroes(splitParam[0]);                        
                    }

                    let writeValue = splitParam[1];
                    if (writeValue.indexOf(".") !== -1)
                    {
                        writeValue = this.CalculateNumberFromIEEEReal(writeValue);
                    }
                    result.request.push({ type: datatype, write: true, reg_addr: regAddr, writeVal: parseInt(writeValue) });
                }
                else
                {                        
                    let regAddr = isBool ? params : parseInt(params);                        
                    bErr = isNaN(regAddr) && !isBool;
                        
                    if (!bErr)
                    {
                        //Check if the hex address has the correct syntax
                        if (isBool && isNaN(regAddr))
                        {
                            bErr = !this.isValidBoolAddress(regAddr);
                            //Delete leading zeroes in case there are some
                            regAddr = this.removeLeadingZeroes(regAddr);
                        }

                        result.request.push({ type: datatype, write: false, reg_addr: regAddr });
                    }
                }
            }            
        }

        if (bErr)
        {
            result.error = true;
            result.message = LocalConstants.API_REQUEST_SYNTAX_ERR;
            LocalConstants.DebugLog(`${LocalConstants.PLC_GET_FAILED}: ${result.message}`);
            return result;
        }

        //Request ok -> continue
        return result;
    }

    /***************************************************************************/
    /**
     * @brief checkPostPayload - Check the user post request if there are any errors
     * if not return the request as an array. Also return the protocol, port and 
     * station number to be used.
     ****************************************************************************/
    checkPostPayload(content)
    {
        let result = {
            error: false,
            message: "",
            protocol: "",
            port: 0,
            station: 0,
            request: []
        };

        let sAdditionalErrorInformation = this.Constants.EMPTY_STRING;
        let bErr = false;

        /******************************************** Error handling *************************************************/
        //Check some settings regarding protcol/ports/interface

        if (!this.isUndefined(content.interface) && !this.isInterfaceInRange(content.interface))
        {
            bErr = true;
            sAdditionalErrorInformation = this.Constants.REQ_IFACE_OR_STATION_OUT_OF_RANGE;
        }

        //Check if the requested interface is enabled
        //First find out which service should be checked since the interface number doesnt correspond to the service number
        if (!bErr)
        {
            //Second parameter determines if the default port should be used (user did not specify a interface)
            let comData = this.getComEnabled(content.interface, this.isUndefined(content.interface));
            if (!comData.enabled) {
                bErr = true;
                sAdditionalErrorInformation = this.Constants.COM_NOT_ENABLED;
            }
            result.protocol = comData.protocol;
            result.station = !this.isUndefined(content.station) ? content.station : comData.station;
            //Check if a tcp port is enabled
            let tcpPortData = this.getTCPPortOpened(comData.serviceID);
            if (!tcpPortData.enabled) {
                bErr = true;
                sAdditionalErrorInformation = this.Constants.TCP_PORT_DISABLED;
            }
            result.port = tcpPortData.port;
        }

        if (!this.isUndefined(content.station) && !this.isStationNumberValid(content.station))
        {
            bErr = true;
            sAdditionalErrorInformation = this.Constants.REQ_IFACE_OR_STATION_OUT_OF_RANGE;
        }

        //Limit the number of max. requests
        if (content.req.length > this.Constants.MAX_POST_REQUESTS)
        {
            bErr = true;
            sAdditionalErrorInformation = `${this.Constants.MAX_POST_REQUEST_EXCEEDED} (${this.Constants.MAX_POST_REQUESTS})`;
        }

        //Other checks only when there is no error yet
        if (!bErr)
        {
            for (let i = 0; i < content.req.length; i++)
            {
                let req = content.req[i];
                let area = req.area.toUpperCase();
                let type = req.type.toUpperCase();

                //Parameters missing
                if (this.isUndefined(req.area) || this.isUndefined(req.start) || this.isUndefined(req.type))
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_MISSING_PARAMS_ERR} (Request position ${i + 1})`;
                    break;
                }

                //Count must be specified except when the type is string
                if (this.isUndefined(req.count) && type !== this.Constants.datatype.string.name)
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_COUNT_MISSING} (Request position ${i + 1})`;
                    break;
                }

                //Limit the string length
                if (type === this.Constants.datatype.string.name && req.value && req.value.length > this.Constants.MAX_STRING_LEN_REQUEST)
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_STRING_LIMIT_EXCEED} at request position ${i + 1} (${this.Constants.MAX_STRING_LEN_REQUEST})`;
                    break;
                }

                //Limit maximum number to read for each request
                if (req.count > this.Constants.MAX_COUNT_PER_REQUEST)
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_COUNT_LIMIT_EXCEED} at request position ${i + 1} (${this.Constants.MAX_COUNT_PER_REQUEST})`;
                    break;
                }

                //Area and type must be in the allowed list
                if (!this.Constants.allowedPostReqTypes.includes(type) || !this.Constants.allowedPostAreas.includes(area))
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_WRONG_AREA_OR_TYPE} (Request position ${i + 1})`;
                    break;
                }

                //Area and type must be conform to each other (e.g. its not allowed to use WR area and type DINT or REAL)
                if (this.isWordArea(area) && (type === this.Constants.datatype.dint.name || type === this.Constants.datatype.udint.name || type === this.Constants.datatype.real.name || type === this.Constants.datatype.bool.name))
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_WRONG_AREA_OR_TYPE} (Request position ${i + 1})`;
                    break;
                }

                //Check if area and type are matched
                if (this.isBoolArea(area) && type !== this.Constants.datatype.bool.name)
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_WRONG_AREA_OR_TYPE} (Request position ${i + 1})`;
                    break;
                }

                if (this.isDataregisterArea(area) && type === this.Constants.datatype.bool.name)
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_WRONG_AREA_OR_TYPE} (Request position ${i + 1})`;
                    break;
                }

                //In case of type "ANY" it is only allowed for reading data
                if (type === this.Constants.datatype.any.name && typeof (req.value) !== this.Constants.UNDEFINED)
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_TYPE_MISMATCH} (Request position ${i + 1})`;
                    break;
                }

                //In case of type "ANY" bools area is not allowed!
                if (type === this.Constants.datatype.any.name && this.isBoolArea(area))
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_ANY_NOT_ALLOWED_BOOL} (Request position ${i + 1})`;
                    break;
                }

                //Check the startaddress and count
                if ((type !== this.Constants.datatype.bool.name && (req.start + req.count) > this.MAX_ADDRESS) || req.count < 0 || req.start < 0)
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.ADDR_OUT_OF_RANGE} (Request position ${i + 1})`;
                    break;
                }

                //When the type is bool then the start address is also allowed as string. 
                if (type === this.Constants.datatype.bool.name && result.protocol !== this.Constants.PROTOCOLS.modbus)
                {
                    let pattern = /[g-zG-Z\\/:äüö&%_#'\`\?=\+\-.,;"<>!~\|\{\}\(\)\[\]\^\*´]/;

                    if (typeof (req.start) === "string" && pattern.test(req.start))
                    {
                        bErr = true;
                        sAdditionalErrorInformation = `${this.Constants.ADDR_OUT_OF_RANGE} (Request position ${i + 1})`;
                        break;
                    }
                }

                //When the protocol is modbus and the type is bool then the start address must be a number
                if (type === this.Constants.datatype.bool.name && result.protocol === this.Constants.PROTOCOLS.modbus && typeof(req.start) !== "number")
                {
                    bErr = true;
                    sAdditionalErrorInformation = `${this.Constants.REQ_START_ADDR_TYPE_MISMATCH} (Request position ${i + 1})`;
                    break;
                }

                //When write data check if the number of values matches with the count. Also check if the write value fits to the given type
                if (!this.isUndefined(req.value))
                {
                    if ((!Array.isArray(req.value) && req.count > 1) || (Array.isArray(req.value) && req.value.length !== req.count))
                    {
                        bErr = true;
                        sAdditionalErrorInformation = `${this.Constants.REQ_CNT_VAL_NO_MATCH} (Request position ${i + 1})`;
                        break;
                    }

                    if ((type === this.Constants.datatype.int.name || type === this.Constants.datatype.uint.name ||
                        type === this.Constants.datatype.dint.name || type === this.Constants.datatype.udint.name))
                    {
                        if (!Array.isArray(req.value))
                        {
                            //Float values or strings/bools are not allowed here
                            if (req.value % 1 !== 0 || typeof (req.value) !== "number")
                            {
                                bErr = true;
                                sAdditionalErrorInformation = `${this.Constants.REQ_TYPE_MISMATCH} (Request position ${i + 1})`;
                                break;
                            }
                        }
                        else
                        {
                            for (let val of req.value)
                            {
                                //Float values or strings/bools are not allowed here
                                if (val % 1 !== 0 || typeof (val) !== "number")
                                {
                                    bErr = true;
                                    sAdditionalErrorInformation = `${this.Constants.REQ_TYPE_MISMATCH} (Request position ${i + 1})`;
                                    break;
                                }
                            }
                        }
                    }

                    if (Array.isArray(req.value))
                    {
                        for (let val of req.value)
                        {
                            if (this.isTypeMismatch(type, val))
                            {
                                bErr = true;
                                sAdditionalErrorInformation = `${this.Constants.REQ_TYPE_MISMATCH} (Request position ${i + 1})`;
                                break;
                            }
                        }
                    }
                    else
                    {
                        if (this.isTypeMismatch(type, req.value))
                        {
                            bErr = true;
                            sAdditionalErrorInformation = `${this.Constants.REQ_TYPE_MISMATCH} (Request position ${i + 1})`;
                            break;
                        }
                    }
                }

                if (bErr)
                    break;
            }
        }
        /****************************************** End Error handling ************************************************/

        if (bErr)
        {
            result.error = true; 
            result.message = `${this.Constants.API_REQUEST_SYNTAX_ERR}\n${sAdditionalErrorInformation}`;
            this.Constants.DebugLog(`${this.Constants.PLC_POST_FAILED}: ${result.message}`);
            return result;
        }

        //Request ok
        result.request = content.req;
        return result;
    }

    /***************************************************************************/
    /**
     * @brief plcPost - Handle the post request for PLC data (/plcpost)
     ****************************************************************************/
    plcPost(req)
    {
        var LocalModule = this;
        LocalModule.Constants.DebugLog(LocalModule.Constants.PLC_POST_REQ);
        //Check which interface and station number to use
        return new Promise((resolve, reject) =>
        {
            try
            {
                //Access right check
                let apiConfigFile = LocalModule.readConfigFileAsJSON(path.join(__dirname, LocalModule.Constants.configFolder, LocalModule.Constants.requests.READ_API + LocalModule.Constants.JSONFILE));
                let startAddr = typeof apiConfigFile.ip1 === LocalModule.Constants.UNDEFINED ? 0 : this.ip2int(apiConfigFile.ip1);
                let endAddr = typeof apiConfigFile.ip2 === LocalModule.Constants.UNDEFINED ? 0 : this.ip2int(apiConfigFile.ip2);
                let reqAddr = this.ip2int(req.ip);

                if (startAddr !== LocalModule.Constants.MIN_IP && startAddr !== LocalModule.Constants.MAX_IP && endAddr !== LocalModule.Constants.MIN_IP && endAddr !== LocalModule.Constants.MAX_IP)
                {
                    if (reqAddr < startAddr || reqAddr > endAddr) {
                        reject(LocalModule.Constants.ACCESS_DENIED);
                        return;
                    }                    
                }

                //Check if there are any contents
                let content = req.body;
                if (!content || !content.req)
                {
                    reject(LocalModule.Constants.REQ_SYNTAX_FORMAT_ERR);
                    return;
                }

                //Extract the content and check the syntax and other error handling                
                let result = LocalModule.checkPostPayload(content);

                if (result.error)
                {
                    reject(result.message);
                    return;
                }

                // Everything ok -> get the protocol to be used
                let protocol;
                if (result.interface === LocalModule.Constants.Interfaces.INTERN.interface)
                {
                    protocol = new Modbus.Modbus();
                }
                else
                {
                    switch (result.protocol)
                    {
                        case LocalModule.Constants.PROTOCOLS.mewtocol:
                            protocol = new Mew.Mewtocol();
                            break;

                        case LocalModule.Constants.PROTOCOLS.mewtocol7:
                            protocol = new Mew7.Mewtocol7();
                            break;

                        case LocalModule.Constants.PROTOCOLS.modbus:
                            protocol = new Modbus.Modbus();
                            break;

                        default:
                            protocol = new Mew.Mewtocol();
                    }
                }

                protocol.Station = result.station;

                LocalModule.startPostRequestProcess(result.request, protocol, result.port)
                    .then((res) =>
                    {
                        resolve(res);
                    })
                    .catch((err) =>
                    {
                        reject(err);
                    });
            }
            catch (ex)
            {
                LocalModule.Constants.DebugLog(`${LocalModule.Constants.PLC_POST_FAILED}: ${ex.message}`);
                reject(ex.message);
            }
        }); 
    }

    /***************************************************************************/
    /**
     * @brief startPostRequestProcess - Start the post request process
     * @param req: All requests for PLC post
     * @param protocol: the protocol object
     * @param port: the port number for the connection to the PLC
     * @return: returns an object with read data
     ****************************************************************************/
    async startPostRequestProcess(req, protocol, port)
    {
        let result = {
            err: false,
            err_msg: "",
            data: []
        };

        try
        {
            for (let item of req)
            {
                //When error occured -> go out
                if (result.err)
                    break;

                protocol.StartAddr = item.start;
                protocol.NumberOfRegs = 1;
                protocol.datatype = item.area.toUpperCase();
                let tempData;
                /*************************************************** Write request ***************************************************/
                if (!this.isUndefined(item.value))
                {
                    switch (item.type.toUpperCase())
                    {
                        case this.Constants.datatype.int.name:
                        case this.Constants.datatype.uint.name:
                            tempData = await this.writeMultipleRegisters(protocol, port, item);
                            result.data.push(tempData);
                            break;

                        case this.Constants.datatype.dint.name:
                        case this.Constants.datatype.udint.name:
                            tempData = await this.writeMultipleRegisters(protocol, port, item, this.Constants.writeDataTypes._32bit);
                            result.data.push(tempData);
                            break;

                        case this.Constants.datatype.real.name:
                            tempData = await this.writeMultipleRegisters(protocol, port, item, this.Constants.writeDataTypes._32bitReal);
                            result.data.push(tempData);
                            break;

                        case this.Constants.datatype.bool.name:
                            tempData = await this.writeMultipleBits(protocol, port, item);
                            result.data.push(tempData);
                            break;

                        case this.Constants.datatype.string.name:
                            //First read the max. write size
                            let tempVar = {};
                            tempVar.start = item.start;
                            tempVar.count = 1;
                            tempVar.type = this.Constants.datatype.int.name;
                            tempVar.area = "DT";
                            let tempRes = await this.readMultipleRegisters(protocol, port, tempVar);

                            if (!tempRes.err) {
                                item.maxWriteLength = tempRes.int[0];
                            }

                            //The starting address must be increased by 1 for the header word 
                            protocol.StartAddr++;
                            //Also increase the item startaddress since this will be used in writeMultipleRegisters
                            item.start++;
                            tempData = await this.writeMultipleRegisters(protocol, port, this.getWriteStringRegisters(item));
                            result.data.push(tempData);
                            break;
                    }
                }
                //Read request
                else
                {
                    //use lower case for type since the decoded properties are all lower case
                    let type = item.type.toLowerCase();
                    switch (type.toUpperCase())
                    {
                        /*****************************************16 bit handling and any type********************************************/
                        case this.Constants.datatype.int.name:
                        case this.Constants.datatype.uint.name:
                        case this.Constants.datatype.any.name:                            
                            tempData = await this.readMultipleRegisters(protocol, port, item);
                            result.data.push(tempData);
                            break;
                        /*****************************************32 bit handling********************************************/
                        case this.Constants.datatype.dint.name:
                        case this.Constants.datatype.udint.name:
                        case this.Constants.datatype.real.name:
                            tempData = await this.readMultipleRegisters(protocol, port, item, true);
                            result.data.push(tempData);
                            break;
                        /*****************************************Bool handling********************************************/
                        case this.Constants.datatype.bool.name:
                            tempData = await this.readMultipleBits(protocol, port, item);
                            result.data.push(tempData);
                            break;
                        /*****************************************String handling********************************************/
                        case this.Constants.datatype.string.name:
                            tempData = await this.readString(protocol, port, item);
                            result.data.push(tempData);
                            break;
                    }
                }
            }
        }
        catch (ex)
        {
            result.err = true;
            result.err_msg = ex.message;
        }

        return result;
    }

    /***************************************************************************/
    /**
     * @brief writeMultipleRegisters - Handle write many registers in a loop
     * protocol: The protocol object used
     * port: The TCP port which is used to read the data
     * item: The requested item with information about register address, data type etc.
     * datatype: 0 = 16bit (int, uint, word), 1 = 32bit (dint, udint, dword), 2 = 32bit Real
     ****************************************************************************/
    async writeMultipleRegisters(protocol, port, item, datatype = this.Constants.writeDataTypes._16bit) {
        let watchdog = false;
        let result = {};
        //Watchdog timer for the loops, avoid endless loop for any case
        let watchdogTimer = new Date().getTime();
        let totalNumberOfRegsToWrite = item.count;
        let communicationErrorCounter = 0;
        //When more registers has to be written than the max. allowed, split it into multiple requests
        let maxWriteRegs = datatype > this.Constants.writeDataTypes._16bit ? protocol.MAX_WRITE_REGS / 2 : protocol.MAX_WRITE_REGS;
        let numberOfRegsToWrite = totalNumberOfRegsToWrite > maxWriteRegs ? maxWriteRegs : totalNumberOfRegsToWrite;
        let localObj = this;
        let writeWordIndex = 0;
        protocol.StartAddr = item.start;

        //Send the request and get the respond
        return new Promise(async (res, rej) => {
            result.request = (item.area + item.start.toString()).toUpperCase();
            if (!this.valueWithinLimit(datatype > this.Constants.writeDataTypes._16bit ? this.Constants.datatype.udint.name : this.Constants.datatype.uint.name, item.value)) {
                result.err = true;
                result.err_msg = localObj.Constants.VAL_OUT_OF_RANGE;
                result.err_code = localObj.Constants.errorCodes.unknown;
                rej(result);
                return;
            }

            do {
                //Reset the write data length
                protocol.writeData.length = 0;
                if (Array.isArray(item.value)) {
                    for (let j = writeWordIndex; j < numberOfRegsToWrite + writeWordIndex; j++) {
                        if (datatype > this.Constants.writeDataTypes._16bit) {
                            let separated16BitsValues = this.SplitValueToTwoSixteenbits(datatype === this.Constants.writeDataTypes._32bitReal ? this.CalculateNumberFromIEEEReal(item.value[j]) : item.value[j]);
                            protocol.writeData.push(separated16BitsValues.first);
                            protocol.writeData.push(separated16BitsValues.second);
                        }
                        else {
                            protocol.writeData.push(item.value[j]);
                        }
                    }
                    protocol.NumberOfRegs = protocol.writeData.length;
                }
                else {
                    if (datatype > this.Constants.writeDataTypes._16bit) {
                        let separated16BitsValues = this.SplitValueToTwoSixteenbits(datatype === this.Constants.writeDataTypes._32bitReal ? this.CalculateNumberFromIEEEReal(item.value) : item.value, datatype);
                        protocol.writeData.push(separated16BitsValues.first);
                        protocol.writeData.push(separated16BitsValues.second);
                    }
                    else {
                        protocol.writeData.push(item.value);
                    }
                    protocol.NumberOfRegs = 1;
                }

                let command = protocol.GenerateWriteRegisters();
                await localObj.sendRequestToPLC(command, port).then((response) => {
                    result = protocol.WriteResponseOK(protocol.selfProtocol === "MODBUS" ? response : response.toString());
                }).catch((err) => {
                    result.err = true;
                    result.err_code = localObj.Constants.errorCodes.unknown;
                    result.err_msg = err;
                    communicationErrorCounter++;
                });

                //Check watchdog
                let loopTime = new Date().getTime();
                if (loopTime - watchdogTimer >= localObj.WATCH_DOG && !LOCALTEST) {
                    watchdog = true;
                    rej();
                }
                totalNumberOfRegsToWrite -= numberOfRegsToWrite;
                protocol.StartAddr += datatype > this.Constants.writeDataTypes._16bit ? numberOfRegsToWrite * 2 : numberOfRegsToWrite;
                writeWordIndex += numberOfRegsToWrite;
                numberOfRegsToWrite = totalNumberOfRegsToWrite > maxWriteRegs ? maxWriteRegs : totalNumberOfRegsToWrite;
            }
            while (totalNumberOfRegsToWrite > 0 && communicationErrorCounter < localObj.RETRIES && !watchdog)
            res(result);
        })
    }

    /***************************************************************************/
    /**
     * @brief writeMultipleBits - Handle write many bits in a loop
     * protocol: The protocol object used
     * port: The TCP port which is used to read the data
     * item: The requested item with information about register address, data type etc.
     ****************************************************************************/
    async writeMultipleBits(protocol, port, item)
    {
        let watchdog = false;
        let result = {};
        //Watchdog timer for the loops, avoid endless loop for any case
        let watchdogTimer = new Date().getTime();
        let totalNumberOfRegsToWrite = item.count;
        let communicationErrorCounter = 0;
        //Bits must be read one by one....
        let numberOfRegsToWrite = 1;
        let iterations = 0;
        let localObj = this;

        //Send the request and get the respond
        return new Promise(async (res, rej) =>
        {
            result.request = (item.area + item.start.toString()).toUpperCase();

            if (!this.valueWithinLimit(this.Constants.datatype.bool.name, item.value))
            {
                result.err = true;
                result.err_msg = localObj.Constants.VAL_OUT_OF_RANGE;
                result.err_code = localObj.Constants.errorCodes.unknown;
                res(result);
                return;
            }
            //Since bits must be written one by one, write bits will be limited to 50
            do
            {
                protocol.writeBit = Array.isArray(item.value) ? Number(item.value[iterations]) : Number(item.value);

                let command = protocol.GenerateWriteSingleBit();
                await localObj.sendRequestToPLC(command, port).then((response) =>
                {
                    result = protocol.WriteResponseOK(response.toString());
                }).catch((err) =>
                {
                    result.err = true;
                    result.err_code = localObj.Constants.errorCodes.unknown;
                    result.err_msg = err;
                    communicationErrorCounter++;
                });

                //Check watchdog
                let loopTime = new Date().getTime();
                if (loopTime - watchdogTimer >= localObj.WATCH_DOG && !LOCALTEST)
                {
                    watchdog = true;
                    rej();
                }
                iterations++;
                totalNumberOfRegsToWrite -= numberOfRegsToWrite;
                protocol.StartAddr = localObj.IncrementMewBoolAddress(protocol.StartAddr.toString());
            }
            while (totalNumberOfRegsToWrite > 0 && communicationErrorCounter < localObj.RETRIES && !watchdog)
            res(result);
        })
    }

    /***************************************************************************/
    /**
     * @brief getWriteStringRegisters - Converts the write string to 16bit registers in FP-String format
     * item: The incoming request item
     * Returns the item object for multipleReadRegisters
     ****************************************************************************/
    getWriteStringRegisters(item)
    {
        let result = item;
        let values = item.value;

        result.value = [];
        //Number of 16bit words to write for the string. Always add 1 for the header word
        //Each character is 1 byte and we write in words (16bits).
        let even = values.length % 2 === 0;        
        result.count = even ? Math.floor(values.length/2) + 1 : Math.floor(values.length/2) + 2;

        //Make sure to write not more than the max string length
        if (item.maxWriteLength) {
            let maxLenEven = item.maxWriteLength % 2 === 0;
            let maxLen = maxLenEven ? Math.floor(item.maxWriteLength / 2) + 1 : Math.floor(item.maxWriteLength / 2) + 2;

            values.length > item.maxWriteLength ? result.value.push(item.maxWriteLength) : result.value.push(values.length);
            if (result.count > maxLen)
                result.count = maxLen;
        }
        else {
            result.value.push(values.length);
        }
        
        for (let i = 0; i < values.length; i+=2)
        {
            //Each character is one byte (8bits), we have to merge two bytes into one word
            if (i % 2 === 0 && (i + 1) < values.length)
            {
                result.value.push(this.byteSwap((values.charCodeAt(i) << 8) + values.charCodeAt(i+1)));
            }
            else
            {
                result.value.push(values.charCodeAt(i));
            }
        }
        return result;
    }

    /***************************************************************************/
    /**
     * @brief readMultipleRegisters - Handle reading many registers in a loop
     * protocol: The protocol object used
     * port: The TCP port which is used to read the data
     * item: The requested item with information about register address, data type etc.
     * isDouble: true if 32bit otherwise 16bit (default)
     ****************************************************************************/
    async readMultipleRegisters(protocol, port, item, isDouble = false)
    {
        let command, plcRes;
        let watchdog = false;
        let iterations = 0;
        let area = item.area.toUpperCase();
        let type = item.type.toLowerCase();
        let totalNumberOfRegsToRead = isDouble ? item.count * 2 : item.count;
        let communicationErrorCounter = 0;
        let result = {
            err: false,
            err_msg: ""
        };
        //Watchdog timer for the loops, avoid endless loop for any case
        let watchdogTimer = new Date().getTime();
        let numberOfRegsToRead = totalNumberOfRegsToRead > protocol.MAX_READ_REGS ? protocol.MAX_READ_REGS : totalNumberOfRegsToRead;
        let localObj = this;

        return new Promise(async (res, rej) =>
        {
            //Send the request at least once, use do while
            do {
                protocol.NumberOfRegs = numberOfRegsToRead;
                //For Mewtocol reading 16bit values of WR, WY, WX and WL a different function must be used. 
                if (protocol.selfProtocol === localObj.Constants.PROTOCOLS.mewtocol && localObj.isWordArea(area)) {
                    command = protocol.GeneratReadWords();
                }
                else {
                    command = protocol.GenerateReadRegisters();
                }
                
                await localObj.sendRequestToPLC(command, port).then((response) => {
                    plcRes = protocol.DecodeReadRegisters(response);
                    plcRes.request = `${area}${item.start.toString()}`;

                    if (iterations === 0) {
                        if (type.toUpperCase() === localObj.Constants.datatype.any.name) {
                            result = plcRes;
                        }
                        else {
                            result.err = plcRes.err;
                            result.err_code = plcRes.err_code;
                            result.err_msg = "";
                            result.request = plcRes.request.toUpperCase();
                            result[type] = plcRes[type];
                        }
                    }
                    else {
                        if (type.toUpperCase() === localObj.Constants.datatype.any.name) {
                            result.int = [...result.int, ...plcRes.int];
                            result.uint = [...result.uint, ...plcRes.uint];
                            result.udint = [...result.udint, ...plcRes.udint];
                            result.dint = [...result.dint, ...plcRes.dint];
                            result.hex = [...result.hex, ...plcRes.hex];
                            result.real = [...result.real, ...plcRes.real];
                            result.string = `${result.string}${plcRes.string}`;
                        }
                        else {
                            result.err = plcRes.err;
                            result.err_msg = plcRes.err_msg;
                            result.err_code = localObj.Constants.errorCodes.noErr;
                            result[type] = [...result[type], ...plcRes[type]];
                        }
                    }
                }).catch((err) => {
                    result.err = true;
                    result.err_msg = err;
                    result.err_code = localObj.Constants.errorCodes.unknown;
                    communicationErrorCounter++;
                })
                
                iterations++;
                totalNumberOfRegsToRead -= numberOfRegsToRead;
                protocol.StartAddr += numberOfRegsToRead;
                numberOfRegsToRead = totalNumberOfRegsToRead > protocol.MAX_READ_REGS ? protocol.MAX_READ_REGS : totalNumberOfRegsToRead;

                //Check watchdog
                let loopTime = new Date().getTime();
                if (loopTime - watchdogTimer >= localObj.WATCH_DOG && !LOCALTEST) {
                    watchdog = true;
                    rej();
                }
            }
            while (totalNumberOfRegsToRead > 0 && communicationErrorCounter < localObj.RETRIES && !watchdog)
            res(result);

        })
    }

    /***************************************************************************/
    /**
     * @brief readMultipleBits - Handle reading many bits in a loop
     * protocol: The protocol object used
     * port: The TCP port which is used to read the data
     * item: The requested item with information about register address, data type etc.
     ****************************************************************************/
    async readMultipleBits(protocol, port, item)
    {
        let command, plcRes;
        let watchdog = false;
        let iterations = 0;
        let totalNumberOfRegsToRead = item.count;
        let communicationErrorCounter = 0;
        let result = {};
        //Watchdog timer for the loops, avoid endless loop for any case
        let watchdogTimer = new Date().getTime();
        let numberOfRegsToRead = totalNumberOfRegsToRead > protocol.MAX_READ_BITS ? protocol.MAX_READ_BITS : totalNumberOfRegsToRead;
        let localObj = this;
        //Send the request at least once, use do while
        return new Promise(async (res, rej) =>
        {
            do
            {
                //For bools there is a different limit
                protocol.NumberOfRegs = numberOfRegsToRead;
                command = protocol.GenerateReadBits();
                await localObj.sendRequestToPLC(command, port).then((response) =>
                {
                    plcRes = protocol.DecodeReadBits(response);
                    plcRes.request = item.area.toUpperCase() + item.start.toString();

                    if (iterations === 0)
                    {
                        result.err = plcRes.err;
                        result.err_code = plcRes.err_code;
                        result.request = plcRes.request.toUpperCase();
                        result.state = plcRes.state;
                    }
                    else
                    {
                        result.state = [...result.state, ...plcRes.state];
                    }
                }).catch((err) =>
                {
                    result.err = true;
                    result.err_msg = err;
                    result.err_code = localObj.Constants.errorCodes.unknown;
                    communicationErrorCounter++;
                });

                iterations++;
                totalNumberOfRegsToRead -= numberOfRegsToRead;
                //For mewtocol and mewtocol7 the address must be calculated in hex
                if (protocol.selfProtocol !== localObj.Constants.PROTOCOLS.modbus)
                {
                    protocol.StartAddr = localObj.AddOffsetToMewBoolAddress(protocol.StartAddr.toString(), numberOfRegsToRead);
                }
                else
                {
                    //Modbus - just add the number of regs to the address for the new start address
                    protocol.StartAddr += numberOfRegsToRead;
                }
                numberOfRegsToRead = totalNumberOfRegsToRead > protocol.MAX_READ_BITS ? protocol.MAX_READ_BITS : totalNumberOfRegsToRead;

                //Check watchdog
                let loopTime = new Date().getTime();
                //Localtest is just for debugging
                if (loopTime - watchdogTimer >= localObj.WATCH_DOG && !LOCALTEST)
                {
                    watchdog = true;
                    rej();
                }
            }
            while (totalNumberOfRegsToRead > 0 && communicationErrorCounter < this.RETRIES && !watchdog)
            res(result);
        })
        
    }

    /***************************************************************************/
    /**
     * @brief readString - Handle read string in a loop
     * protocol: The protocol object used
     * port: The TCP port which is used to read the data
     * item: The requested item with information about register address, data type etc.
     ****************************************************************************/
    async readString(protocol, port, item)
    {
        let command, plcRes;
        let watchdog = false;
        let iterations = 0;
        let communicationErrorCounter = 0;
        let result = {};
        //Watchdog timer for the loops, avoid endless loop for any case
        let watchdogTimer = new Date().getTime();
        //First read the header 1 word with information of how many characters are needed to read
        let numberOfRegsToRead = 1;
        let numberOfCharactersIsOdd;
        let totalNumberOfRegsToRead;
        let localObj = this;
        
        //For Strings we need to read the header word first, which tells us how many characters we need to read
        protocol.StartAddr = item.start + 1;
        return new Promise(async (res, rej) =>
        {
            //String is a special case. 
            do
            {
                protocol.NumberOfRegs = numberOfRegsToRead;
                command = protocol.GenerateReadRegisters();
                await localObj.sendRequestToPLC(command, port).then((response) =>
                {
                    plcRes = protocol.DecodeReadRegisters(response);
                    plcRes.request = item.area.toUpperCase() + item.start.toString();

                    //First iteration, only read the single header word with the information of how many characters to read
                    if (iterations === 0)
                    {
                        result.err = plcRes.err;
                        result.err_code = plcRes.err_code;
                        result.request = plcRes.request.toUpperCase();
                        //Initialize the return string
                        result.string = "";
                        if (plcRes.err || !plcRes.int[0])
                        {
                            return;
                        }                       

                        //Now we know how many characters to read. Remember that the characters are byte wise! We always read word wise, so we need to calculate the half of the needed number
                        totalNumberOfRegsToRead = plcRes.int[0];
                        //Since it is the number of characters we need to calculate it into number of words. In case of odd number of characters we need to add an additional word and only read one byte of the last word.
                        numberOfCharactersIsOdd = totalNumberOfRegsToRead % 2 !== 0;
                        totalNumberOfRegsToRead = numberOfCharactersIsOdd ? Math.floor(totalNumberOfRegsToRead / 2) + 1 : totalNumberOfRegsToRead / 2;
                        //Set the start address to the first word of the characters
                        protocol.StartAddr = item.start + 2;

                        //Reset the numberofRegs, so that the calculation for the total number is calculated correctly at the end since we are not reading any characters yet!
                        numberOfRegsToRead = 0;
                    }
                    else
                    {
                        result.err = plcRes.err;
                        result.err_code = plcRes.err_code;
                        //In case the number of chars is odd we need to omit the last character (second byte of the last word)
                        if (numberOfCharactersIsOdd && totalNumberOfRegsToRead <= protocol.MAX_READ_REGS)
                        {
                            result.string += plcRes.string.substring(0, plcRes.string.length - 1);
                        }
                        else
                        {
                            result.string += plcRes.string;
                        }
                    }
                }).catch((err) =>
                {
                    result.err = true;
                    result.err_msg = err;
                    communicationErrorCounter++;
                });

                iterations++;
                //First iteration the numberOfRegsToRead has been reset to 0, so nothing will be changed to totalNumberOfRegsToread and startaddress
                totalNumberOfRegsToRead -= numberOfRegsToRead;

                //The first 2 requests are fix for strings 
                //1. Read the header word
                //2. Read the first words starting from startaddress + 2 (string header offset)
                if (iterations > 1)
                    protocol.StartAddr += numberOfRegsToRead;

                numberOfRegsToRead = totalNumberOfRegsToRead > protocol.MAX_READ_REGS ? protocol.MAX_READ_REGS : totalNumberOfRegsToRead;
                //Check watchdog
                let loopTime = new Date().getTime();
                if (loopTime - watchdogTimer >= localObj.WATCH_DOG && !LOCALTEST)
                {
                    watchdog = true;
                    result.err = true;
                    result.err_msg = "";
                    rej(result);
                }
            }
            while (totalNumberOfRegsToRead > 0 && communicationErrorCounter < localObj.RETRIES && !watchdog)
            res(result);
        })
    }

    /***************************************************************************/
    /**
     * @brief plcGet - Handle the get request for PLC data (e.g. /plcget?DT0&DT1=1&R0)
     ****************************************************************************/
    plcGet(req)
    {
        var LocalModule = this;

        LocalModule.Constants.DebugLog(LocalModule.Constants.PLC_GET_REQ);
        //Check which interface and station number to use
        return new Promise((resolve, reject) =>
        {
            try
            {
                let apiConfigFile = LocalModule.readConfigFileAsJSON(path.join(__dirname, LocalModule.Constants.configFolder, LocalModule.Constants.requests.READ_API + LocalModule.Constants.JSONFILE));
                let startAddr = typeof apiConfigFile.ip1 === LocalModule.Constants.UNDEFINED ? 0 : this.ip2int(apiConfigFile.ip1);
                let endAddr = typeof apiConfigFile.ip2 === LocalModule.Constants.UNDEFINED ? 0 : this.ip2int(apiConfigFile.ip2);
                let reqAddr = this.ip2int(req.ip);

                if (startAddr !== LocalModule.Constants.MIN_IP && startAddr !== LocalModule.Constants.MAX_IP && endAddr !== LocalModule.Constants.MIN_IP && endAddr !== LocalModule.Constants.MAX_IP) {
                    if (reqAddr < startAddr || reqAddr > endAddr) {
                        reject(LocalModule.Constants.ACCESS_DENIED);
                        return;
                    }
                }

                //Extract the request string into single requests
                let request = LocalModule.extractGetRequest(req);

                if (request.error)
                {
                    LocalModule.Constants.DebugLog(`${LocalModule.Constants.EXTRACT_GET_REQ_FAILED}: ${request.message}`);
                    reject(request.message);
                    return;
                }

                let iface, station;
                //Check if write operations are requested, if so check if the write access is granted
                for (let val of request.request)
                {
                    if (val.type === LocalModule.Constants.dataAreas.interface) {
                        iface = val;
                        continue;
                    }
                    
                    if (val.type === LocalModule.Constants.dataAreas.station) {
                        station = val;
                        continue;
                    }                        

                    if ((val.type === LocalModule.Constants.dataAreas.WX.name || val === LocalModule.Constants.dataAreas.X.name) && val.write)
                    {
                        reject(LocalModule.Constants.INPUT_WRITING_FORBIDDEN);
                        return;
                    }
                }

                //Get the port number to use for reading data
                let parameters = LocalModule.setInterface(iface, station);

                //Failed to read the interface settings
                if (parameters.err)
                {
                    LocalModule.Constants.DebugLog(`${LocalModule.Constants.SET_INTERFACE_FAILED}: ${parameters.err_msg}`);
                    reject(parameters.err_msg);
                    return;
                }

                //Check some parameters if the request is valid
                for (let val of request.request)
                {
                    if (this.isStationOrInterface(val.type))
                        continue;

                    //Mewtocol7 with FL request -> not allowed
                    if (parameters.protocol === LocalModule.Constants.PROTOCOLS.mewtocol7 && this.isFLArea(val.type))
                    {
                        reject(LocalModule.Constants.MEW7_NO_FL);
                        return;
                    }

                    let outOfRange = false;
                    //Address range check
                    if (this.isValidArea(val.type))
                    {
                        let maxAddr = val.reg_addr + 1;
                        if (this.is32BitArea(val.type))
                        {
                            maxAddr += 1;
                        }
                        
                        outOfRange = maxAddr > LocalModule.Constants.MAX_MEW7_ADDR;
                        outOfRange |= maxAddr > LocalModule.Constants.MAX_MEW_ADDR && parameters.protocol === LocalModule.Constants.PROTOCOLS.mewtocol;
                        outOfRange |= maxAddr > LocalModule.Constants.MAX_MODBUS_ADDR && parameters.protocol === LocalModule.Constants.PROTOCOLS.modbus;
                        outOfRange |= this.isFLArea(val.type) && maxAddr > LocalModule.Constants.MAX_FL_ADDR;
                        outOfRange |= (val.type === LocalModule.Constants.dataAreas.LD.name || val.type === LocalModule.Constants.dataAreas.DLD.name) && maxAddr > LocalModule.Constants.MAX_LD_ADDR;
                        outOfRange |= val.type === LocalModule.Constants.dataAreas.WR.name && maxAddr > LocalModule.Constants.MAX_WR_ADDR;
                        outOfRange |= val.type === LocalModule.Constants.dataAreas.WL.name && maxAddr > LocalModule.Constants.MAX_WL_ADDR;
                        outOfRange |= (val.type === LocalModule.Constants.dataAreas.WX.name || val.type === LocalModule.Constants.dataAreas.WY.name) && maxAddr > LocalModule.Constants.MAX_WXWY_ADDR;

                        if (outOfRange)
                        {
                            reject(LocalModule.Constants.ADDR_OUT_OF_RANGE);
                            return;
                        }
                    }
                    //Bool area
                    else
                    {
                        outOfRange = val.reg_addr.length > LocalModule.Constants.MAX_BOOL_ADDR_LEN;
                        outOfRange |= val.reg_addr.length > LocalModule.Constants.MAX_BOOL_ADDR_LEN - 1 && parameters.protocol === LocalModule.Constants.PROTOCOLS.mewtocol;

                        if (outOfRange)
                        {
                            reject(LocalModule.Constants.ADDR_OUT_OF_RANGE);
                            return;
                        }
                    }
                }

                let protocol;
                if (parameters.interface === LocalModule.Constants.Interfaces.INTERN.interface)
                {
                    protocol = new Modbus.Modbus();
                }
                else
                {
                    switch (parameters.protocol)
                    {
                        case LocalModule.Constants.PROTOCOLS.mewtocol:
                            protocol = new Mew.Mewtocol();
                            break;

                        case LocalModule.Constants.PROTOCOLS.mewtocol7:
                            protocol = new Mew7.Mewtocol7();
                            break;

                        case LocalModule.Constants.PROTOCOLS.modbus:
                            protocol = new Modbus.Modbus();
                            break;

                        default:
                            protocol = new Mew.Mewtocol();
                    }
                }

                //Set the station number
                protocol.Station = parameters.station;

                if (LOCALTEST)
                    parameters.port = LocalModule.Constants.defaultPort;

                LocalModule.startRequestProcess(request.request, protocol, parameters.port)
                    .then((result) =>
                    {
                        resolve(result);
                    })
                    .catch((err) =>
                    {
                        reject(err);
                    });

            }
            catch (ex)
            {
                LocalModule.Constants.DebugLog(`${LocalModule.Constants.PLC_GET_FAILED}: ${ex.message}`);
                reject(ex.message);
            }
        });
    }

    /***************************************************************************/
    /**
     * @brief startRequestProcess - Start the request process
     * @param req: All requests for PLC get
     * @param protocol: the protocol object
     * @param port: the port number for the connection to the PLC
     * @return: returns an object with read data
     ****************************************************************************/
    async startRequestProcess(req, protocol, port)
    {
        let result = {
            err: false,
            err_msg: "",
            data: []
        };

        for (let item of req)
        {
            //When error occured -> go out
            if (result.err)
                break;

            let Command;
            //A (station number) and I (interface) are not needed for data request
            if (!this.isStationOrInterface(item.type))
            {
                protocol.StartAddr = item.reg_addr;
                protocol.NumberOfRegs = 1;
                switch (item.type)
                {
                    case this.Constants.dataAreas.DT.name:
                        protocol.datatype = this.Constants.dataAreas.DT.name;
                        break;
                    case this.Constants.dataAreas.DDT.name:
                        protocol.datatype = this.Constants.dataAreas.DT.name;
                        protocol.NumberOfRegs = 2;
                        break;
                    case this.Constants.dataAreas.FL.name:
                        protocol.datatype = this.Constants.dataAreas.FL.name;
                        break;
                    case this.Constants.dataAreas.DFL.name:
                        protocol.datatype = this.Constants.dataAreas.FL.name;
                        protocol.NumberOfRegs = 2;
                        break;
                    case this.Constants.dataAreas.LD.name:
                        protocol.datatype = this.Constants.dataAreas.LD.name;
                        break;
                    case this.Constants.dataAreas.DLD.name:
                        protocol.datatype = this.Constants.dataAreas.LD.name;
                        protocol.NumberOfRegs = 2;
                        break;
                    case this.Constants.dataAreas.WR.name:
                        protocol.datatype = this.Constants.dataAreas.WR.name;
                        break;
                    case this.Constants.dataAreas.WL.name:
                        protocol.datatype = this.Constants.dataAreas.WL.name;
                        break;
                    case this.Constants.dataAreas.WX.name:
                        protocol.datatype = this.Constants.dataAreas.WX.name;
                        break;
                    case this.Constants.dataAreas.WY.name:
                        protocol.datatype = this.Constants.dataAreas.WY.name;
                        break;
                    case this.Constants.dataAreas.R.name:
                        protocol.datatype = this.Constants.dataAreas.R.name;
                        break;
                    case this.Constants.dataAreas.Y.name:
                        protocol.datatype = this.Constants.dataAreas.Y.name;
                        break;
                    case this.Constants.dataAreas.X.name:
                        protocol.datatype = this.Constants.dataAreas.X.name;
                        break;
                    case this.Constants.dataAreas.L.name:
                        protocol.datatype = this.Constants.dataAreas.L.name;
                        break;
                }

                var plcRes;

                //Write request
                if (item.write)
                {
                    protocol.writeData.length = 0;
                    switch (protocol.selfProtocol)
                    {
                        //Mewtocol
                        case this.Constants.PROTOCOLS.mewtocol:
                            switch (item.type)
                            {
                                case this.Constants.dataAreas.DT.name:
                                case this.Constants.dataAreas.DDT.name:
                                case this.Constants.dataAreas.FL.name:
                                case this.Constants.dataAreas.DFL.name:
                                case this.Constants.dataAreas.LD.name:
                                case this.Constants.dataAreas.DLD.name:
                                    if (this.is32BitArea(item.type))
                                    {
                                        let res = this.SplitValueToTwoSixteenbits(item.writeVal, this.Constants.writeDataTypes._32bit);
                                        protocol.writeData.push(res.first);
                                        protocol.writeData.push(res.second);
                                    }
                                    else
                                    {
                                        protocol.writeData.push(item.writeVal);
                                    }

                                    Command = protocol.GenerateWriteRegisters();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getWriteResponse(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;

                                case this.Constants.dataAreas.WR.name:
                                case this.Constants.dataAreas.WL.name:
                                case this.Constants.dataAreas.WY.name:
                                    protocol.writeData.push(item.writeVal);
                                    Command = protocol.GenerateWriteWords();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getWriteResponse(protocol, res, item));                                        
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;

                                case this.Constants.dataAreas.R.name:
                                case this.Constants.dataAreas.L.name:
                                case this.Constants.dataAreas.Y.name:
                                    protocol.writeBit = item.writeVal == 0 ? 0 : 1;
                                    Command = protocol.GenerateWriteSingleBit();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getWriteResponse(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;
                            }
                            break;
                        //Mewtocol7
                        case this.Constants.PROTOCOLS.mewtocol7:
                            switch (item.type)
                            {
                                case this.Constants.dataAreas.DT.name:
                                case this.Constants.dataAreas.DDT.name:
                                case this.Constants.dataAreas.FL.name:
                                case this.Constants.dataAreas.DFL.name:
                                case this.Constants.dataAreas.LD.name:
                                case this.Constants.dataAreas.DLD.name:
                                case this.Constants.dataAreas.WR.name:
                                case this.Constants.dataAreas.WL.name:
                                case this.Constants.dataAreas.WY.name:
                                    if (this.is32BitArea(item.type))
                                    {
                                        let res = this.SplitValueToTwoSixteenbits(item.writeVal);
                                        protocol.writeData.push(res.first);
                                        protocol.writeData.push(res.second);
                                    }
                                    else
                                    {
                                        protocol.writeData.push(item.writeVal);
                                    }
                                    Command = protocol.GenerateWriteRegisters();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getWriteResponse(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;

                                case this.Constants.dataAreas.R.name:
                                case this.Constants.dataAreas.L.name:
                                case this.Constants.dataAreas.Y.name:
                                    protocol.writeBit = item.writeVal == 0 ? 0 : 1;
                                    Command = protocol.GenerateWriteSingleBit()
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getWriteResponse(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;
                            }
                            break;
                        //Modbus
                        case this.Constants.PROTOCOLS.modbus:
                            switch (item.type)
                            {
                                case this.Constants.dataAreas.DT.name:
                                case this.Constants.dataAreas.DDT.name:
                                case this.Constants.dataAreas.FL.name:
                                case this.Constants.dataAreas.DFL.name:
                                case this.Constants.dataAreas.LD.name:
                                case this.Constants.dataAreas.DLD.name:
                                case this.Constants.dataAreas.WR.name:
                                case this.Constants.dataAreas.WL.name:
                                case this.Constants.dataAreas.WY.name:
                                    if (this.is32BitArea(item.type))
                                    {
                                        let res = this.SplitValueToTwoSixteenbits(item.writeVal);
                                        protocol.writeData.push(res.first);
                                        protocol.writeData.push(res.second);
                                    }
                                    else
                                    {
                                        protocol.writeData.push(item.writeVal);
                                    }
                                    Command = protocol.GenerateWriteRegisters();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getWriteResponseMew(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;

                                case this.Constants.dataAreas.R.name:
                                case this.Constants.dataAreas.L.name:
                                case this.Constants.dataAreas.Y.name:
                                    protocol.writeBit = item.writeVal == 0 ? 0 : 0xFF00;    //FF00 = set bit to 1
                                    Command = protocol.GenerateWriteSingleBit();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getWriteResponseMew(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;
                            }
                            break;
                    }
                }
                //Read request
                else
                {
                    switch (protocol.selfProtocol)
                    {
                        case this.Constants.PROTOCOLS.mewtocol:
                            switch (item.type)
                            {
                                case this.Constants.dataAreas.DT.name:
                                case this.Constants.dataAreas.DDT.name:
                                case this.Constants.dataAreas.FL.name:
                                case this.Constants.dataAreas.DFL.name:
                                case this.Constants.dataAreas.LD.name:
                                case this.Constants.dataAreas.DLD.name:
                                    Command = protocol.GenerateReadRegisters();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {                                        
                                        result.data.push(this.getDecodedRegs(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;

                                case this.Constants.dataAreas.WR.name:
                                case this.Constants.dataAreas.WL.name:
                                case this.Constants.dataAreas.WY.name:
                                case this.Constants.dataAreas.WX.name:
                                    Command = protocol.GeneratReadWords();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getDecodedRegs(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;

                                case this.Constants.dataAreas.R.name:
                                case this.Constants.dataAreas.L.name:
                                case this.Constants.dataAreas.Y.name:
                                case this.Constants.dataAreas.X.name:
                                    Command = protocol.GenerateReadSingleBit();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getDecodedSingleBit(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;
                            }
                            break;

                        case this.Constants.PROTOCOLS.mewtocol7:
                            switch (item.type)
                            {
                                case this.Constants.dataAreas.DT.name:
                                case this.Constants.dataAreas.DDT.name:
                                case this.Constants.dataAreas.FL.name:
                                case this.Constants.dataAreas.DFL.name:
                                case this.Constants.dataAreas.LD.name:
                                case this.Constants.dataAreas.DLD.name:
                                case this.Constants.dataAreas.WR.name:
                                case this.Constants.dataAreas.WL.name:
                                case this.Constants.dataAreas.WY.name:
                                case this.Constants.dataAreas.WX.name:
                                    Command = protocol.GenerateReadRegisters();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getDecodedRegs(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;

                                case this.Constants.dataAreas.R.name:
                                case this.Constants.dataAreas.L.name:
                                case this.Constants.dataAreas.Y.name:
                                case this.Constants.dataAreas.X.name:
                                    Command = protocol.GenerateReadBits();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        plcRes = protocol.DecodeReadBits(res.toString());
                                        plcRes.request = item.type + item.reg_addr.toString();
                                        result.data.push(plcRes);
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;
                            }
                            break;

                        case this.Constants.PROTOCOLS.modbus:
                            switch (item.type)
                            {
                                case this.Constants.dataAreas.DT.name:
                                case this.Constants.dataAreas.DDT.name:
                                case this.Constants.dataAreas.FL.name:
                                case this.Constants.dataAreas.DFL.name:
                                case this.Constants.dataAreas.LD.name:
                                case this.Constants.dataAreas.DLD.name:
                                case this.Constants.dataAreas.WR.name:
                                case this.Constants.dataAreas.WL.name:
                                case this.Constants.dataAreas.WY.name:
                                case this.Constants.dataAreas.WX.name:
                                    Command = protocol.GenerateReadRegisters();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getDecodedRegs(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;

                                case this.Constants.dataAreas.R.name:
                                case this.Constants.dataAreas.L.name:
                                case this.Constants.dataAreas.Y.name:
                                case this.Constants.dataAreas.X.name:
                                    Command = protocol.GenerateReadSingleBit();
                                    await this.sendRequestToPLC(Command, port).then((res) =>
                                    {
                                        result.data.push(this.getDecodedSingleBit(protocol, res, item));
                                    }).catch((err) =>
                                    {
                                        result.err = true;
                                        result.err_msg = err;
                                    });
                                    break;
                            }
                            break;
                    }
                }
            }
            else
                continue;

        }
        return result;
    }

    /***************************************************************************/
    /**
     * @brief setInterface - Set the interface to use for the data requests
     * @param iface: parameter with the user information which interface to use.
     * @param station: parameter with user information which station number to use.
     * When empty (null or undefined) then the default plc interface will be used
     * @return: returns an object (always with an error flag and error message)
     * When everything ok -> returns the port number to use
     * When failure -> returns error with an error message
     ****************************************************************************/
    setInterface(iface, station)
    {
        /*Interfaces:
         * I = 0: RS232 (COM1)  -> Service 0
         * I = 1: RS485 (COM2)  -> Service 3
         * I = 2: USB (COM3)    -> Service 1
         * I = 3: INTERN        -> Service 4
         * I = 4: Ethernet (COM4) -> Service 5
         * */
        let result = {
            err: false,
            err_msg: "",
            port: 0,
            station: 0,
            protocol: "",
            interface: 0
        };

        if (iface && station && iface.type && station.type)
        {
            let interfaceNum = iface.writeVal;            
            // Is interface in range?
            if  (!this.isInterfaceInRange(interfaceNum))
            {
                result.err = true;
                result.err_msg = this.Constants.INTERFACE_OUT_OF_RANGE;
                return result;
            }
            //Take over the correct service ID
            switch (interfaceNum) {
                case 0:
                    result.interface = this.Constants.Interfaces.RS232.interface;
                    break;
                case 1:
                    result.interface = this.Constants.Interfaces.RS485.interface;
                    break;
                case 2:
                    result.interface = this.Constants.Interfaces.USB.interface;
                    break;
                case 3:
                    result.interface = this.Constants.Interfaces.INTERN.interface;
                    break;
                case 4:
                    result.interface = this.Constants.Interfaces.ETHERNET.interface;
                    break;

                default:
                    result.interface = this.Constants.Interfaces.RS232.interface;
                    break;
            }

            // Is station number in range?
            if (!this.isStationNumberValid(station.writeVal))
            {
                result.err = true;
                result.err_msg = this.Constants.STATION_OUT_OF_RANGE;
                return result;
            }

            try 
            {
                //Check if the com port is enabled
                let comData = this.getComEnabled(interfaceNum);
                if (!comData.enabled) {
                    result.err = true;
                    result.err_msg = this.Constants.COM_NOT_ENABLED;
                    return result;
                }

                //Now check if a TCP port is enabled
                let tcpPortData = this.getTCPPortOpened(comData.serviceID);
                if (tcpPortData.enabled) {
                    result.station = station.writeVal;
                    result.protocol = comData.protocol;
                    result.port = tcpPortData.port;
                    return result;
                }

                //No port found -> return error
                result.err = true;
                result.err_msg = this.Constants.TCP_PORT_DISABLED;
                return result;
            }
            catch (ex)
            {
                result.err = true;
                result.err_msg = ex.message;
                return result;
            }
        }
        //No interface settings provided -> use default PLC settings
        else
        {
            //Check if the com port is enabled
            let comData = this.getComEnabled(0, true);
            if (!comData.enabled) {
                result.err = true;
                result.err_msg = this.Constants.COM_NOT_ENABLED;
                return result;
            }

            //Now check if a TCP port is enabled
            let tcpPortData = this.getTCPPortOpened(comData.serviceID);
            if (tcpPortData.enabled) {
                result.station = comData.station;
                result.protocol = comData.protocol;
                result.port = tcpPortData.port;
                return result;
            }

            //No port found -> return error
            result.err = true;
            result.err_msg = this.Constants.TCP_PORT_DISABLED;
            return result;
        }
    }

    /***************************************************************************/
    /**
     * @brief sendRequestToPLC - Send the command to PLC (e.g. Mewtocol)
     * @param command: The command to send
     * @param port: The TCP port on localhost to connect to
     * @return: returns the raw data
     ****************************************************************************/
    async sendRequestToPLC(command, port)
    {
        var client = new net.Socket();
        let LocalConstants = this.Constants;
        let isRejected = false, isResolved = false;
        return new Promise((resolve, reject) =>
        {            
            client.setTimeout(LocalConstants.Timeout_Status_request, function () {
                client.destroy();
                isRejected = true;
                reject(LocalConstants.TIMEOUT);
            });

            client.connect(port, LOCALTEST ? LocalConstants.REMOTE_IP : LocalConstants.LOCAL_HOST, function () {
                client.write(command);
            });

            client.on('data', function (data) {
                client.destroy();
                isResolved = true;
                resolve(data);
            });

            client.on('error', function (err) {
                client.destroy();
                isRejected = true;
                reject(err.message);
            });

            client.on("close", function (hadErr) {
                if (!isRejected && !isResolved) {
                    reject("no respond");
                }
            })

        })
    }

    //Returns the config file as JSON
    readConfigFileAsJSON(filepath)
    {
        let apiConfigFile = fs.readFileSync(filepath);
        return JSON.parse(apiConfigFile);
    }

    /***************************************************************************/
    /**
     * @brief resetApplicationSettings - Reset all application settings to default
     * @return: returns either a success message or an error message
     ****************************************************************************/
    resetApplicationSettings(req)
    {
        var LocalModule = this;
        var LocalConstants = LocalModule.Constants;
        let removelogfiles = req.query.removelogfiles;

        //Also remove the content of folders
        if (removelogfiles)
        {
            exec("rm -r /var/volatile/pew/log/*", function (err)
            {
                err && LocalConstants.DebugLog(`failed to remove temp log files: ${err}`)
            });
            exec("rm -r /home/log/log/*", function (err)
            {
                err && LocalConstants.DebugLog(`failed to remove internal log files: ${err}`)
            });
            exec("rm -r /mnt/usbmemory/log/*", function (err)
            {
                err && LocalConstants.DebugLog(`failed to remove usb log files: ${err}`)
            });
        }

        return new Promise((resolve, reject) =>
        {
            //first delete all content in the config folder 
            exec(`rm -r ${path.join(LocalConstants.FULL_APP_PATH, "node", LocalConstants.configFolder)}*`, function ()
            {
                exec(`cp -r ${path.join(LocalConstants.FULL_APP_PATH, LocalConstants.configDefault)}* ${path.join(LocalConstants.FULL_APP_PATH, "node", LocalConstants.configFolder)}`, function (cpErr)
                {
                    if (cpErr)
                    {
                        if (cpErr.message.includes("omitting"))
                        {
                            resolve(LocalConstants.OPERATION_SUCCESSFUL)
                            LocalModule.sendSIGToEmbeddedApp(LocalConstants.PEWMAIN);
                        }
                        else
                        {
                            reject(cpErr.message);
                        }
                        return;
                    }
                    LocalModule.restartApplications();
                    resolve(LocalConstants.OPERATION_SUCCESSFUL);
                    LocalModule.sendSIGToEmbeddedApp(LocalConstants.PEWMAIN);
                })
            })
        })
    }

    /***************************************************************************/
    /**
     * @brief getCountValues - Read live count values for monitoring page
     * @return: returns all count values and version number
     ****************************************************************************/
    async getCountValues()
    {
        return new Promise(async (res, rej) =>
        {
            let result = {
                err: false,
                err_code: "",
                err_msg: ""
            };
            let mew = new Mew.Mewtocol();
            mew.StartAddr = this.Constants.startAddrPLCVersion;
            let cmd = mew.GenerateReadRegisters();
            //Read Version number of PLC
            await this.sendRequestToPLC(cmd, this.Constants.defaultPort).then(respond =>
            {
                let decodedData = mew.DecodeReadRegisters(respond);
                if (decodedData && !decodedData.err)
                {
                    result.plcVersion = decodedData.uint[0];
                }
                else
                {
                    result.err = true;
                    result.err_code = decodedData.err_code;
                    rej(JSON.stringify(result));
                }
            }).catch(err =>
            {
                result.err = true;
                result.err_msg = err;
                rej(JSON.stringify(result));
            })

            //Read PLC Date and time
            mew.StartAddr = this.Constants.startAddrPLCDateTime;
            mew.NumberOfRegs = 2;
            cmd = mew.GenerateReadRegisters();
            await this.sendRequestToPLC(cmd, this.Constants.defaultPort).then(respond =>
            {
                let decodedData = mew.DecodeReadRegisters(respond);
                if (decodedData && !decodedData.err)
                {
                    result.plcDateTime = decodedData.udint[0];
                }
                else
                {
                    result.err = true;
                    result.err_code = decodedData.err_code;
                    rej(JSON.stringify(result));
                }
            }).catch(err =>
            {
                result.err = true;
                result.err_msg = err;
                rej(JSON.stringify(result));
            })

            //Read Status of FP-I4C and MQTT
            mew.StartAddr = this.Constants.startAddrStatus;
            mew.NumberOfRegs = 2;
            cmd = mew.GenerateReadRegisters();
            await this.sendRequestToPLC(cmd, this.Constants.defaultPort).then(respond =>
            {
                let decodedData = mew.DecodeReadRegisters(respond);
                if (decodedData && !decodedData.err)
                {
                    result.status_fpi4c = decodedData.int[0];
                    result.status_mqtt = decodedData.int[1];
                }
                else
                {
                    result.err = true;
                    result.err_code = decodedData.err_code;
                    rej(JSON.stringify(result));
                }
            }).catch(err =>
            {
                result.err = true;
                result.err_msg = err;
                rej(JSON.stringify(result));
            })

            //Read the last error code
            mew.StartAddr = this.Constants.startAddrErrorLogs;
            mew.NumberOfRegs = 5;
            cmd = mew.GenerateReadRegisters();
            await this.sendRequestToPLC(cmd, this.Constants.defaultPort).then(respond =>
            {
                let decodedData = mew.DecodeReadRegisters(respond);
                if (decodedData && !decodedData.err)
                {
                    result.last_error = decodedData.hex;
                }
                else
                {
                    result.err = true;
                    result.err_code = decodedData.err_code;
                    rej(JSON.stringify(result));
                }
            }).catch(err =>
            {
                result.err = true;
                result.err_msg = err;
                rej(JSON.stringify(result));
            })

            //Read the live count values
            let countValues = [];
            mew.StartAddr = this.Constants.startAddrCountValues;
            let numberOfRegsToRead = this.Constants.numberOfRegsCountValues;
            do
            {
                mew.NumberOfRegs = numberOfRegsToRead > mew.MAX_READ_REGS ? mew.MAX_READ_REGS : numberOfRegsToRead;
                cmd = mew.GenerateReadRegisters();
                await this.sendRequestToPLC(cmd, this.Constants.defaultPort).then(respond =>
                {
                    let decodedData = mew.DecodeReadRegisters(respond);
                    if (decodedData && !decodedData.err)
                    {
                        for (let val of decodedData.udint)
                        {
                            countValues.push(val);
                        }
                    }
                    else
                    {
                        result.err = true;
                        result.err_code = decodedData.err_code;
                        rej(JSON.stringify(result));
                        return;
                    }
                }).catch(err =>
                {
                    result.err = true;
                    result.err_msg = err;
                    rej(JSON.stringify(result));
                    return;
                });

                numberOfRegsToRead -= mew.NumberOfRegs;
                mew.StartAddr += mew.NumberOfRegs;
            }
            while (numberOfRegsToRead > 0 && !result.err)

            result.countsTodayIn = [];
            result.countsTodayOut = [];
            result.countsYstIn = [];
            result.countsYstOut = [];
            result.countsDayB4YstIn = [];
            result.countsDayB4YstOut = [];

            //There are always 8 count values, since max. number of sensors are 8
            //But each count value contains count for In and Out -> 16 total iterations
            let j = 0;
            for (let i = 0; i < 8; i++)
            {
                result.countsTodayIn[i] = countValues[j];
                result.countsTodayOut[i] = countValues[j+1];

                result.countsYstIn[i] = countValues[j+16];
                result.countsYstOut[i] = countValues[j + 16 + 1];

                result.countsDayB4YstIn[i] = countValues[j + 32];
                result.countsDayB4YstOut[i] = countValues[j + 32 +1];

                j += 2;
            }

            res(JSON.stringify(result));
        })
    }

    /***************************************************************************/
    /**
     * @brief getErrorLogs - Read all error logs (except the first one) from the PLC
     * @return: returns all historical errors
     ****************************************************************************/
    getErrorLogs()
    {
        return new Promise(async (res, rej) =>
        {
            let result = {
                err: false,
                err_code: "",
                err_msg: ""
            };

            result.errorLog = [];
            let mew = new Mew.Mewtocol();
            //Don't read the first error log since this will be read with the count values
            mew.StartAddr = this.Constants.startAddrErrorLogs + 5;
            let cmd = "";
            let numberOfRegsToRead = this.Constants.numberOfRegsErrorLogs;
            do
            {
                mew.NumberOfRegs = numberOfRegsToRead > mew.MAX_READ_REGS ? mew.MAX_READ_REGS : numberOfRegsToRead;
                cmd = mew.GenerateReadRegisters();
                await this.sendRequestToPLC(cmd, this.Constants.defaultPort).then(respond =>
                {
                    let decodedData = mew.DecodeReadRegisters(respond);
                    if (decodedData && !decodedData.err)
                    {
                        for (let val of decodedData.hex)
                        {
                            result.errorLog.push(val);
                        }
                    }
                    else
                    {
                        result.err = true;
                        result.err_code = decodedData.err_code;
                        rej(JSON.stringify(result));
                    }
                }).catch(err =>
                {
                    result.err = true;
                    result.err_msg = err;
                    rej(JSON.stringify(result));
                });

                numberOfRegsToRead -= mew.NumberOfRegs;
                mew.StartAddr += mew.NumberOfRegs;
            }
            while (numberOfRegsToRead > 0 && !result.err)

            res(JSON.stringify(result));
        });
    }

    /***************************************************************************/
    /**
     * @brief setBit - Sets a single bit in the PLC
     * @return: returns err and err_msg (if any)
     ****************************************************************************/
    setBit(request)
    {
        if (request.removelogfiles)
        {
            exec("rm -r /var/volatile/pew/log/*", function () { 
                 //Dont need to do anything
            });
            exec("rm -r /home/log/log/*", function () {
                //Dont need to do anything
            });
            exec("rm -r /mnt/usbmemory/log/*", function () {
                //Dont need to do anything
            });
        }

        return new Promise((res, rej) =>
        {
            let result = {
                err: false,
                err_msg: ""
            };

            if (!request || !request.addr || !request.area)
            {
                result.err = true;
                result.err_msg = this.Constants.MISSING_PARAM;
                rej(JSON.stringify(result));
            }

            let area = request.area;
            let addr = request.addr;

            let mew = new Mew.Mewtocol();
            mew.datatype = area.toUpperCase();
            mew.StartAddr = addr;
            mew.writeBit = 1;

            let cmd = mew.GenerateWriteSingleBit();
            this.sendRequestToPLC(cmd, this.Constants.defaultPort).then(respond =>
            {
                let writeOK = mew.WriteResponseOK(respond);
                result.err = writeOK.err;
                result.err_msg = writeOK.err_code;
                res(JSON.stringify(result));
            }).catch(err =>
            {
                result.err = true;
                result.err_msg = err;
                rej(JSON.stringify(result));
            })
            
        })
    }    

    /***************************************************************************/
    /**
     * @brief sendSIGToEmbeddedApp - Sends a user signal to pew_main or other app
     * @app: the app name
     ****************************************************************************/
    sendSIGToEmbeddedApp(app)
    {
        var LocalConstants = this.Constants;
        //Debug information
        LocalConstants.DebugLog(`${LocalConstants.SEND_USR_SIG}: ${app}`);

        this.getPIDofProcess(app).then(processId => {
            //Now send the USR1 Signal to the pew main application
            exec(`kill -s USR1 ${processId}`, killErr => {
                if (killErr) {
                    //Debug information
                    LocalConstants.DebugLog(`${LocalConstants.USR_SIG1_FAILED}: ${killErr}`);
                }
            })
        }).catch(err => {
            return LocalConstants.DebugLog(`${LocalConstants.PID_NOT_FOUND}: ${err}`);
        })
    }
    //Helper functions
    /***************************************************************************
     * @brief valueWithinLimit - returns true if the value is within the data type range
     * otherwise return false
     * type: data type (int, dint, real etc.)
     * values: the values which should be checked against the limit
     ****************************************************************************/
    valueWithinLimit(type, values)
    {
        let result = true;
        if (Array.isArray(values))
        {
            for (let val of values)
            {
                switch (type)
                {
                    case this.Constants.datatype.int.name:
                    case this.Constants.datatype.uint.name:
                        result = val >= this.MIN_INT16 && val <= this.MAX_INT16;
                        break;
                    case this.Constants.datatype.dint.name:
                    case this.Constants.datatype.udint.name:
                        result = val >= this.MIN_INT32 && val <= this.MAX_INT32;
                        break;
                    case this.Constants.datatype.real.name:
                        result = val >= this.MIN_REAL && val <= this.MAX_REAL;
                        break;
                    case this.Constants.datatype.bool.name:
                        result = Number(val) >= 0 && Number(val) <= 1 && values.length <= this.Constants.MAX_BITS_WRITE;
                        break;
                }
                if (!result)
                {
                    break;
                }
            }
        }
        else
        {
            switch (type)
            {
                case this.Constants.datatype.int.name:
                case this.Constants.datatype.uint.name:
                    result = values >= this.MIN_INT16 && values <= this.MAX_INT16;
                    break;
                case this.Constants.datatype.dint.name:
                case this.Constants.datatype.udint.name:
                    result = values >= this.MIN_INT32 && values <= this.MAX_INT32;
                    break;
                case this.Constants.datatype.real.name:
                    result = values >= this.MIN_REAL && values <= this.MAX_REAL;
                    break;
                case this.Constants.datatype.bool.name:
                    result = Number(values) >= 0 && Number(values) <= 1;
                    break;
            }
        }

        return result;
    }

    /***************************************************************************
     * @brief createFolderIfNotExists - creates the folder at the path if it doesnt exists
     * path: the path to check whether it exist or not and to create if it doesnt exist
     ****************************************************************************/
    createFolderIfNotExists(folderpath)
    {
        try
        {
            fs.accessSync(folderpath, fs.constants.F_OK);
        }
        catch (ex)
        {            
            //Failed -> folder does not exist, create the folder
            fs.mkdirSync(folderpath, true);
        }
    }

    /***************************************************************************
     * @brief isRequestFromLocalhost - Returns true when request is from localhost
     ****************************************************************************/
    isRequestFromLocalhost(requestIP){
        return requestIP === this.Constants.LOCAL_HOST_v6 || requestIP === this.Constants.LOCAL_HOST;
    }

    /***************************************************************************
     * @brief returnForbidden - Add debug information and returns forbidden to user
     ****************************************************************************/
    returnForbidden(requestIP, res)
    {
        //Debug information
        this.Constants.DebugLog(`${this.Constants.FORBIDDEN_REQ}: ${requestIP}`);
        res.writeHead(this.Constants.httpStatus.FORBIDDEN)
        res.end(this.Constants.NOT_ALLOWED)
    }

    /***************************************************************************
     * @brief returnWithFailHeaderAndMsg - return a failed http header and a fail message (including debug message)
     ****************************************************************************/
    returnWithFailHeaderAndMsg(debugMsg, httpHeader, failMsg = '', res)
    {
        debugMsg && this.Constants.DebugLog(debugMsg);
        httpHeader && res.writeHead(httpHeader);
        res.end(failMsg);
    }

    /***************************************************************************
     * @brief isBoolArea - Returns true when the given area is a bool area
     ****************************************************************************/
    isBoolArea(area, second = '')
    {
        return area === this.Constants.dataAreas.R.name || area === this.Constants.dataAreas.X.name || area === this.Constants.dataAreas.Y.name || (area === this.Constants.dataAreas.L.name && second !== 'D');
    }

    /***************************************************************************
     * @brief isValidDataType - Returns true when the given type is valid
     ****************************************************************************/
    isValidDataType(type) {
        for (let key of Object.keys(this.Constants.datatype)) {
            if (type.toUpperCase() === this.Constants.datatype[key].name) {
                return true;
            }
        }
        return false;
    }

    /***************************************************************************
     * @brief isValidArea - Returns true when the given area is valid
     ****************************************************************************/
    isValidArea(area) {
        return area === this.Constants.dataAreas.DT.name || area === this.Constants.dataAreas.DDT.name ||
            area === this.Constants.dataAreas.FL.name || area === this.Constants.dataAreas.DFL.name ||
            area === this.Constants.dataAreas.LD.name || area === this.Constants.dataAreas.DLD.name ||
            area === this.Constants.dataAreas.WR.name || area === this.Constants.dataAreas.WL.name ||
            area === this.Constants.dataAreas.WX.name || area === this.Constants.dataAreas.WY.nam
    }

    /***************************************************************************
     * @brief isInterfaceInRange - Returns true when the given interface is valid
     ****************************************************************************/
    isInterfaceInRange(iface) {
        return iface <= this.Constants.Interfaces.ETHERNET.interface && iface >= this.Constants.Interfaces.RS232.interface;
    }

    /***************************************************************************
     * @brief is32BitArea - Returns true when the given area is 32bit (ddt, dfl etc.)
     ****************************************************************************/
    is32BitArea(area) {
        return area === this.Constants.dataAreas.DDT.name || area === this.Constants.dataAreas.DFL.name || area === this.Constants.dataAreas.DLD.name;
    }

    /***************************************************************************
     * @brief isFLArea - Returns true when the given area is FL or DFL
     ****************************************************************************/
    isFLArea(area) {
        return area === this.Constants.dataAreas.FL.name || area === this.Constants.dataAreas.DFL.name
    }

    /***************************************************************************
     * @brief isDataregisterArea - Returns true when the given area is DT, FL or LD
     ****************************************************************************/
    isDataregisterArea(area)
    {
        return area === this.Constants.dataAreas.DT.name || area === this.Constants.dataAreas.FL.name || area === this.Constants.dataAreas.LD.name;   
    }

    /***************************************************************************
     * @brief isWordArea - Returns true when the given area is WR, WX, WY or WL
     ****************************************************************************/
    isWordArea(area)
    {
        return area === this.Constants.dataAreas.WR.name || area === this.Constants.dataAreas.WX.name || area === this.Constants.dataAreas.WY.name || area === this.Constants.dataAreas.WL.name;
    }

    /***************************************************************************
     * @brief isTypeMismatch - Returns true when the type doesnt match
     ****************************************************************************/
    isTypeMismatch(type, value) {
        return (type === this.Constants.datatype.bool.name && typeof (value) !== "boolean" && typeof (value) !== "number") ||
            (type === this.Constants.datatype.string.name && typeof (value) !== "string") ||
            (type === this.Constants.datatype.real.name && typeof (value) !== "number")
    }

    /***************************************************************************
     * @brief isStationOrInterface - Returns true when the type is either 
     * station number or interface
     ****************************************************************************/
    isStationOrInterface(type) {
        return type === this.Constants.dataAreas.station || type === this.Constants.dataAreas.interface
    }

    /***************************************************************************
     * @brief isValidBoolAddress - Returns true when the address is a valid
     * bool address (e.g. R1A, Y8). Invalid address are for example RA1
     ****************************************************************************/
    isValidBoolAddress(addr) {
        let hexAddr;
        for (let c = 0; c < addr.length; c++) {
            //if the hex address is not the last digit then it's an error
            if (isNaN(addr[c]) && c !== addr.length - 1) {
                return false;
            }

            if (isNaN(addr[c]) && c === addr.length - 1) {
                hexAddr = addr[c];
                //Check if the hex address is in range (A-F allowed!)
                if (!this.allowedHexAddresses.includes(hexAddr.toUpperCase())) {
                    return false;
                }
            }
        }
        return true;
    }

    /***************************************************************************
     * @brief removeLeadingZeroes - removes all leading zeroes of the input string
     ****************************************************************************/
    removeLeadingZeroes(val) {
        let posLeadZero = val.indexOf("0");
        if (posLeadZero === 0 && posLeadZero !== val.length - 1 && val.length > 1) {
            return val.replace(/^0+/, '');
        }
    }

    /***************************************************************************
     * @brief getWriteResponse - returns the write response object with
     * err, err_code and request for Mewtocol, Mewtocol7 or Modbus
     * protocol: the mewtocol protocol object
     * responseMsg: the response message from request
     * request: requested item
     ****************************************************************************/
    getWriteResponse(protocol, responseMsg, request) {
        let result = protocol.WriteResponseOK(responseMsg.toString());
        result.request = request.type + request.reg_addr.toString();
        return result;
    }

    /***************************************************************************
     * @brief getDecodedRegs - returns the read response object with
     * err, err_code and data request for Mewtocol, Mewtocol7 or Modbus
     * protocol: the mewtocol protocol object
     * responseMsg: the response message from request
     * request: requested item
     ****************************************************************************/
    getDecodedRegs(protocol, responseMsg, request) {
        let result = protocol.DecodeReadRegisters(responseMsg);
        result.request = request.type + request.reg_addr.toString();
        return result;
    }

    /***************************************************************************
     * @brief getDecodedSingleBit - returns the read response object with
     * err, err_code and data request for Mewtocol, Mewtocol7 or Modbus
     * protocol: the mewtocol protocol object
     * responseMsg: the response message from request
     * request: requested item
     ****************************************************************************/
    getDecodedSingleBit(protocol, responseMsg, request) {
        let result = protocol.DecodeSingleBit(responseMsg);
        result.request = request.type + request.reg_addr.toString();
        return result;
    }

    /***************************************************************************
     * @brief getConfFile - Returns a config json file
     ****************************************************************************/
    getConfFile(filename) {
        let LocalModule = this;
        let file = path.join(__dirname, LocalModule.Constants.configFolder, filename + LocalModule.Constants.JSONFILE)
        return new Promise((res, rej) => {
            fs.readFile(file, (err, data) => {
                if (err) {
                    rej(err);
                    return;
                }

                let settings = JSON.parse(data);
                res(settings);
            })
        })
    }

    /***************************************************************************
     * @brief getConfFile - Returns a config json file
     ****************************************************************************/
    getConfFileSync(filename) {
        let result = {};
        let LocalModule = this;
        let file = path.join(__dirname, LocalModule.Constants.configFolder, filename + LocalModule.Constants.JSONFILE)
        try {
            let data = fs.readFileSync(file);
            result.data = JSON.parse(data.toString());
            result.err = false;
            result.err_msg = "";
            return result;
        }
        catch (ex) {
            result.err = true;
            result.err_msg = ex.message;
        }
    }

    /***************************************************************************
     * @brief getFolderContent - Returns the content of a folder
     ****************************************************************************/
    getFolderContent(filepath) {
        return new Promise((res, rej) => {
            glob(filepath + '/**/*', {nodir: true}, function (err, files) {
                if (err) {
                    rej(err);
                    return;
                }
                let filesRelative = files.map(function (match) {
                    return path.relative(filepath, match);
                })
                res(JSON.stringify(filesRelative));
            })
        })
    }

    /***************************************************************************
     * @brief getComEnabled - Returns the com object with information if
     * com port is enabled and which protocol is used
     * useDefaultCom: check the default settings
     ****************************************************************************/
    getComEnabled(com, useDefaultCom = false) {
        let result = {
            enabled: false,
            protocol: ""
        }

        try
        {            
            let data = fs.readFileSync(path.join(__dirname, this.Constants.configFolder, this.Constants.requests.READ_INTERFACE + this.Constants.JSONFILE));
            let jsonData = JSON.parse(data);
            let serviceID;
            
            if (useDefaultCom) {
                serviceID = jsonData.interface_plc;                
            }
            else {
                //IN case the Interface is coming from the JSON object, the values has different meanings!!
                /*
                     •	0 = COM1 (RS232)
                    •	1 = COM2 (RS485)
                    •	2 = COM3 (USB)
                    •	3 = INTERN
                    •	4 = COM4 (Ethernet)
                */
                switch (com) {
                    case 0:
                        serviceID = this.Constants.Interfaces.RS232.service;
                        break;
                    case 1:
                        serviceID = this.Constants.Interfaces.RS485.service;
                        break;
                    case 2:
                        serviceID = this.Constants.Interfaces.USB.service;
                        break;
                    case 3:
                        serviceID = this.Constants.Interfaces.INTERN.service;
                        break;
                    case 4:
                        serviceID = this.Constants.Interfaces.ETHERNET.service;
                        break;

                    default:
                        serviceID = this.Constants.Interfaces.RS232.service;
                }
            }

            for (let val of jsonData.interface) {
                if (val.service === serviceID) {
                    if (!val.enabled) {
                        return result;
                    }

                    result.enabled = true;
                    result.protocol = val.protocol;
                    result.station = jsonData.address_plc;
                    result.serviceID = serviceID;
                    return result;
                }
            }

            return result;
        }
        catch (ex) {
            return result;
        }
    }

    /***************************************************************************
     * @brief getTCPPortOpened - Returns the tcp port object with information if
     * port is opened and which port is used
     ****************************************************************************/
    getTCPPortOpened(serviceID) {
        let result = {
            enabled: false,
            port: 0
        }
        let data = fs.readFileSync(path.join(__dirname, this.Constants.configFolder, this.Constants.requests.READ_PORTS + this.Constants.JSONFILE));
        let jsonData = JSON.parse(data);

        for (let val of jsonData.thread) {
            if (val.service === serviceID) {
                if (val.port !== 0) {  
                    result.enabled = true;
                    result.port = val.port;
                    return result;
                }
            }
        }

        return result;
    }

    /***************************************************************************
     * @brief extractUserWebFiles - extract the zip file that the user has 
     * uploaded into the http folder
     * returns a promise
     ****************************************************************************/
    extractUserWebFiles() {
        var LocalModule = this;
        return new Promise((res, rej) => {
            try {
                //First check if file exists
                let userFilePath = path.join(__dirname, this.Constants.tempFolder, this.Constants.USER_WEB_FILE_NAME);
                if (!fs.existsSync(userFilePath)) {
                    return false;
                }

                //Check if target folder exists, if not create it
                let targetFolder = path.join(__dirname, this.Constants.configSubfolders.http_server);
                LocalModule.createFolderIfNotExists(targetFolder);

                let fileContent = fs.readFileSync(userFilePath);                
                let readzip = new JSZip();

                readzip.loadAsync(fileContent).then(function (zip) {
                    //Delete all content from http server folder
                    exec(`rm -r ${targetFolder}/*`, () => {
                        //unzip the file
                        Object.keys(zip.files).forEach(filename => {
                            try {
                                var file = filename;
                                var isSubFolder = file.substring(filename.length - 1) === "/";

                                //Check if it is a foldername, if so dont do anything (our node version doesnt support "continue" )
                                if (!isSubFolder) {
                                    zip.file(filename).async('nodebuffer').then(function (content) {                                        
                                        let isBinaryfile = isBinaryFileSync(content, content.length);
                                        //Check if the file is in a sub directory. If so we need to check if the directory exists, if not create it.
                                        if (filename.indexOf("\\") !== -1) {
                                            var subdirFile = filename.split("\\");
                                            var subsubdir = targetFolder;
                                            file = path.join(targetFolder, file);
                                            //In case the subdirectory has more subfolders.....
                                            for (let i = 0; i < subdirFile.length - 1; i++) {
                                                subsubdir += i !== subdirFile.length - 1 ? `${subdirFile[i]}\\` : subdirFile[i];
                                                LocalModule.createFolderIfNotExists(subsubdir);
                                            }

                                            fs.writeFileSync(file, isBinaryfile ? content : content.toString());
                                        }
                                        else {
                                            fs.writeFileSync(path.join(targetFolder, file), isBinaryfile ? content : content.toString());
                                        }
                                    })
                                }
                                else {
                                    //Check if folder already exists
                                    let subFolderName = file.substr(0, file.length - 1);
                                    LocalModule.createFolderIfNotExists(path.join(targetFolder, subFolderName));
                                }
                            }
                            catch (ex) {
                                //Show Debug message
                                LocalModule.Constants.DebugLog(`${LocalModule.Constants.EXTRACT_WEB_FILES_FAILED}: ${ex.message}`);
                                return rej(ex.message);
                            }
                        })

                        //Everything ok -> send a message to pew_main so that it can reload all the settings
                        res(LocalModule.Constants.OPERATION_SUCCESSFUL);
                    });
                })
            }
            catch (ex) {
                this.Constants.DebugLog(`${LocalModule.Constants.EXTRACT_WEB_FILES_FAILED}: ${ex.message}`);
                rej();
            }
        })
    }

    /***************************************************************************
     * @brief removeUserWebFiles - removes all user defined web files
     ****************************************************************************/
    removeUserWebFiles() {
        return new Promise((res, rej) => {
            try {
                let targetFolder = path.join(__dirname, this.Constants.configSubfolders.http_server);
                exec(`rm -r ${targetFolder}/*`, (err) => {
                    if (err) {
                        return rej(err);
                    }

                    res();
                });
            }
            catch { }
        })        
    }

    /***************************************************************************
     * @brief isPortOccupied - returns true if the given port is occupied, otherwise false
     ****************************************************************************/
    isPortOccupied(port) {
        try {
            //When lsof was successful, the port is occupied
            execSync(`netstat -pnltu | grep \"\\b:${port}\\b\"`);
            return true;
        }
        catch (ex) {
            //When lsof fails, it has not found any applications using the port
            return false;
        }
    }

    /***************************************************************************
     * @brief sysLogMessage - outputs a standard message from nodejs
     ****************************************************************************/
    sysLogMessage(moduleName, msg) {
        let message = `${moduleName}: ${msg}`;
        console.log(message);
    }

    /***************************************************************************
     * @brief getPIDofProcess - returns a promise. In case of success the PID(s) is returned
     * otherwise the error is returned
     ****************************************************************************/
    getPIDofProcess(processName) {
        return new Promise((res, rej) => {
            exec(`pgrep -f ${processName}`, (error, stout) => {
                if (error) {
                    rej(error);
                }
                else {
                    res(stout);
                }
            })
        })

    }

    /***************************************************************************
     * @brief killProcessByID - kill a process by a given processID
     ****************************************************************************/
    killProcessByID(processID) {
        return new Promise((res, rej) => {
            exec(`kill -9 ${processID}`, killErr => {
                if (killErr) {
                    rej(killErr);
                }
                else {
                    res()
                }                
            })
        })
    }

    /***************************************************************************
     * @brief killProcessByName - kill a process by a given process name
     ****************************************************************************/
    killProcessByName(processName) {
        var LocalModule = this;
        return new Promise((res, rej) => {
            LocalModule.getPIDofProcess(processName).then(processiId => {
                LocalModule.killProcessByID(processiId.replace(/[\n\r]/g, ' ')).then(() => {
                    res();
                }).catch(err => {
                    rej(err);
                })
            }).catch(() => {
                res();
            })
        })        
    }

    /***************************************************************************
     * @brief restartProcess - restarts a process with a given process name
     * If the process has not been started, it will start the process. If it 
     * has been started before it will be restarted
     ****************************************************************************/
    restartProcess(processName, appParam) {
        var LocalModule = this;
        return new Promise((res, rej) => {
            LocalModule.getPIDofProcess(processName).then(processiId => {
                //App started already
                LocalModule.killProcessByID(processiId.replace(/[\n\r]/g, ' ')).then(() => {
                    //Successfully killed the process now start it again
                    LocalModule.startApp(appParam);
                    res();
                }).catch(err => {
                    //Failed to kill the process
                    LocalModule.sysLogMessage(err.message);
                    rej();
                })
            }).catch(() => {
                //App not started yet
                LocalModule.startApp(appParam);
                res();
            })
        })        
    }

    /***************************************************************************
     * @brief removeCertBe4Upload - Before uploading the new certificate/key, it should
     * be checked if a file exists. It could happen that two certificates with different
     * file type will be uploaded (e.g. ca.crt and ca.der). To avoid this, clear
     * file before upload
     ****************************************************************************/
    removeCertBe4Upload(certName, directory) {
        try {
            let files = fs.readdirSync(directory); 
            files.forEach(file => {
                let filename = file.split('.');
                if (certName == filename[0]) {
                    fs.unlinkSync(path.join(directory, file));
                }
            })
        }
        catch (ex) {
            sysLogMessage("Upload", ex.message);
        }
    }

    /***************************************************************************
     * @brief sqlTestConnection - Test the sql connection
     ****************************************************************************/
    sqlTestConnection() {
        return new Promise((res, rej) => {
            exec(`${this.Constants.NODE_V16_PATH} ${path.join(__dirname, "sqlClientConnectionTest.js")}`, (err, stdout) => {
                if (err) {
                    rej(err);
                }
                else {
                    res(stdout);
                }
            })
        })
    }

    /***************************************************************************
     * @brief nosqlTestConnection - Test the db connection to a no sql database e.g. mongodb
     ****************************************************************************/
    nosqlTestConnection() {
        return new Promise((res, rej) => {
            exec(`${LOCALTEST ? "node" : this.Constants.NODE_V16_PATH} ${path.join(__dirname, "nosqlClientConnectionTest.js")}`, (err, stdout) => {
                if (err) {
                    rej(err);
                }
                else {
                    res(stdout);
                }
            })
        })
    }

    /***************************************************************************
     * @brief encryptPW 
     ****************************************************************************/
    encryptPW(pw) {
        let mac = LOCALTEST ? os.networkInterfaces().WLAN[0].mac : this.getMac();
        let pwk = LOCALTEST ? `${mac}${this.Constants.localtestkey}` : `${mac}${this.Constants.pwkey}`;
        let iv = Buffer.from(mac.substring(0, mac.length - 1));
        let cipher = crypto.createCipheriv(this.Constants.cipher, Buffer.from(pwk), iv);
        let encrypted = cipher.update(pw);
        encrypted = Buffer.concat([encrypted, cipher.final()]);
        return encrypted.toString('hex');
    }

    /***************************************************************************
     * @brief encryptPWWithKeyAndIV by using own key and iv
     ****************************************************************************/
    encryptPWWithKeyAndIV(pw, key, iv) {        
        let cipher = crypto.createCipheriv(this.Constants.cipher, Buffer.from(key), iv);
        let encrypted = cipher.update(pw);
        encrypted = Buffer.concat([encrypted, cipher.final()]);
        return encrypted.toString('hex');
    }

    /***************************************************************************
     * @brief decryptPW
     ****************************************************************************/
    decryptPW(content) {
        let mac = LOCALTEST ? os.networkInterfaces().WLAN[0].mac : this.getMac();
        let pwk = `${mac}${this.Constants.pwkey}`;
        let iv = Buffer.from(mac.substring(0, mac.length - 1));
        let decipher = crypto.createDecipheriv(this.Constants.cipher, Buffer.from(pwk), iv);
        let decrypted = decipher.update(Buffer.from(content, 'hex'));
        decrypted = Buffer.concat([decrypted, decipher.final()]);
        return decrypted.toString();
    }

    /***************************************************************************
     * @brief decryptPWWithKeyAndIV by using own key and iv
     ****************************************************************************/
    decryptPWWithKeyAndIV(content, key, iv) {
        let decipher = crypto.createDecipheriv(this.Constants.cipher, Buffer.from(key), Buffer.from(iv, "hex"));
        let decrypted = decipher.update(Buffer.from(content, 'hex'));
        decrypted = Buffer.concat([decrypted, decipher.final()]);
        return decrypted.toString();
    }

    /***************************************************************************
     * @brief encryptIncomingPW by using own key and iv
     ****************************************************************************/
    encryptIncomingPW(uri, settings) {
        let set = settings;
        //Currently only on 4 pages passwords must be encryted
        switch (uri) {
            case this.Constants.requests.READ_FTPC:
            case this.Constants.requests.READ_HTTPC:
            case this.Constants.requests.READ_SMTP:
                if (set.server_password !== this.Constants.pwPlaceHolder) {
                    if (set.server_password !== "")
                        set.server_password = this.encryptPW(set.server_password);
                }
                else {
                    let oldSetting = this.readConfigFileAsJSON(path.join(__dirname, this.Constants.configFolder, `${uri}${this.Constants.JSONFILE}`));
                    set.server_password = oldSetting.server_password;
                }
                break;

            case this.Constants.requests.READ_SQL:
            case this.Constants.requests.READ_NOSQL:
                if (set.pass !== this.Constants.pwPlaceHolder) {
                    if (set.pass !== "")
                        set.pass = this.encryptPW(set.pass);
                }
                else {
                    let oldSetting = this.readConfigFileAsJSON(path.join(__dirname, this.Constants.configFolder, `${uri}${this.Constants.JSONFILE}`));
                    set.pass = oldSetting.pass;
                }
                break;

            case this.Constants.requests.READ_MQTT:
                if (this.Constants.isCustomized) {
                    if (set.broker_password !== this.Constants.pwPlaceHolder) {
                        if (set.broker_password !== "")
                            set.broker_password = this.encryptPW(set.broker_password);
                    }
                    else {
                        let oldSetting = this.readConfigFileAsJSON(path.join(__dirname, this.Constants.configFolder, `${uri}${this.Constants.JSONFILE}`));
                        set.broker_password = oldSetting.broker_password;
                    }
                }                
                break;
        }
        return JSON.stringify(set);
    }   

    /***************************************************************************
     * @brief getKeyContent by using own key and iv
     ****************************************************************************/
    getKeyContent() {        
        var LocalConstants = this.Constants;
        var client = new net.Socket();

        return new Promise((res, rej) => {
            if (LOCALTEST) {
                res(this.Constants.localtestkey);
                return;
            }
            try {
                client.setTimeout(LocalConstants.Timeout_Status_request, () => {
                    client.end();
                    rej();
                    client.destroy();
                });

                client.connect(LocalConstants.keyServicePort, LocalConstants.LOCAL_HOST, function () {
                    let statusCommand = `#\r\n`;
                    client.write(statusCommand);
                });

                client.on('data', function (data) {
                    res(data.toString());
                    client.destroy();
                });

                client.on('error', err => {
                    rej(err.message);
                    client.destroy();
                });
            }
            catch (ex) {
                rej(ex.message);
            }
        })
    }

    /***************************************************************************
     * @brief getMac returns the mac address of current device
     ****************************************************************************/
    getMac() {
        let defaultResult = "00:00:00:00:00:00";
        try {
            let result = execSync('ip -o link show');            
            const interfaces = result.toString().split('\n').map(line => {
                const match = line.match(/^\d+: (\S+):.*link\/\S+ (\S+)/);
                if (match) {
                    return {
                        name: match[1],
                        mac: match[2]
                    };
                }
                return(null);
            }).filter(Boolean);

            const interfacesObj = {};
            interfaces.forEach(iface => {
                interfacesObj[iface.name] = {
                    mac: iface.mac
                };
            });

            if (interfacesObj.eth0) {
                return interfacesObj.eth0.mac;
            }
            else {
                return defaultResult;
            }
        }
        catch (ex) {
            console.log("Error in reading mac: " + ex.message);
            return (defaultResult);
        }
    }

    /***************************************************************************
     * @brief isIPWithinRange returns true if ip is within range
     * inputs:
     * ipRangeStart: Start ip address in int format
     * ipRangeEnd: End ip address in int format
     * ipAddr: The ip address to be checked if its within the start and end range
     ****************************************************************************/
    isIPWithinRange(ipRangeStart, ipRangeEnd, ipAddr) {
        return ipAddr >= ipRangeStart && ipAddr <= ipRangeEnd;
    }
}

exports.pewModule = pewModule