﻿var base = require('./base');

class Mewtocol7 extends base
{
    constructor()
    {
        super();
        this.CRC_LookupTable = new Array(
            0x0000, 0x1189, 0x2312, 0x329B, 0x4624, 0x57AD, 0x6536, 0x74BF, 0x8C48, 0x9DC1,
            0xAF5A, 0xBED3, 0xCA6C, 0xDBE5, 0xE97E, 0xF8F7, 0x1081, 0x0108, 0x3393, 0x221A,
            0x56A5, 0x472C, 0x75B7, 0x643E, 0x9CC9, 0x8D40, 0xBFDB, 0xAE52, 0xDAED, 0xCB64,
            0xF9FF, 0xE876, 0x2102, 0x308B, 0x0210, 0x1399, 0x6726, 0x76AF, 0x4434, 0x55BD,
            0xAD4A, 0xBCC3, 0x8E58, 0x9FD1, 0xEB6E, 0xFAE7, 0xC87C, 0xD9F5, 0x3183, 0x200A,
            0x1291, 0x0318, 0x77A7, 0x662E, 0x54B5, 0x453C, 0xBDCB, 0xAC42, 0x9ED9, 0x8F50,
            0xFBEF, 0xEA66, 0xD8FD, 0xC974, 0x4204, 0x538D, 0x6116, 0x709F, 0x0420, 0x15A9,
            0x2732, 0x36BB, 0xCE4C, 0xDFC5, 0xED5E, 0xFCD7, 0x8868, 0x99E1, 0xAB7A, 0xBAF3,
            0x5285, 0x430C, 0x7197, 0x601E, 0x14A1, 0x0528, 0x37B3, 0x263A, 0xDECD, 0xCF44,
            0xFDDF, 0xEC56, 0x98E9, 0x8960, 0xBBFB, 0xAA72, 0x6306, 0x728F, 0x4014, 0x519D,
            0x2522, 0x34AB, 0x0630, 0x17B9, 0xEF4E, 0xFEC7, 0xCC5C, 0xDDD5, 0xA96A, 0xB8E3,
            0x8A78, 0x9BF1, 0x7387, 0x620E, 0x5095, 0x411C, 0x35A3, 0x242A, 0x16B1, 0x0738,
            0xFFCF, 0xEE46, 0xDCDD, 0xCD54, 0xB9EB, 0xA862, 0x9AF9, 0x8B70, 0x8408, 0x9581,
            0xA71A, 0xB693, 0xC22C, 0xD3A5, 0xE13E, 0xF0B7, 0x0840, 0x19C9, 0x2B52, 0x3ADB,
            0x4E64, 0x5FED, 0x6D76, 0x7CFF, 0x9489, 0x8500, 0xB79B, 0xA612, 0xD2AD, 0xC324,
            0xF1BF, 0xE036, 0x18C1, 0x0948, 0x3BD3, 0x2A5A, 0x5EE5, 0x4F6C, 0x7DF7, 0x6C7E,
            0xA50A, 0xB483, 0x8618, 0x9791, 0xE32E, 0xF2A7, 0xC03C, 0xD1B5, 0x2942, 0x38CB,
            0x0A50, 0x1BD9, 0x6F66, 0x7EEF, 0x4C74, 0x5DFD, 0xB58B, 0xA402, 0x9699, 0x8710,
            0xF3AF, 0xE226, 0xD0BD, 0xC134, 0x39C3, 0x284A, 0x1AD1, 0x0B58, 0x7FE7, 0x6E6E,
            0x5CF5, 0x4D7C, 0xC60C, 0xD785, 0xE51E, 0xF497, 0x8028, 0x91A1, 0xA33A, 0xB2B3,
            0x4A44, 0x5BCD, 0x6956, 0x78DF, 0x0C60, 0x1DE9, 0x2F72, 0x3EFB, 0xD68D, 0xC704,
            0xF59F, 0xE416, 0x90A9, 0x8120, 0xB3BB, 0xA232, 0x5AC5, 0x4B4C, 0x79D7, 0x685E,
            0x1CE1, 0x0D68, 0x3FF3, 0x2E7A, 0xE70E, 0xF687, 0xC41C, 0xD595, 0xA12A, 0xB0A3,
            0x8238, 0x93B1, 0x6B46, 0x7ACF, 0x4854, 0x59DD, 0x2D62, 0x3CEB, 0x0E70, 0x1FF9,
            0xF78F, 0xE606, 0xD49D, 0xC514, 0xB1AB, 0xA022, 0x92B9, 0x8330, 0x7BC7, 0x6A4E,
            0x58D5, 0x495C, 0x3DE3, 0x2C6A, 0x1EF1, 0x0F78);

        this.StartAddr = 0;
        this.NumberOfRegs = 1;
        this.datatype = "DT";
        this.Station = 0;
        this.writeData = [];
        this.writeBit = 0;
        this.MAX_STATION = 99;
        this.selfProtocol = "MEWTOCOL7";
        this.MAX_READ_REGS = 512;
        this.MAX_WRITE_REGS = 500;
        this.MAX_READ_BITS = 64;
    }

    /***************************************************************************/
    /**
     * @brief fastLookUp_CRC - CRC lookup for the mewtocol7 command
     *   
     * @param returns a string with the Mewtocol command including the CRC
     ****************************************************************************/
    fastLookUp_CRC(sCommand)
    {
        let crc = 0xFFFF, zeroes = "0000";
        let DecodedCRC;

        crc = this.Reflect(crc);
        //Calculate according to the look up table the CRC for Mewtocol7 Requests	
        for (let val of sCommand)
            crc = (crc >> 8) ^ this.CRC_LookupTable[(crc & 0xFF) ^ val.charCodeAt(0)];

        //Decode CRC to Ascii
        DecodedCRC = crc.toString(16);
        if (DecodedCRC.length < 4)
        {
            DecodedCRC = zeroes + DecodedCRC;
            DecodedCRC = DecodedCRC.slice(DecodedCRC.length - 4);
        }

        DecodedCRC = DecodedCRC.toUpperCase();
        //Add CRC to command and add a Carriage return
        sCommand = sCommand + DecodedCRC + String.fromCharCode(13);
        return sCommand;
    }

    Reflect(crc)
    {
        var i, crcout = 0, j = 1;
        for (i = 0x8000; i != 0; i >>= 1)
        {
            if ((crc & i) != 0)
                crcout |= j;

            j <<= 1;
        }

        return (crcout);
    }

    /***************************************************************************/
    /**
     * @brief GenerateReadDT - Generate the Mewtocol read command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateReadRegisters(startAddr = this.StartAddr, numbOfRegs = this.NumberOfRegs, station = this.Station, datatype = this.datatype)
    {
        var ReadCommand, PLCStation, temp;
        var StartAddrZeroes = "000000", NumOfDTsZeroes = "0000000";

        switch (true)
        {
            case isNaN(station):
            case station <= 0:
                PLCStation = "EEE";
                break;

            case station > 0 && station < 10:
                PLCStation = "00" + station.toString();
                break;

            case station >= 10 && station <= this.MAX_STATION:
                PLCStation = "0" + station.toString();
                break;

            case station > this.MAX_STATION:
                PLCStation = "EEE";
                break;

            default:
                PLCStation = "EEE";
        }

        temp = StartAddrZeroes + startAddr.toString();
        temp = temp.slice(temp.length - 6);

        switch (datatype.toUpperCase())
        {
            case 'SD':
                ReadCommand = `>@${PLCStation}00#00MMRDDG2SD${temp}`;
                break;
            case 'DT':
                ReadCommand = `>@${PLCStation}00#00MMRDDG2DT${temp}`;
                break;
            case 'LD':
                ReadCommand = `>@${PLCStation}00#00MMRDDG2LD${temp}`;
                break;
            case 'WR':
                ReadCommand = `>@${PLCStation}00#00MMRDDG2WR${temp}`;
                break;
            case 'WL':
                ReadCommand = `>@${PLCStation}00#00MMRDDG2WL${temp}`;
                break;
            case 'WY':
                ReadCommand = `>@${PLCStation}00#00MMRDDG2WY${temp}`;
                break;
            case 'WX':
                ReadCommand = `>@${PLCStation}00#00MMRDDG2WX${temp}`;
                break;
            default:
                ReadCommand = `>@${PLCStation}00#00MMRDDG2DT${temp}`;
        }

        temp = NumOfDTsZeroes + numbOfRegs;
        temp = temp.slice(temp.length - 7);

        ReadCommand += temp;
        return this.fastLookUp_CRC(ReadCommand);
    }

    /***************************************************************************/
    /**
     * @brief GenerateReadBits - Generate the Mewtocol read command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateReadBits(startAddr = this.StartAddr, numbOfRegs = this.NumberOfRegs, station = this.Station, datatype = this.datatype)
    {
        var sRequest, i, iLength, sCnt;
        var sTypeMultiRead;
        sRequest = ">@";

        //Insert Station number (max. station number is 99)
        switch (true)
        {
            case isNaN(station):
            case station <= 0:
                sRequest += "EEE";
                break;

            case station > 0 && station < 10:
                sRequest += "00" + station.toString();
                break;

            case station >= 10 && station <= this.MAX_STATION:
                sRequest += "0" + station.toString();
                break;

            case station > this.MAX_STATION:
                sRequest += "EEE";
                break;

            default:
                sRequest += "EEE";
        }

        //Insert fixed parameters
        sRequest += "00#00MMRDC";

        //Insert number of contacts to read at position nnn	
        switch (true)
        {
            case numbOfRegs <= 0:
                sCnt = "1";
                break;

            case numbOfRegs > 0 && numbOfRegs <= this.MAX_READ_BITS:
                sCnt = numbOfRegs.toString(10);
                break;

            case numbOfRegs > this.MAX_READ_BITS:
                sCnt = "64";
                break;

            default:
                sCnt = numbOfRegs.toString(10);
        }

        iLength = sCnt.length;

        switch (iLength)
        {
            case 1:
                sRequest += "00" + sCnt;
                break;
            case 2:
                sRequest += "0" + sCnt;
                break;
            case 3:
                sRequest += sCnt;
                break;
            default:
                sRequest += sCnt;
        }

        //Insert Type
        switch (datatype.toUpperCase())
        {
            case "R":
                sTypeMultiRead = "GR0";
                break;	//Read R
            case "X":
                sTypeMultiRead = "GX0";
                break;	//Read X -> change R0 to X0
            case "Y":
                sTypeMultiRead = "GY0";
                break;	//Read Y -> change R0 to Y0
            case "L":
                sTypeMultiRead = "GL0";
                break;	//Read L -> change R0 to L0
            default:
                sTypeMultiRead = "GR0";
        }

        //Insert start address (6 dec numbers + 1 hex)
        var zeroes = "0000000";
        var temp, tempStartAddr;

        tempStartAddr = startAddr;
        //Insert all address which should be read in sequence (e.g. 16 relays to read starting at R0 - RF)
        for (i = 0; i < numbOfRegs; i++)
        {
            temp = zeroes + tempStartAddr;
            temp = temp.slice(temp.length - 7);
            temp = sTypeMultiRead + temp;
            sRequest += temp;

            tempStartAddr = this.IncrementMewBoolAddress(tempStartAddr);
        }

        //Insert CRC
        return this.fastLookUp_CRC(sRequest);
    }

    /***************************************************************************/
    /**
     * @brief GenerateWriteRegisters - Generate the Mewtocol write command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateWriteRegisters(startAddr = this.StartAddr, station = this.Station, datatype = this.datatype, writedata = this.writeData)
    {
        var WriteCommand, PLCStation, temp, i, valuesTemp = "";
        var StartAddrZeroes = "000000", NumOfDTsZeroes = "0000000", ValuesZeroes = "0000";
        let len = Array.isArray(writedata) ? writedata.length : 1;

        switch (true)
        {
            case isNaN(station):
            case station <= 0:
                PLCStation = "EEE";
                break;

            case station > 0 && station < 10:
                PLCStation = "00" + station.toString();
                break;

            case station >= 10 && station <= this.MAX_STATION:
                PLCStation = "0" + station.toString();
                break;

            case station > this.MAX_STATION:
                PLCStation = "EEE";
                break;

            default:
                PLCStation = "EEE";
        }

        temp = StartAddrZeroes + startAddr.toString();
        temp = temp.slice(temp.length - 6);

        switch (datatype.toUpperCase())
        {
            case "DT":
                WriteCommand = `>@${PLCStation}00#01MMWTDG2DT${temp}`;
                break;
            case "LD":
                WriteCommand = `>@${PLCStation}00#01MMWTDG2LD${temp}`;
                break;
            case "WR":
                WriteCommand = `>@${PLCStation}00#01MMWTDG2WR${temp}`;
                break;
            case "WY":
                WriteCommand = `>@${PLCStation}00#01MMWTDG2WY${temp}`;
                break;
            case "WL":
                WriteCommand = `>@${PLCStation}00#01MMWTDG2WL${temp}`;
                break;
            default:
                WriteCommand = `>@${PLCStation}00#01MMWTDG2DT${temp}`;
        }

        temp = NumOfDTsZeroes + len;
        temp = temp.slice(temp.length - 7);

        WriteCommand += temp;

        for (i = 0; i < len; i++)
        {
            valuesTemp = ValuesZeroes + this.d2h(Array.isArray(writedata) ? writedata[i] : writedata);
            valuesTemp = valuesTemp.slice(valuesTemp.length - 4);
            WriteCommand += valuesTemp;
        }

        return this.fastLookUp_CRC(WriteCommand);
    }

    /***************************************************************************/
    /**
     * @brief GenerateWriteSingleBit - Generate the Mewtocol write command
     *   
     * @param returns a string with the Mewtocol command
     ****************************************************************************/
    GenerateWriteSingleBit(startAddr = this.StartAddr, station = this.Station, datatype = this.datatype, writebit = this.writeBit)
    {
        var WriteCommand, PLCStation, temp;
        var StartAddrZeroes = "0000000";

        switch (true)
        {
            case isNaN(station):
            case station <= 0:
                PLCStation = "EEE";
                break;

            case station > 0 && station < 10:
                PLCStation = "00" + station.toString();
                break;

            case station >= 10 && station <= this.MAX_STATION:
                PLCStation = "0" + station.toString();
                break;

            case station > this.MAX_STATION:
                PLCStation = "EEE";
                break;

            default:
                PLCStation = "EEE";
        }

        temp = StartAddrZeroes + startAddr.toString();
        temp = temp.slice(temp.length - 7);

        switch (datatype.toUpperCase())
        {
            case "R":
                WriteCommand = `>@${PLCStation}00#01MMWTC001GR0${temp}${writebit.toString()}`;
                break;
            case "L":
                WriteCommand = `>@${PLCStation}00#01MMWTC001GL0${temp}${writebit.toString()}`;
                break;
            case "Y":
                WriteCommand = `>@${PLCStation}00#01MMWTC001GY0${temp}${writebit.toString()}`;
                break;
            default:
                WriteCommand = `>@${PLCStation}00#01MMWTC001GR0${temp}${writebit.toString()}`;
        }

        return this.fastLookUp_CRC(WriteCommand);
    }

    /***************************************************************************/
    /**
     * @brief DecodeReadBits - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns an object with decoded values
     ****************************************************************************/
    DecodeReadBits(response, iNumberOfContacts = this.NumberOfRegs)
    {
        let responseASCII = response.toString();

        let result = {
            err: false,
            err_code: 0,
            state: [],
        }

        if (responseASCII.charAt(7) != '$')
        {
            result.err = true;
            result.err_code = responseASCII.substr(14, 2);
            return result;
        }
        var Result = new Array();
        var i, j, words, iMod, temp, firstByte, secondByte;

        iMod = iNumberOfContacts % 8;
        words = parseInt(iNumberOfContacts / 8, 10);

        if (iMod !== 0)
            words++;

        for (i = 0; i < words; i++)
        {
            j = i * 8;
            temp = responseASCII.slice(14 + i * 2, 16 + i * 2);
            firstByte = temp.slice(1, 2);
            secondByte = temp.slice(0, 1);

            firstByte = parseInt(firstByte, 16);
            secondByte = parseInt(secondByte, 16);

            Result[j] = firstByte & 1;
            Result[j + 1] = (firstByte & 2) >> 1;
            Result[j + 2] = (firstByte & 4) >> 2;
            Result[j + 3] = (firstByte & 8) >> 3;
            Result[j + 4] = secondByte & 1;
            Result[j + 5] = (secondByte & 2) >> 1;
            Result[j + 6] = (secondByte & 4) >> 2;
            Result[j + 7] = (secondByte & 8) >> 3;
        }

        if (iNumberOfContacts === 1) {
            result.state[0] = Result[0]
        }
        else {
            result.state = Result;
        }
        return result;
    }

    /***************************************************************************/
    /**
     * @brief DecodeReadRegisters - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns an object with decoded values
     ****************************************************************************/
    DecodeReadRegisters(response)
    {
        let responseASCII = response.toString();

        let result = {
            err: false,
            err_code: 0,
            int: [],
            uint: [],
            udint: [],
            dint: [],
            hex: [],
            real: [],
            string: ""
        }

        if (responseASCII.charAt(7) != '$')
        {
            result.err = true;
            result.err_code = responseASCII.substr(14, 2);
            return result;
        }

        var iNum, i, j = 0, temp;
        //Cut the header and CRC + CR
        var sResponseWithoutHeaderAndCRC = responseASCII.slice(14, responseASCII.length - 5);
        iNum = sResponseWithoutHeaderAndCRC.length / 4;

        for (i = 0; i < iNum; i++)
        {
            j = i * 4;
            temp = sResponseWithoutHeaderAndCRC.substring(j, j + 4);
            result.uint.push(this.h2d(temp));
            result.int.push(this.HexStringToINT(temp));
            result.hex.push(temp);

            //Read the values as string
            temp = this.h2d(sResponseWithoutHeaderAndCRC.substring(j + 2, j + 4));
            result.string += String.fromCharCode(temp);
            temp = this.h2d(sResponseWithoutHeaderAndCRC.substring(j, j + 2));
            result.string += String.fromCharCode(temp);
        }

        //combine DINT and UDINT
        let len = result.uint.length;
        for (i = 0; i < len; i = i + 2)
        {
            if (i + 1 !== len)
            {
                temp = result.hex[i + 1] + result.hex[i];
                let udint = this.h2d(temp);
                result.udint.push(udint);
                result.dint.push(this.HexStringToDINT(temp));
                result.real.push(this.CalculateIEEEReal(udint));
            }
        }

        return result;
    }

    /***************************************************************************/
    /**
     * @brief WriteResponseOK - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns true when the response was ok, otherwise false
     ****************************************************************************/
    WriteResponseOK(response)
    {
        let responseASCII = response.toString();

        let result = {
            err: false,
            err_code: 0,
            err_msg: ""
        }

        if (responseASCII.charAt(7) != '$')
        {
            result.err = true;
            result.err_code = responseASCII.substr(14, 2);
            result.err_msg = "Unexpected respond";
        }

        return result;
    }
}

exports.Mewtocol7 = Mewtocol7;